<?php
/**
 * Registers the licensing metabox.
 *
 * @package   EDD\SoftwareLicensing\Admin\Downloads\Metaboxes
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Downloads\Metaboxes;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Admin\Downloads\Metaboxes\Metabox;
use EDD\SoftwareLicensing\Admin\Downloads\Editor;

/**
 * Licensing metabox.
 *
 * @since 3.9.0
 */
class Licensing extends Metabox {

	/**
	 * Metabox ID.
	 *
	 * @var string
	 */
	protected $id = 'edd_sl_download_settings';

	/**
	 * Context.
	 *
	 * @var string
	 */
	protected $context = 'normal';

	/**
	 * Priority.
	 *
	 * @var string
	 */
	protected $priority = 'high';

	/**
	 * Get the metabox title.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	public function get_title(): string {
		return __( 'Licensing Settings', 'edd_sl' );
	}

	/**
	 * Renders the metabox.
	 *
	 * @param \WP_Post $post Post object.
	 * @return void
	 */
	public function render( \WP_Post $post ) {

		$post_id = $post->ID;
		if ( is_numeric( $post_id ) && ! current_user_can( 'edit_product', $post_id ) ) {
			return;
		}

		if ( is_null( $post_id ) && ! current_user_can( 'edit_products' ) ) {
			return;
		}

		if ( ! $this->download ) {
			return;
		}

		$registered_sections = $this->get_registered_sections( $this->download );
		if ( empty( $registered_sections ) ) {
			return;
		}

		$metabox_sections = new Editor\Sections();
		$metabox_sections->set_sections( $registered_sections );
		$metabox_sections->set_item( $this->download );
		$metabox_sections->display();

		wp_nonce_field( 'edd_sl_metabox_licensing_settings', 'edd_sl_meta_box_nonce' );
	}

	/**
	 * Get the sections.
	 *
	 * @since 3.9.0
	 * @param \EDD_Download $download The download.
	 * @return array
	 */
	private function get_registered_sections( $download ) {

		/**
		 * Allow developers to add, remove, or modify the sections in the licensing settings metabox.
		 *
		 * @since 3.9.0
		 * @param array $sections Array of sections.
		 */
		$sections = apply_filters(
			'edd_licensing_settings_sections',
			array(
				'general'      => Editor\General::class,
				'versions'     => Editor\Versions::class,
				'betas'        => Editor\Betas::class,
				'requirements' => Editor\Requirements::class,
				'upgrades'     => Editor\Upgrades::class,
				'preset_keys'  => Editor\PresetKeys::class,
			)
		);

		if ( edd_get_option( 'edd_sl_readme_parsing' ) ) {
			$sections['readme'] = Editor\WPReadme::class;
		}

		if ( empty( $sections ) ) {
			return array();
		}

		foreach ( $sections as $key => $section ) {
			$valid_section = $this->validate_section( $section, $key );
			if ( ! $valid_section ) {
				continue;
			}

			$registered_sections[] = $valid_section->get_config();
		}

		// Now sort the sections by priority.
		usort( $registered_sections, array( $this, 'sort_sections_by_priority' ) );

		return $registered_sections;
	}
}
