<?php
/**
 * Registers our metaboxes for Software Licensing.
 *
 * @package   EDD\SoftwareLicensing\Admin\Downloads
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Downloads;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;
use EDD\HTML\Tooltip;

/**
 * Registers our metaboxes for Software Licensing.
 *
 * @since 3.9.0
 */
class Metabox implements SubscriberInterface {

	/**
	 * Returns an array of events that this subscriber wants to listen to.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	public static function get_subscribed_events() {
		return array(
			'edd_download_editor_metaboxes'                     => 'register_metaboxes',
			'edd_meta_box_fields'                               => array( 'add_license_enabled_field', 10, 2 ),
			'postbox_classes_download_edd_sl_download_settings' => 'add_classes_to_metabox',
		);
	}

	/**
	 * Registers our metaboxes.
	 *
	 * @since 3.9.0
	 * @param array $metaboxes The metaboxes.
	 * @return array
	 */
	public function register_metaboxes( $metaboxes ) {
		$metaboxes[] = new Metaboxes\Licensing();

		return $metaboxes;
	}

	/**
	 * Add the license enabled field to the main Download Details Metabox.
	 *
	 * @since 3.9.0
	 * @param int           $download_id The download ID.
	 * @param \EDD_Download $download The download object.
	 * @return void
	 */
	public function add_license_enabled_field( $download_id, $download ) {
		$enabled = get_post_meta( $download_id, '_edd_sl_enabled', true );
		?>
		<div class="edd-form-group">
			<?php do_action( 'edd_sl_license_metabox_before_license_enabled', $download_id ); ?>
			<div class="edd-form-group__control edd-toggle">
				<input
					type="checkbox"
					name="edd_license_enabled"
					id="edd_license_enabled"
					value="1"
					class="edd-requirement"
					<?php checked( true, $enabled ); ?>
					data-edd-requirement="licensing"
				/>
				<label for="edd_license_enabled"><?php esc_html_e( 'Create licenses for this product.', 'edd_sl' ); ?></label>
				<?php
				$tooltip = new Tooltip(
					array(
						'title'   => __( 'Bundle Licensing', 'edd_sl' ),
						'content' => __( 'Purchasing a bundled license key will generate a Bundle License key as well as a Child License key for each licensed product in the bundle.<br /><br /><strong>Bundle License Key</strong><br />This key can be used to renew all products in the bundle, however cannot be used to activate the bundled products.<br /><br /><strong>Child License Key</strong><br />These keys can be used to activate the licensed products in the bundle.', 'edd_sl' ),
					)
				);
				?>
				<span data-edd-supports-product-type="bundle"><?php $tooltip->output(); ?></span>
				<?php do_action( 'edd_sl_license_metabox_after_license_enabled', $download_id ); ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Add the classes to the metabox.
	 *
	 * Always start out hidden, we can show it after the page is done loading.
	 *
	 * @since 3.9.0
	 * @param array $classes The classes to add.
	 * @return array
	 */
	public function add_classes_to_metabox( $classes ) {
		$download_id = filter_input( INPUT_GET, 'post', FILTER_SANITIZE_NUMBER_INT );
		if ( ! $download_id || ! get_post_meta( $download_id, '_edd_sl_enabled', true ) ) {
			$classes[] = 'edd-hidden';
		}

		return $classes;
	}
}
