<?php
/**
 * Handle Download Meta hooks and filters.
 *
 * @package EDD\SoftwareLicensing\Admin\Downloads
 * @since 3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Downloads;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * Metabox for the download editor.
 *
 * @since 3.9.0
 */
class Meta implements SubscriberInterface {

	/**
	 * Get the subscribed events.
	 *
	 * @since 3.9.0
	 * @return array
	 */
	public static function get_subscribed_events() {
		return array(
			'edd_save_download' => array( 'save', 10, 2 ),
		);
	}

	/**
	 * Save the metabox.
	 *
	 * This has been moved from the 'save_post' hook to the 'edd_save_download' hook
	 * to allow us to not repeat EDD's metabox saving logic and access controls.
	 *
	 * @since 3.9.0
	 * @param int     $post_id The post ID.
	 * @param WP_Post $post The post object.
	 * @return void
	 */
	public function save( $post_id, $post ) {
		// This should only run if EDD is saving the download.
		if ( ! did_action( 'edd_save_download' ) ) {
			return;
		}

		// Nonce check specifically for Software Licensing.
		if ( ! isset( $_POST['edd_sl_meta_box_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['edd_sl_meta_box_nonce'] ) ), 'edd_sl_metabox_licensing_settings' ) ) {
			return;
		}

		$posted_data = $_POST;

		// The Setting to enable licensing is not in a metabox, so we need to handle it here.
		if ( isset( $posted_data['edd_license_enabled'] ) ) {
			update_post_meta( $post_id, '_edd_sl_enabled', true );
		} else {
			delete_post_meta( $post_id, '_edd_sl_enabled' );
		}

		/** General Section */
		// License Activation Limit.
		if ( ! empty( $posted_data['edd_sl_limit'] ) ) {
			update_post_meta( $post_id, '_edd_sl_limit', (int) $posted_data['edd_sl_limit'] );
		} else {
			delete_post_meta( $post_id, '_edd_sl_limit' );
		}

		// License Length.
		if ( isset( $posted_data['edd_sl_is_lifetime'] ) ) {
			$is_lifetime = (int) (bool) $posted_data['edd_sl_is_lifetime'];
			update_post_meta( $post_id, 'edd_sl_download_lifetime', $is_lifetime );
		} else {
			delete_post_meta( $post_id, 'edd_sl_download_lifetime' );
		}

		if ( isset( $posted_data['edd_sl_exp_unit'] ) ) {
			update_post_meta( $post_id, '_edd_sl_exp_unit', addslashes( $posted_data['edd_sl_exp_unit'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_sl_exp_unit' );
		}

		if ( ! empty( $posted_data['edd_sl_exp_length'] ) ) {
			update_post_meta( $post_id, '_edd_sl_exp_length', addslashes( $posted_data['edd_sl_exp_length'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_sl_exp_length' );
		}

		// Renewal Discounts.
		if ( ! empty( $posted_data['edd_sl_disable_renewal_discount'] ) ) {
			update_post_meta( $post_id, '_edd_sl_disable_renewal_discount', 1 );
		} else {
			delete_post_meta( $post_id, '_edd_sl_disable_renewal_discount' );
		}

		if ( ! empty( $posted_data['edd_sl_renewal_discount'] ) ) {
			update_post_meta( $post_id, '_edd_sl_renewal_discount', edd_sanitize_amount( $posted_data['edd_sl_renewal_discount'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_sl_renewal_discount' );
		}

		/** Versions Section */
		// Current Product Version.
		if ( ! empty( $posted_data['edd_sl_version'] ) ) {
			update_post_meta( $post_id, '_edd_sl_version', (string) $posted_data['edd_sl_version'] );
		} else {
			delete_post_meta( $post_id, '_edd_sl_version' );
		}

		// Update File.
		if ( isset( $posted_data['edd_sl_upgrade_file'] ) ) {

			$file_id = intval( $posted_data['edd_sl_upgrade_file'] );
			$files   = edd_get_download_files( $post_id );

			if ( '-1' !== $file_id && is_array( $files ) && array_key_exists( $file_id, $files ) ) {
				update_post_meta( $post_id, '_edd_sl_upgrade_file_key', $file_id );
			} else {
				delete_post_meta( $post_id, '_edd_sl_upgrade_file_key' );
			}
		} else {
			delete_post_meta( $post_id, '_edd_sl_upgrade_file_key' );
		}

		// Changelog.
		if ( ! empty( $posted_data['edd_sl_changelog'] ) ) {
			update_post_meta( $post_id, '_edd_sl_changelog', addslashes( $posted_data['edd_sl_changelog'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_sl_changelog' );
		}

		// Upgrade Notice.
		if ( ! empty( $posted_data['edd_sl_upgrade_notice'] ) ) {
			update_post_meta( $post_id, '_edd_sl_upgrade_notice', wp_kses( stripslashes( trim( $posted_data['edd_sl_upgrade_notice'] ) ), wp_kses_allowed_html( 'post' ) ) );
		} else {
			delete_post_meta( $post_id, '_edd_sl_upgrade_notice' );
		}

		// Staged Rollouts.
		if ( isset( $posted_data['edd_sr_rollouts_enabled'] ) && isset( $posted_data['edd_license_enabled'] ) ) {
			update_post_meta( $post_id, 'edd_sr_enabled', true );
		} else {
			delete_post_meta( $post_id, 'edd_sr_enabled' );
		}

		if ( isset( $posted_data['edd_sr_batch_enabled'] ) && isset( $posted_data['edd_sr_rollouts_enabled'] ) ) {
			update_post_meta( $post_id, 'edd_sr_batch_enabled', true );
		} else {
			delete_post_meta( $post_id, 'edd_sr_batch_enabled' );
		}

		if ( isset( $posted_data['edd_sr_version_enabled'] ) && isset( $posted_data['edd_sr_rollouts_enabled'] ) ) {
			update_post_meta( $post_id, 'edd_sr_version_enabled', true );
		} else {
			delete_post_meta( $post_id, 'edd_sr_version_enabled' );
		}

		// @todo Remove this at a later date.
		delete_post_meta( $post_id, 'edd_sr_batch_min' );

		if ( isset( $posted_data['edd_sr_batch_max'] ) ) {
			update_post_meta( $post_id, 'edd_sr_batch_max', (int) $posted_data['edd_sr_batch_max'] );
		} else {
			delete_post_meta( $post_id, 'edd_sr_batch_max' );
		}

		if ( isset( $posted_data['edd_sr_version_limit'] ) ) {
			$above_below = (int) (bool) $posted_data['edd_sr_version_limit'];
			update_post_meta( $post_id, 'edd_sr_version_limit', $above_below );
		} else {
			delete_post_meta( $post_id, 'edd_sr_version_limit' );
		}

		if ( isset( $posted_data['edd_sr_version'] ) && ! empty( $posted_data['edd_sr_version'] ) ) {
			update_post_meta( $post_id, 'edd_sr_version', (string) $posted_data['edd_sr_version'] );
		} else {
			delete_post_meta( $post_id, 'edd_sr_version' );
		}

		/** Betas Section */
		// Beta Enabled.
		if ( ! empty( $posted_data['edd_sl_beta_enabled'] ) ) {
			update_post_meta( $post_id, '_edd_sl_beta_enabled', true );
		} else {
			delete_post_meta( $post_id, '_edd_sl_beta_enabled' );
		}

		// Beta File(s).
		if ( isset( $posted_data['edd_sl_beta_files'] ) && false !== $posted_data['edd_sl_beta_files'] ) {
			/**
			 * Filter the beta files.
			 *
			 * @since 3.9.0
			 * @param array[] $files Beta files array. File arrays have `file` and `name` keys.
			 * @return array
			 */
			$beta_files = apply_filters(
				'edd_metabox_save_beta_files',
				$this->sanitize_beta_files( $posted_data['edd_sl_beta_files'] )
			);
			if ( $beta_files ) {
				update_post_meta( $post_id, '_edd_sl_beta_files', $beta_files );
			} else {
				delete_post_meta( $post_id, '_edd_sl_beta_files' );
			}
		}

		// Beta Upgrade File. (Hidden input with a value of '1' so this will always be added).
		if ( isset( $posted_data['edd_sl_beta_upgrade_file'] ) && $posted_data['edd_sl_beta_upgrade_file'] !== false ) {
			update_post_meta( $post_id, '_edd_sl_beta_upgrade_file_key', (int) $posted_data['edd_sl_beta_upgrade_file'] );
		} else {
			delete_post_meta( $post_id, '_edd_sl_beta_upgrade_file_key' );
		}

		// Beta Version.
		if ( ! empty( $posted_data['edd_sl_beta_version'] ) ) {
			update_post_meta( $post_id, '_edd_sl_beta_version', sanitize_text_field( $posted_data['edd_sl_beta_version'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_sl_beta_version' );
		}

		// Beta Changelog.
		if ( ! empty( $posted_data['edd_sl_beta_changelog'] ) ) {
			update_post_meta( $post_id, '_edd_sl_beta_changelog', wp_kses( stripslashes( $posted_data['edd_sl_beta_changelog'] ), wp_kses_allowed_html( 'post' ) ) );
		} else {
			delete_post_meta( $post_id, '_edd_sl_beta_changelog' );
		}

		/** Requirements Section */
		// Required platform versions.
		$required_versions = array();
		if ( ! empty( $posted_data['edd_sl_required_versions'] ) ) {
			$required_versions = array_filter( array_map( 'sanitize_text_field', $posted_data['edd_sl_required_versions'] ) );
		}
		if ( ! empty( $required_versions ) ) {
			update_post_meta( $post_id, '_edd_sl_required_versions', $required_versions );
		} else {
			delete_post_meta( $post_id, '_edd_sl_required_versions' );
		}

		/** WP Readme Section */
		// Readme Location.
		if ( ! empty( $posted_data['_edd_readme_location'] ) ) {
			update_post_meta( $post_id, '_edd_readme_location', sanitize_text_field( $posted_data['_edd_readme_location'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_readme_location' );
		}

		// Readme Plugin Homepage.
		if ( ! empty( $posted_data['_edd_readme_plugin_homepage'] ) ) {
			update_post_meta( $post_id, '_edd_readme_plugin_homepage', sanitize_text_field( $posted_data['_edd_readme_plugin_homepage'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_readme_plugin_homepage' );
		}

		// Readme Plugin Added.
		if ( ! empty( $posted_data['_edd_readme_plugin_added'] ) ) {
			update_post_meta( $post_id, '_edd_readme_plugin_added', sanitize_text_field( $posted_data['_edd_readme_plugin_added'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_readme_plugin_added' );
		}

		// Readme Meta.
		if ( ! empty( $posted_data['_edd_readme_meta'] ) ) {
			update_post_meta( $post_id, '_edd_readme_meta', $posted_data['_edd_readme_meta'] );
		} else {
			delete_post_meta( $post_id, '_edd_readme_meta' );
		}

		// Readme Sections.
		if ( ! empty( $posted_data['_edd_readme_sections'] ) ) {
			update_post_meta( $post_id, '_edd_readme_sections', $posted_data['_edd_readme_sections'] );
		} else {
			delete_post_meta( $post_id, '_edd_readme_sections' );
		}

		// Readme Banner High.
		if ( ! empty( $posted_data['_edd_readme_plugin_banner_high'] ) ) {
			update_post_meta( $post_id, '_edd_readme_plugin_banner_high', sanitize_text_field( $posted_data['_edd_readme_plugin_banner_high'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_readme_plugin_banner_high' );
		}

		// Readme Banner Low.
		if ( ! empty( $posted_data['_edd_readme_plugin_banner_low'] ) ) {
			update_post_meta( $post_id, '_edd_readme_plugin_banner_low', sanitize_text_field( $posted_data['_edd_readme_plugin_banner_low'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_readme_plugin_banner_low' );
		}

		/** Upgrades Section */
		// Upgrade paths.
		if ( ! empty( $posted_data['edd_sl_upgrade_paths'] ) && is_array( $posted_data['edd_sl_upgrade_paths'] ) ) {

			$upgrade_paths = array();

			foreach ( $posted_data['edd_sl_upgrade_paths'] as $key => $path ) {

				$upgrade_download_id = ! empty( $path['download_id'] ) ? absint( $path['download_id'] ) : false;
				if ( empty( $upgrade_download_id ) ) {
					continue;
				}

				$upgrade_paths[ $key ]['download_id'] = $upgrade_download_id;
				$upgrade_paths[ $key ]['price_id']    = isset( $path['price_id'] ) ? absint( $path['price_id'] ) : false;
				$upgrade_paths[ $key ]['discount']    = edd_sanitize_amount( $path['discount'] );
				$upgrade_paths[ $key ]['pro_rated']   = isset( $path['pro_rated'] ) ? 1 : 0;

			}

			if ( ! empty( $upgrade_paths ) ) {
				update_post_meta( $post_id, '_edd_sl_upgrade_paths', $upgrade_paths );
			} else {
				delete_post_meta( $post_id, '_edd_sl_upgrade_paths' );
			}
		}

		/** Preset Keys Section */
		// Preset keys.
		if ( ! empty( $posted_data['edd_sl_keys'] ) ) {
			update_post_meta( $post_id, '_edd_sl_keys', addslashes( $posted_data['edd_sl_keys'] ) );
		} else {
			delete_post_meta( $post_id, '_edd_sl_keys' );
		}

		if ( ! empty( $posted_data['edd_sl_no_keys_behavior'] ) ) {
			update_post_meta( $post_id, '_edd_sl_no_keys_behavior', 'draft' );
		} else {
			delete_post_meta( $post_id, '_edd_sl_no_keys_behavior' );
		}
	}

	/**
	 * Sanitize beta files.
	 *
	 * @since 3.9.0
	 * @param array[] $files Beta files array. File arrays have `file` and `name` keys.
	 * @return array
	 */
	private function sanitize_beta_files( $files ) {
		$sanitized_files = array();
		foreach ( $files as $id => $file ) {
			if ( empty( $file['file'] ) && empty( $file['name'] ) ) {
				continue;
			}
			$sanitized_files[ $id ]['name'] = esc_html( $files[ $id ]['name'] );
			$sanitized_files[ $id ]['file'] = sanitize_text_field( $files[ $id ]['file'] );
		}
		return $sanitized_files;
	}
}
