<?php
/**
 * Main section.
 *
 * @package EDD\SoftwareLicensing\Admin\Downloads\Editor
 * @since 3.9.0
 */

namespace EDD\SoftwareLicensing\Admin\Downloads\Editor;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Admin\Downloads\Editor\Section;
use EDD\HTML\Tooltip;

/**
 * Main section.
 *
 * @since 3.9.0
 */
class General extends Section {

	/**
	 * Section ID.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $id = 'general';

	/**
	 * Section priority.
	 *
	 * @since 3.9.0
	 * @var int
	 */
	protected $priority = 1;

	/**
	 * Section icon.
	 *
	 * @since 3.9.0
	 * @var string
	 */
	protected $icon = 'admin-settings';

	/**
	 * Get the section label.
	 *
	 * @since 3.9.0
	 * @return string
	 */
	public function get_label() {
		return esc_html__( 'General', 'edd_sl' );
	}

	/**
	 * Render the section.
	 *
	 * @since 3.9.0
	 * @return void
	 */
	public function render() {
		$download = $this->item;
		if ( ! current_user_can( 'edit_product', $download->id ) ) {
			return;
		}

		$limit            = get_post_meta( $download->id, '_edd_sl_limit', true );
		$is_limited       = empty( get_post_meta( $download->id, 'edd_sl_download_lifetime', true ) );
		$exp_length       = get_post_meta( $download->id, '_edd_sl_exp_length', true );
		$exp_unit         = get_post_meta( $download->id, '_edd_sl_exp_unit', true );
		$disable_discount = get_post_meta( $download->id, '_edd_sl_disable_renewal_discount', true );
		$discount         = get_post_meta( $download->id, '_edd_sl_renewal_discount', true );
		?>

		<div class="edd-form-group">
			<?php do_action( 'edd_sl_license_metabox_before_activation_limit', $download->id ); ?>

			<label for="edd_sl_limit" class="edd-form-group__label">
				<?php
				esc_html_e( 'Activation Limit', 'edd_sl' );
				$tooltip = new Tooltip(
					array(
						'title'   => __( 'Activation Limit', 'edd_sl' ),
						'content' => implode(
							'<br />',
							array(
								__( 'Set the number of activations allowed per license. To allow unlimited activations, leave this empty, or set it to 0.', 'edd_sl' ),
								'<br /><strong>' . __( 'Variable Pricing', 'edd_sl' ) . '</strong>',
								__( 'When variable pricing is enabled, activation limits set on each price variation take precedence over the activation limit set here.', 'edd_sl' ),
								'<br /><strong>' . __( 'Bundles', 'edd_sl' ) . '</strong>',
								__( 'If your product is a bundle, the activation limit set here will override the activation limits set on the individual products included in the bundle.', 'edd_sl' ),
							)
						),
					)
				);
				$tooltip->output();
				?>
			</label>
			<div class="edd-form-group__control">
				<input type="number" class="small-text" name="edd_sl_limit" id="edd_sl_limit" value="<?php echo absint( $limit ); ?>"/>
			</div>
			<?php do_action( 'edd_sl_license_metabox_after_activation_limit', $download->id ); ?>
		</div>

		<?php do_action( 'edd_sl_license_metabox_before_license_length', $download->id ); ?>
		<div class="edd-form-group__label">
			<?php
			esc_html_e( 'License Length', 'edd_sl' );
			$tooltip = new Tooltip(
				array(
					'title'   => __( 'License Length', 'edd_sl' ),
					'content' => __( 'How long are license keys valid for? Licenses can be set to never expire, or after a specific amount of time. Once expired, license keys cannot be activated without being renewed.', 'edd_sl' ),
				)
			);
			$tooltip->output();
			?>
		</div>
		<div class="edd-form-group">
			<div class="edd-form-group__control edd-toggle">
				<input
					type="checkbox"
					name="edd_sl_is_lifetime"
					id="edd_license_is_lifetime"
					value="1"
					<?php checked( false, $is_limited ); ?>
					class="edd-requirement"
					data-edd-requirement="lifetime-licenses"
				/>
				<label for="edd_license_is_lifetime"><?php esc_html_e( 'Licenses never expire.', 'edd_sl' ); ?></label>
			</div>
		</div>

		<div class="edd-form-group">
			<div class="edd-form-group__control" data-edd-requires-lifetime-licenses="false">
				<span class="edd-amount-type-wrapper">
					<?php
					$exp_length_input = new \EDD\HTML\Number(
						array(
							'name'         => 'edd_sl_exp_length',
							'id'           => 'edd_sl_exp_length',
							'value'        => esc_attr( $exp_length ),
							'placeholder'  => '1',
							'class'        => array( 'edd-amount-input', 'small-text', 'no-controls' ),
							'include_span' => false,
						)
					);
					$exp_length_input->output();
					?>
					<select name="edd_sl_exp_unit" id="edd_sl_exp_unit">
						<option value="days"<?php selected( 'days', $exp_unit ); ?>><?php esc_html_e( 'Days', 'edd_sl' ); ?></option>
						<option value="weeks"<?php selected( 'weeks', $exp_unit ); ?>><?php esc_html_e( 'Weeks', 'edd_sl' ); ?></option>
						<option value="months"<?php selected( 'months', $exp_unit ); ?>><?php esc_html_e( 'Months', 'edd_sl' ); ?></option>
						<option value="years"<?php selected( 'years', $exp_unit ); ?>><?php esc_html_e( 'Years', 'edd_sl' ); ?></option>
					</select>
				</span>
			</div>
		</div>

		<?php do_action( 'edd_sl_license_metabox_after_license_length', $download->id ); ?>

		<?php
		if ( edd_get_option( 'edd_sl_renewals', false ) ) {
			$disable_discount = get_post_meta( $download->id, '_edd_sl_disable_renewal_discount', true );
			?>
			<div class="edd-form-group" data-edd-requires-lifetime-licenses="false">
				<div class="edd-form-group">
					<?php do_action( 'edd_sl_license_metabox_before_renewal_discount', $download->id ); ?>
					<div class="edd-form-group__control edd-toggle">
						<input
							type="checkbox"
							name="edd_sl_disable_renewal_discount"
							id="edd_sl_disable_renewal_discount"
							value="1"
							<?php checked( true, $disable_discount ); ?>
							class="edd-requirement"
							data-edd-requirement="disable-renewal-discount"
						/>
						<label for="edd_sl_disable_renewal_discount"><?php esc_html_e( 'Disable renewal discounts.', 'edd_sl' ); ?></label>
					</div>
				</div>

				<div class="edd-form-group" data-edd-requires-disable-renewal-discount="false">
					<?php $discount = get_post_meta( $download->id, '_edd_sl_renewal_discount', true ); ?>
					<div class="edd-form-group__control">
						<label for="edd_sl_renewal_discount" class="edd-form-group__label">
							<?php
							esc_html_e( 'Renewal Discount', 'edd_sl' );
							$tooltip = new Tooltip(
								array(
									'title'   => __( 'When is this renewal discount used?', 'edd_sl' ),
									'content' => sprintf(
										/* translators: %1$s: HTML break tag. %2$d: Renewal discount value. */
										__( 'When a customer manually renews their existing license key, this discount is automatically applied.%1$s<strong>Current global value:</strong> %2$d%%', 'edd_sl' ),
										'<br /><br />',
										edd_get_option( 'edd_sl_renewal_discount', 0 )
									),
								)
							);
							$tooltip->output();
							?>
						</label>
						<span class="edd-amount-type-wrapper">
							<?php
							$renewal_discount_input = new \EDD\HTML\Number(
								array(
									'name'         => 'edd_sl_renewal_discount',
									'id'           => 'edd_sl_renewal_discount',
									'value'        => esc_attr( $discount ),
									'placeholder'  => esc_attr( edd_get_option( 'edd_sl_renewal_discount', 0 ) ),
									'class'        => array( 'edd__input', 'edd__input--left', 'small-text', 'no-controls' ),
									'step'         => 0.01,
									'min'          => 0,
									'max'          => 100,
									'include_span' => false,
								)
							);
							$renewal_discount_input->output();
							?>
							<span class="edd-input__symbol edd-input__symbol--suffix">%</span>
						</span>
					</div>
				</div>
				<?php
				do_action( 'edd_sl_license_metabox_after_renewal_discount', $download->id );
				?>
			</div>
			<?php
		}
	}
}
