<?php
/**
 * License Activations List Table
 *
 * @package   EDD\SoftwareLicensing\Admin
 * @copyright Copyright (c) 2024, Sandhills Development, LLC
 * @license   https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since     3.9.1
 */

namespace EDD\SoftwareLicensing\Admin;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Admin\List_Table;
use EDD\SoftwareLicensing\Database\Queries\LicenseActivation as ActivationQuery;

/**
 * License Activations List Table Class
 *
 * @since 3.9.1
 */
class ActivationsListTable extends List_Table {
	use Traits\ProductSelect;

	/**
	 * Array of activation counts.
	 *
	 * @var array
	 */
	public $counts = array();

	/**
	 * Cache for license objects to avoid duplicate queries.
	 *
	 * @var array
	 */
	private $license_cache = array();

	/**
	 * Cache for customer objects to avoid duplicate queries.
	 *
	 * @var array
	 */
	private $customer_cache = array();

	/**
	 * The class constructor.
	 *
	 * @since 3.9.1
	 */
	public function __construct() {
		parent::__construct(
			array(
				'singular' => 'activation',
				'plural'   => 'activations',
				'ajax'     => false,
			)
		);

		$this->get_counts();

		add_action( 'edd_admin_filter_bar_activations', array( $this, 'filter_bar_items' ) );
		add_action( 'edd_after_admin_filter_bar_activations', array( $this, 'filter_bar_searchbox' ) );
	}

	/**
	 * Gets the name of the primary column.
	 *
	 * @since 3.9.1
	 * @access protected
	 *
	 * @return string Name of the primary column.
	 */
	protected function get_primary_column_name() {
		return 'site_name';
	}

	/**
	 * Retrieve the table columns
	 *
	 * @since 3.9.1
	 * @return array $columns Array of all the list table columns
	 */
	public function get_columns() {
		return array(
			'site_name'    => __( 'Activation', 'edd_sl' ),
			'status'       => __( 'Status', 'edd_sl' ),
			'product'      => __( 'Product', 'edd_sl' ),
			'customer'     => __( 'Customer', 'edd_sl' ),
			'date_created' => __( 'First Request', 'edd_sl' ),
			'last_request' => __( 'Last Request', 'edd_sl' ),
		);
	}

	/**
	 * Retrieve the sortable columns
	 *
	 * @since 3.9.1
	 * @return array Array of all the sortable columns
	 */
	public function get_sortable_columns() {
		return array(
			'site_name'    => array( 'site_name', false ),
			'status'       => array( 'activated', false ),
			'date_created' => array( 'date_created', false ),
			'last_request' => array( 'last_request', false ),
		);
	}

	/**
	 * This function renders most of the columns in the list table.
	 *
	 * @since 3.9.1
	 *
	 * @param object $item        The current activation object.
	 * @param string $column_name The name of the column.
	 *
	 * @return string Column content
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {

			case 'status':
				$status       = $item->activated ? 'active' : 'inactive';
				$status_badge = new \EDD\Utils\StatusBadge(
					array(
						'status' => $status,
						'label'  => $item->activated ? __( 'Active', 'edd_sl' ) : __( 'Inactive', 'edd_sl' ),
						'class'  => "edd-sl-admin-activation-status-badge--{$status}",
					)
				);
				return $status_badge->get();

			case 'product':
				return $this->get_product_name( $item->license_id );

			case 'customer':
				return $this->get_customer_name( $item->license_id );

			case 'date_created':
				if ( empty( $item->date_created ) || '0000-00-00 00:00:00' === $item->date_created ) {
					return __( 'Unknown', 'edd_sl' );
				}
				return '<time datetime="' . esc_attr( $item->date_created ) . '">' .
					edd_date_i18n( $item->date_created, 'M. d, Y' ) . '<br>' .
					edd_date_i18n( $item->date_created, 'H:i' ) . ' ' .
					edd_get_timezone_abbr() .
					'</time>';

			case 'last_request':
				if ( empty( $item->last_request ) || '0000-00-00 00:00:00' === $item->last_request ) {
					return __( 'Unknown', 'edd_sl' );
				}
				return '<time datetime="' . esc_attr( $item->last_request ) . '">' .
					edd_date_i18n( $item->last_request, 'M. d, Y' ) . '<br>' .
					edd_date_i18n( $item->last_request, 'H:i' ) . ' ' .
					edd_get_timezone_abbr() .
					'</time>';

			default:
				return isset( $item->$column_name )
					? $item->$column_name
					: null;
		}
	}

	/**
	 * Render the site_name column
	 *
	 * @since 3.9.1
	 *
	 * @param object $item The current activation object.
	 * @return string
	 */
	public function column_site_name( $item ) {
		$site_name = ! empty( $item->site_name )
			? untrailingslashit( $item->site_name )
			: '&mdash;';

		// Concatenate and return.
		return '<strong>' . esc_html( $site_name ) . '</strong>' . $this->row_actions( $this->get_row_actions( $item ) );
	}

	/**
	 * Display advanced filters.
	 *
	 * @since 3.9.1
	 */
	public function advanced_filters() {
		edd_admin_filter_bar( 'activations' );
	}

	/**
	 * Outputs the filter bar searchbox.
	 *
	 * @since 3.9.1
	 * @return void
	 */
	public function filter_bar_searchbox() {
		$this->search_box( __( 'Search', 'edd_sl' ), 'edd-activations' );
	}

	/**
	 * Display filter bar items.
	 *
	 * @since 3.9.1
	 */
	public function filter_bar_items() {
		add_filter( 'edd_product_dropdown_args', array( $this, 'filter_product_dropdown' ) );
		$products = new \EDD\HTML\ProductSelect(
			array(
				'chosen'            => true,
				'show_option_none'  => false,
				'show_option_all'   => false,
				'selected'          => $this->get_product_id(),
				/* translators: %s: download label plural */
				'show_option_empty' => sprintf( __( 'All %s', 'edd_sl' ), edd_get_label_plural() ),
			)
		);
		$products->output();
		remove_filter( 'edd_product_dropdown_args', array( $this, 'filter_product_dropdown' ) );
		?>
		<div id="edd-after-core-filters">
			<input type="submit" class="button button-secondary" value="<?php esc_html_e( 'Filter', 'edd_sl' ); ?>"/>
			<?php
			if ( ! empty( $this->get_product_id() ) || ! empty( $this->get_status() ) ) :
				$clear_url = edd_get_admin_url(
					array(
						'page'      => 'edd-licenses',
						'page_type' => 'activations',
					)
				);
				?>
				<a href="<?php echo esc_url( $clear_url ); ?>" class="button-secondary">
					<?php esc_html_e( 'Clear', 'edd_sl' ); ?>
				</a>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * Generate the table navigation above or below the table.
	 * We're overriding this to turn off the referer param in `wp_nonce_field()`.
	 *
	 * @param string $which If we're rendering the top or bottom nav.
	 * @since 3.9.1
	 */
	protected function display_tablenav( $which ) {
		if ( 'top' === $which ) {
			wp_nonce_field( 'bulk-' . $this->_args['plural'], '_wpnonce', false );
		}
		?>
		<div class="tablenav <?php echo esc_attr( $which ); ?>">

			<?php if ( $this->has_items() ) : ?>
				<div class="alignleft actions bulkactions">
					<?php $this->bulk_actions( $which ); ?>
				</div>
				<?php
			endif;
			$this->extra_tablenav( $which );
			$this->pagination( $which );
			?>
		</div>
		<?php
	}

	/**
	 * Generate the bulk actions dropdown.
	 *
	 * @since 3.9.1
	 * @param string $which If we're rendering the top or bottom nav.
	 */
	protected function bulk_actions( $which = '' ) {
		if ( 'top' === $which ) {
			$this->views();
		}
	}

	/**
	 * Get row actions for an activation.
	 *
	 * @since 3.9.1
	 *
	 * @param object $item The current activation object.
	 * @return array
	 */
	private function get_row_actions( $item ) {
		$actions = array();

		if ( current_user_can( 'manage_licenses' ) ) {
			$actions['view'] = sprintf(
				'<a href="%s">%s</a>',
				esc_url(
					edd_get_admin_url(
						array(
							'page'       => 'edd-licenses',
							'license_id' => absint( $item->license_id ),
							'view'       => 'activations',
						)
					)
				),
				__( 'View Details', 'edd_sl' )
			);
		}

		return $actions;
	}

	/**
	 * Get product name for a license.
	 *
	 * @since 3.9.1
	 *
	 * @param int $license_id The license ID.
	 * @return string
	 */
	private function get_product_name( $license_id ) {
		$license = $this->get_license( $license_id );

		if ( ! $license || empty( $license->download_id ) ) {
			return '&mdash;';
		}

		$name = edd_get_download_name( $license->download_id );
		if ( ! $name ) {
			return '&mdash;';
		}

		if ( ! current_user_can( 'edit_product', $license->download_id ) ) {
			return esc_html( $name );
		}

		return sprintf(
			'<a href="%s">%s</a>',
			esc_url( get_edit_post_link( $license->download_id ) ),
			esc_html( $name )
		);
	}

	/**
	 * Get customer name for a license.
	 *
	 * @since 3.9.1
	 *
	 * @param int $license_id The license ID.
	 * @return string
	 */
	private function get_customer_name( $license_id ) {
		$license = $this->get_license( $license_id );

		if ( ! $license || empty( $license->customer_id ) ) {
			return '&mdash;';
		}

		$customer = $this->get_customer( $license->customer_id );

		if ( ! $customer ) {
			return '&mdash;';
		}

		$name = ! empty( $customer->name ) ? $customer->name : $customer->email;

		if ( current_user_can( 'view_shop_reports' ) ) {
			return sprintf(
				'<a href="%s">%s</a>',
				esc_url(
					add_query_arg(
						array(
							'post_type' => 'download',
							'page'      => 'edd-customers',
							'view'      => 'overview',
							'id'        => absint( $customer->id ),
						),
						admin_url( 'edit.php' )
					)
				),
				esc_html( $name )
			);
		}

		return esc_html( $name );
	}

	/**
	 * Get a license object with caching.
	 *
	 * @since 3.9.1
	 *
	 * @param int $license_id The license ID.
	 * @return \EDD_SL_License|false
	 */
	private function get_license( $license_id ) {
		if ( ! isset( $this->license_cache[ $license_id ] ) ) {
			$this->license_cache[ $license_id ] = edd_software_licensing()->get_license( $license_id );
		}

		return $this->license_cache[ $license_id ];
	}

	/**
	 * Get a customer object with caching.
	 *
	 * @since 3.9.1
	 *
	 * @param int $customer_id The customer ID.
	 * @return \EDD_Customer|false
	 */
	private function get_customer( $customer_id ) {
		if ( ! isset( $this->customer_cache[ $customer_id ] ) ) {
			$this->customer_cache[ $customer_id ] = edd_get_customer( $customer_id );
		}

		return $this->customer_cache[ $customer_id ];
	}

	/**
	 * Retrieve the total activation counts based on status.
	 *
	 * @since 3.9.1
	 * @return void
	 */
	public function get_counts() {
		$query = new ActivationQuery();

		// Get all activations count.
		$this->counts['all'] = $query->query(
			array(
				'count'         => true,
				'activated__in' => array( 0, 1 ),
			)
		);

		// Get active activations count.
		$this->counts['active'] = $query->query(
			array(
				'count'     => true,
				'activated' => 1,
			)
		);

		// Get inactive activations count.
		$this->counts['inactive'] = $query->query(
			array(
				'count'     => true,
				'activated' => 0,
			)
		);
	}

	/**
	 * Retrieve all the data for the table.
	 * Setup the final data for the table
	 *
	 * @since 3.9.1
	 * @return array Array of activation objects
	 */
	public function get_data(): array {
		$page     = $this->get_paged();
		$per_page = $this->per_page;
		$offset   = $per_page * ( $page - 1 );

		$args = array_merge(
			$this->get_args(),
			array(
				'number'  => $per_page,
				'offset'  => $offset,
				'orderby' => isset( $_GET['orderby'] ) ? sanitize_text_field( $_GET['orderby'] ) : 'date_created',
				'order'   => isset( $_GET['order'] ) ? sanitize_text_field( $_GET['order'] ) : 'DESC',
			)
		);

		$status = $this->get_status();
		switch ( $status ) {
			case 'active':
				$args['activated'] = 1;
				break;
			case 'inactive':
				$args['activated'] = 0;
				break;
			default:
				$args['activated__in'] = array( 0, 1 );
				break;
		}

		$query = new ActivationQuery();

		return $query->query( $args );
	}

	/**
	 * Setup the final data for the table
	 *
	 * @since 3.9.1
	 * @return void
	 */
	public function prepare_items() {
		$this->_column_headers = array(
			$this->get_columns(),
			array(),
			$this->get_sortable_columns(),
		);

		$this->items = $this->get_data();
		$total_items = $this->get_total();

		$this->set_pagination_args(
			array(
				'total_items' => $total_items,
				'per_page'    => $this->per_page,
				'total_pages' => ceil( $total_items / $this->per_page ),
			)
		);
	}

	/**
	 * Message to be displayed when there are no items.
	 *
	 * @since 3.9.1
	 */
	public function no_items() {
		esc_html_e( 'No license activations found.', 'edd_sl' );
	}

	/**
	 * Gets the views for the list table.
	 *
	 * @since 3.9.1
	 * @return array
	 */
	public function get_views(): array {
		$current = $this->get_status();

		return array(
			'all'      => sprintf(
				'<a href="%s"%s>%s <span class="count">(%s)</span></a>',
				esc_url(
					remove_query_arg( 'status' )
				),
				'' === $current ? ' class="current"' : '',
				__( 'All', 'edd_sl' ),
				number_format_i18n( $this->counts['all'] )
			),
			'active'   => sprintf(
				'<a href="%s"%s>%s <span class="count">(%s)</span></a>',
				esc_url(
					add_query_arg( 'status', 'active' )
				),
				'active' === $current ? ' class="current"' : '',
				__( 'Active', 'edd_sl' ),
				number_format_i18n( $this->counts['active'] )
			),
			'inactive' => sprintf(
				'<a href="%s"%s>%s <span class="count">(%s)</span></a>',
				esc_url(
					add_query_arg( 'status', 'inactive' )
				),
				'inactive' === $current ? ' class="current"' : '',
				__( 'Inactive', 'edd_sl' ),
				number_format_i18n( $this->counts['inactive'] )
			),
		);
	}

	/**
	 * Get the queried product ID.
	 *
	 * @since 3.9.1
	 * @return string|false
	 */
	private function get_product_id() {
		return isset( $_GET['products'] ) ? sanitize_text_field( $_GET['products'] ) : false;
	}

	/**
	 * Get the arguments for the query.
	 *
	 * @since 3.9.1
	 * @return array
	 */
	private function get_args(): array {
		$args   = array();
		$search = $this->get_search();
		if ( $search ) {
			// Check if search looks like an email address.
			if ( is_email( $search ) ) {
				// Get customer IDs matching this email.
				$email_addresses = edd_get_customer_email_addresses(
					array(
						'email'  => $search,
						'fields' => 'customer_id',
						'number' => 9999,
					)
				);

				if ( ! empty( $email_addresses ) ) {
					$args['customer_id__in'] = array_unique( array_map( 'absint', $email_addresses ) );
				} else {
					// No customers found, return empty results.
					$args['id'] = 0;
				}
			} else {
				// Not an email, search site_name.
				$args['search'] = $search;
			}
		}

		$product_id = $this->get_product_id();
		if ( $product_id ) {
			$args['download_id'] = $product_id;
		}

		return $args;
	}

	/**
	 * Get the total number of activations.
	 *
	 * @since 3.9.1
	 * @return int
	 */
	private function get_total(): int {
		$args          = $this->get_args();
		$args['count'] = true;
		$query         = new ActivationQuery( $args );

		return $query->found_items;
	}
}
