<?php
/**
 * Main Software Licensing Class
 *
 * @package   edd-software-licensing
 * @copyright Copyright (c) 2021, Sandhills Development, LLC
 * @license   GPL2+
 */

class EDD_Software_Licensing {
	use EDD\SoftwareLicensing\Deprecated\Core;
	use EDD\SoftwareLicensing\Licenses\Traits\Generator;

	/**
	 * @var EDD_Software_Licensing The one true EDD_Software_Licensing
	 * @since 1.5
	 */
	private static $instance;

	/**
	 * @var EDD\SoftwareLicensing\Legacy\Database\License
	 * @since 3.6
	 */
	public $licenses_db;

	/**
	 * @var EDD\SoftwareLicensing\Legacy\Database\Meta
	 * @since 3.6
	 */
	public $license_meta_db;

	/**
	 * @var EDD\SoftwareLicensing\Legacy\Database\Activations
	 * @since 3.6
	 */
	public $activations_db;

	/**
	 * @var EDD_SL_Roles
	 * @since 3.6
	 */
	public $roles;

	/**
	 * @const FILE
	 */
	const FILE = EDD_SL_PLUGIN_FILE;

	/**
	 * @var EDD\SoftwareLicensing\Emails\Notices
	 * @since 3.8.12
	 */
	public $notices;

	/**
	 * Main EDD_Software_Licensing Instance
	 *
	 * Insures that only one instance of EDD_Software_Licensing exists in memory at any one
	 * time. Also prevents needing to define globals all over the place.
	 *
	 * @since     1.4
	 * @static
	 * @staticvar array $instance
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof EDD_Software_Licensing ) ) {
			self::$instance = new EDD_Software_Licensing();
			new EDD\SoftwareLicensing\Core();

			self::$instance->includes();
			self::$instance->actions();
			self::$instance->licensing();

			self::$instance->licenses_db     = new EDD\SoftwareLicensing\Legacy\Database\License();
			self::$instance->license_meta_db = new EDD\SoftwareLicensing\Legacy\Database\Meta();
			self::$instance->activations_db  = new EDD\SoftwareLicensing\Legacy\Database\Activations();
			self::$instance->roles           = new EDD_SL_Roles();
			self::$instance->notices         = new EDD\SoftwareLicensing\Emails\Notices();
		}

		return self::$instance;
	}

	/**
	 * Load the includes for EDD SL
	 *
	 * @since  3.2.4
	 * @return void
	 */
	private function includes() {

		require_once EDD_SL_PLUGIN_DIR . 'includes/deprecated/class-names.php';
		require_once EDD_SL_PLUGIN_DIR . 'includes/deprecated/functions.php';

		if ( defined( 'WP_CLI' ) && WP_CLI ) {
			require_once EDD_SL_PLUGIN_DIR . '/includes/integrations/wp-cli.php';
		}

		include_once EDD_SL_PLUGIN_DIR . 'includes/misc-functions.php';

		if ( is_admin() ) {

			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/customers.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/metabox.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/settings.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/export.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/reports.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/upgrades.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/licenses.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/license-actions.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/license-functions.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/payment-filters.php';
			include_once EDD_SL_PLUGIN_DIR . 'includes/admin/deprecated-admin-functions.php';
		}

		include_once EDD_SL_PLUGIN_DIR . 'includes/scripts.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/errors.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/post-types.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/widgets.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/templates.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/license-upgrades.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/license-actions.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/license-emails.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/license-renewals.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/readme.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/shortcodes.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/rest-api.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/filters.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/staged-rollouts.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/classes/class-sl-emails.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/classes/class-sl-changelog-widget.php';
		include_once EDD_SL_PLUGIN_DIR . 'includes/classes/class-sl-roles.php';
	}

	public function actions() {

		add_action( 'init', array( $this, 'localization' ), -1 );

		add_action( 'init', array( $this, 'load_api_endpoint' ) );

		// creates / stores a license during purchase for EDD 1.6+
		add_action( 'edd_complete_download_purchase', array( $this, 'generate_license' ), 0, 5 );

		// Revokes license keys on payment status change (if needed)
		add_action( 'edd_transition_order_item_status', array( $this, 'revoke' ), 10, 3 );

		// Delete license keys on order deletion.
		add_action( 'edd_order_deleted', array( $this, 'delete_license' ), 10, 1 );

		// Renews a license on purchase
		add_action( 'edd_complete_download_purchase', array( $this, 'process_renewal' ), 0, 4 );

		// Add /changelog enpoint
		add_action( 'init', array( $this, 'changelog_endpoint' ) );

		// Display a plain-text changelog
		add_action( 'template_redirect', array( $this, 'show_changelog' ), -999 );

		// Prevent downloads on purchases with expired keys.
		add_action( 'edd_process_verified_download', array( $this, 'prevent_expired_downloads' ), 10, 4 );

		// Reduce query load for EDD API calls
		add_action( 'after_setup_theme', array( $this, 'reduce_query_load' ) );

		add_action( 'edd_updated_edited_purchase', array( $this, 'update_licenses_on_payment_update' ) );

		add_action( 'user_register', array( $this, 'add_past_license_keys_to_new_user' ), 50 );

		// Register email templates.
		new EDD\SoftwareLicensing\Emails\Registry();
		new EDD\SoftwareLicensing\Emails\Tags();
	}

	/**
	 * Sets up licensing with EDD core.
	 *
	 * @since 3.8.3
	 */
	private function licensing() {
		add_action(
			'edd_extension_license_init',
			function ( \EDD\Extensions\ExtensionRegistry $registry ) {
				$registry->addExtension( EDD_SL_PLUGIN_FILE, 'Software Licensing', 4916, EDD_SL_VERSION, 'edd_sl_license_key' );
			}
		);
	}

	/**
	 * Load the localization files
	 *
	 * @since  3.2.4
	 * @return void
	 */
	public function localization() {
		load_plugin_textdomain( 'edd_sl', false, dirname( plugin_basename( EDD_SL_PLUGIN_FILE ) ) . '/languages/' );
	}

	/**
	 * Load API endpoint.
	 *
	 * @return void
	 */
	public function load_api_endpoint() {
		// If this is an API Request, load the endpoint
		if ( ! is_admin() && $this->is_api_request() !== false && ! defined( 'EDD_SL_DOING_API_REQUEST' ) ) {
			$request_type = $this->get_api_endpoint();

			if ( ! empty( $request_type ) ) {
				$request_class = str_replace( '_', ' ', $request_type );
				$request_class = 'EDD_SL_' . ucwords( $request_class );
				$request_class = str_replace( ' ', '_', $request_class );

				if ( class_exists( $request_class ) ) {
					define( 'EDD_SL_DOING_API_REQUEST', true );
					$api_request = new $request_class();
					$api_request->process_request();
				}

				/**
				 * Allow further processing of requests.
				 *
				 * @since 3.6
				 *
				 * @param string $request_type  Type of API request.
				 * @param string $request_class Class that will handle the API request.
				 */
				do_action( 'edd_sl_load_api_endpoint', $request_type, $request_class );
			}
		}
	}

	/**
	 * The whitelisted endpoints for the Software Licensing
	 *
	 * @since  3.2.4
	 * @return array Array of endpoints whitelisted for EDD SL
	 */
	private function allowed_api_endpoints() {
		$default_endpoints = array(
			'package_download',
		);

		return apply_filters( 'edd_sl_allowed_api_endpoints', $default_endpoints );
	}

	/**
	 * Verify an endpoint is the one being requested
	 *
	 * @since  3.2.4
	 * @param  string $endpoint The endpoint to check
	 * @return boolean           If the endpoint provided is the one currently being requested
	 */
	private function is_endpoint_active( $endpoint = '' ) {
		$is_active = stristr( $_SERVER['REQUEST_URI'], 'edd-sl/' . $endpoint ) !== false;

		if ( $is_active ) {
			$is_active = true;
		}

		/**
		 * Filter whether or not the endpoint is active.
		 *
		 * @since 3.6
		 *
		 * @param bool   $is_active Is the endpoint active?
		 * @param string $endpoint  Endpoint to check.
		 */
		$is_active = apply_filters( 'edd_sl_is_endpoint_active', $is_active, $endpoint );

		return (bool) $is_active;
	}

	/**
	 * Is this a request we should respond to?
	 *
	 * @since  3.2.4
	 * @return bool
	 */
	private function is_api_request() {
		$trigger = false;

		$allowed_endpoints = $this->allowed_api_endpoints();

		foreach ( $allowed_endpoints as $endpoint ) {

			$trigger = $this->is_endpoint_active( $endpoint );

			if ( $trigger ) {
				$trigger = true;
				break;
			}
		}

		return (bool) apply_filters( 'edd_sl_is_api_request', $trigger );
	}

	/**
	 * Parse the API endpoint being requested
	 *
	 * @since  3.2.4
	 * @return string The endpoint being requested
	 */
	private function get_api_endpoint() {
		$url_parts = parse_url( $_SERVER['REQUEST_URI'] );
		$paths     = explode( '/', $url_parts['path'] );
		$endpoint  = '';
		foreach ( $paths as $index => $path ) {
			if ( 'edd-sl' === $path ) {
				$endpoint = $paths[ $index + 1 ];
				break;
			}
		}

		/**
		 * Allow the API endpoint to be filtered.
		 *
		 * @since 3.6
		 *
		 * @param string $endpoint API endpoint.
		 */
		$endpoint = apply_filters( 'edd_sl_get_api_endpoint', $endpoint );

		return $endpoint;
	}

	/**
	 * Retrieve a EDD_SL_License object by ID or key
	 *
	 * @since  3.5
	 * @since  3.8.7 When using an ID, if a valid license is not found, return false.
	 * @param  $id_or_key string|int License key or license ID
	 * @param  $by_key    bool       True if retrieving with a key instead of ID
	 * @return EDD_SL_License|bool  License object if found. False if not found.
	 */
	public function get_license( $id_or_key, $by_key = false ) {
		if ( $by_key || ! is_numeric( $id_or_key ) ) {
			$result    = edd_software_licensing()->licenses_db->get_column_by( 'id', 'license_key', sanitize_text_field( $id_or_key ) );
			$id_or_key = is_numeric( $result ) ? (int) $result : false;
		}

		if ( empty( $id_or_key ) ) {
			return false;
		}

		$id      = $id_or_key;
		$license = new EDD_SL_License( $id );

		// The ID used does not exist as a valid license.
		if ( empty( $license->ID ) && false === $license->exists ) {
			return false;
		}

		return $license;
	}

	/*
	|--------------------------------------------------------------------------
	| License Renewal
	|--------------------------------------------------------------------------
	*/

	/**
	 * @param int    $download_id
	 * @param int    $payment_id
	 * @param string $type (unused)
	 * @param array  $cart_item
	 * @return void
	 */
	public function process_renewal( $download_id = 0, $payment_id = 0, $type = 'default', $cart_item = array() ) {

		// Bail if this is not a renewal item.
		if ( empty( $cart_item['item_number']['options']['is_renewal'] ) ) {
			return;
		}

		$license_id = ! empty( $cart_item['item_number']['options']['license_id'] ) ? absint( $cart_item['item_number']['options']['license_id'] ) : false;

		if ( $license_id ) {

			$license = $this->get_license( $license_id );

			if ( false === $license ) {
				return;
			}

			$license->renew( $payment_id );

		}
	}

	/**
	 * @param int $license_id
	 * @param int $payment_id
	 * @param int $download_id
	 * @return void
	 */
	public function renew_license( $license_id = 0, $payment_id = 0, $download_id = 0 ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		return $license->renew( $payment_id );
	}

	/**
	 * Retrieve the renewal URL for a license key
	 *
	 * @since  3.4
	 * @param int $license_id
	 * @return string The renewal URL
	 */
	public function get_renewal_url( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return '';
		}

		return $license->get_renewal_url();
	}

	/**
	 * Determine if a license is allowed to be extended.
	 * Note: any usage of this should be replaced with EDD_SL_License::can_extend()
	 * as this method may be deprecated in the future.
	 *
	 * @since  3.4.7
	 * @since  3.8.12 Updated to use EDD_SL_License method.
	 * @param int $license_id
	 * @return bool
	 */
	public function can_extend( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		return $license ? $license->can_extend() : false;
	}

	/**
	 * Determine if a license is allowed to be renewed after its expiration.
	 * Note: any usage of this should be replaced with EDD_SL_License::can_renew()
	 * as this method may be deprecated in the future.
	 *
	 * @since  3.5.4
	 * @since  3.8.12   Updated to use EDD_SL_License method.
	 * @param int $license_id
	 * @return bool
	 */
	public function can_renew( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		return $license ? $license->can_renew() : false;
	}

	/*
	|--------------------------------------------------------------------------
	| Revoke License
	|--------------------------------------------------------------------------
	*/

	/**
	 * Revokes a license key when the order item status indicates that it should be.
	 * Hooked into the Berlin transition hook on the order item status.
	 *
	 * @param string $old_status
	 * @param string $new_status
	 * @param int    $order_item_id
	 * @return void
	 */
	public function revoke( $old_status, $new_status, $order_item_id ) {
		$revokable_order_statuses = apply_filters( 'edd_sl_revoke_license_statuses', array( 'revoked', 'refunded', 'on_hold', 'trash'  ) );
		if ( ! in_array( $new_status, $revokable_order_statuses, true ) ) {
			return;
		}
		$order_item = edd_get_order_item( $order_item_id );
		$order_id   = $order_item->order_id;

		$licenses = $this->get_licenses_of_purchase( $order_id );
		if ( ! $licenses ) {
			return;
		}
		foreach ( $licenses as $license ) {
			if ( $license->download_id != $order_item->product_id ) {
				continue;
			}
			if ( $license->cart_index != $order_item->cart_index ) {
				continue;
			}
			$is_upgrade = (bool) edd_get_order_item_meta( $order_item->id, '_option_is_upgrade', true );
			if ( $is_upgrade ) {
				$payment_index  = array_search( $order_id, $license->payment_ids );
				$previous_items = false;

				// Work our way backwards through the payment IDs until we find the first completed payment, ignoring others
				$key = $payment_index - 1;
				while ( $key >= 0 ) {
					$previous_items = edd_get_order_items(
						array(
							'order_id'   => $license->payment_ids[ $key ],
							'status__in' => edd_get_deliverable_order_item_statuses(),
							'cart_index' => $license->cart_index,
						)
					);
					if ( ! empty( $previous_items ) ) {
						break;
					}

					--$key;
				}

				if ( empty( $previous_items ) ) {
					continue;
				}
				foreach ( $previous_items as $previous_item ) {
					// Set the download ID to the initial download/price ID.
					$license->download_id = $previous_item->product_id;
					$license->price_id    = $previous_item->price_id;

					edd_delete_order_meta( $previous_item->order_id, '_edd_sl_upgraded_to_payment_id' );
					break;
				}

				// Reset the activation limits.
				$license->reset_activation_limit();
			} else {
				$is_renewal = (bool) edd_get_order_item_meta( $order_item->id, '_option_is_renewal', true );
				if ( $is_renewal && 'expired' !== $license->status ) {
					continue;
				}

				$license->revoke( $order_id );
			}
		}
	}

	/*
	|--------------------------------------------------------------------------
	| Delete License
	|--------------------------------------------------------------------------
	*/

	/**
	 * Delete licenses associated with a payment/order.
	 *
	 * @since 3.2.6
	 *
	 * @param int $order_id The payment/order ID.
	 * @param int $download_id Optional. The download ID to delete licenses for. Default 0.
	 * @return bool $license_deleted Wether the delete operation was successful.
	 */
	public function delete_license( $order_id, $download_id = 0 ) {
		// Ensure we have a valid order ID.
		if ( empty( $order_id ) ) {
			return false;
		}
		$license_deleted = false;
		// Handle specific download ID deletion.
		if ( ! empty( $download_id ) ) {
			$license_deleted = $this->delete_license_for_download( $order_id, $download_id );
		} else {
			$license_deleted = $this->delete_all_licenses_for_order( $order_id );
		}

		return $license_deleted;
	}

	/*
	|--------------------------------------------------------------------------
	| Version Checking
	|--------------------------------------------------------------------------
	*/

	/**
	 * @param int $item_id
	 *
	 * @return bool|mixed
	 */
	public function get_latest_version( $item_id ) {
		return $this->get_download_version( $item_id );
	}

	/*
	|--------------------------------------------------------------------------
	| Logging Functions
	|--------------------------------------------------------------------------
	*/

	/**
	 * @param string $license_id
	 *
	 * @return array|bool
	 */
	public function get_license_logs( $license_id = '' ) {
		if ( $license = $this->get_license( $license_id ) ) {
			return $license->get_logs();
		}

		return false;
	}

	/**
	 * Logs a license activation.
	 *
	 * @param int   $license_id   The license ID.
	 * @param array $server_data The server data.
	 */
	public function log_license_activation( $license_id, $server_data ) {
		$license = $this->get_license( $license_id );
		if ( $license ) {
			$license->add_log( __( 'LOG - License Activated: ', 'edd_sl' ) . $license_id, $server_data );
		}
	}

	/**
	 * Logs a license deactivation.
	 *
	 * @param int   $license_id  The license ID.
	 * @param array $server_data The server data.
	 */
	public function log_license_deactivation( $license_id, $server_data ) {
		$license = $this->get_license( $license_id );
		if ( $license ) {
			$license->add_log( __( 'LOG - License Deactivated: ', 'edd_sl' ) . $license_id, $server_data );
		}
	}

	/*
	|--------------------------------------------------------------------------
	| Site tracking
	|--------------------------------------------------------------------------
	*/

	/**
	 * @param int $license_id
	 *
	 * @return array
	 */
	public function get_sites( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		return $license->sites;
	}

	/**
	 * @param int $license_id
	 *
	 * @return mixed|void
	 */
	public function get_site_count( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}
		return $license->activation_count;
	}

	/**
	 * @param int    $license_id
	 * @param string $site_url
	 *
	 * @return bool|mixed|void
	 */
	public function is_site_active( $license_id = 0, $site_url = '' ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		return $license->is_site_active( $site_url );
	}

	/**
	 * @param int    $license_id
	 * @param string $site_url
	 *
	 * @return bool|int
	 */
	public function insert_site( $license_id = 0, $site_url = '' ) {

		if ( empty( $license_id ) ) {
			return false;
		}

		if ( empty( $site_url ) ) {
			return false;
		}

		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		return (bool) $license->add_site( $site_url );
	}

	/**
	 * @param int    $license_id
	 * @param string $site_url
	 *
	 * @return bool|int
	 */
	public function delete_site( $license_id = 0, $site_url = '' ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		return $license->remove_site( $site_url );
	}

	/*
	|--------------------------------------------------------------------------
	| Misc Functions
	|--------------------------------------------------------------------------
	*/

	/**
	 * @param int $download_id
	 *
	 * @return mixed
	 */
	public function get_new_download_license_key( $download_id = 0 ) {
		$download = new EDD_SL_Download( $download_id );
		return $download->get_new_license_key();
	}

	/**
	 * @param string $license_key
	 *
	 * @return bool|null|string
	 */
	public function get_license_by_key( $license_key ) {
		$license = $this->get_license( $license_key, true );

		if ( false === $license ) {
			return false;
		}

		return $license->ID;
	}

	/**
	 * @param int $license_id
	 *
	 * @return bool|mixed
	 */
	public function get_license_key( $license_id ) {
		$license = $this->get_license( $license_id );

		if ( ! $license ) {
			return false;
		}

		return $license->key;
	}

	/**
	 * @param string $license_key
	 *
	 * @return mixed|void
	 */
	public function get_download_id_by_license( $license_key ) {
		$license = $this->get_license( $license_key, true );

		if ( false === $license ) {
			return false;
		}

		$download_id = $license->download_id;

		return apply_filters( 'edd_sl_get_download_id_by_license', $download_id, $license_key, $license->ID );
	}

	/**
	 * Retrieves the download ID by the name.
	 *
	 * @param  string $name Download name
	 * @since  3.4.4
	 * @return int     Download ID
	 */
	public function get_download_id_by_name( $name = '' ) {

		$download_id = false;
		$download    = new WP_Query(
			array(
				'post_type'              => 'download',
				'title'                  => urldecode( $name ),
				'post_status'            => 'all',
				'posts_per_page'         => 1,
				'no_found_rows'          => true,
				'ignore_sticky_posts'    => true,
				'update_post_term_cache' => false,
				'update_post_meta_cache' => false,
				'orderby'                => 'post_date ID',
				'order'                  => 'ASC',
			)
		);

		if ( ! empty( $download->post ) ) {
			$download_id = $download->post->ID;
		}

		return apply_filters( 'edd_sl_get_download_id_by_name', $download_id, $name );
	}

	/**
	 * Check if the license key is attributed to the download id given.
	 * Constant EDD_BYPASS_ITEM_ID_CHECK can bypass this check if true.
	 *
	 * @param  integer $download_id Download/Item ID (post_id)
	 * @param  string  $license_key License key
	 * @param  bool    $bypass_constant Allows a way to bypass the constant for cases outside of the download process
	 * @return bool               true/false
	 */
	public function is_download_id_valid_for_license( $download_id = 0, $license_key = '', $bypass_constant = false ) {

		$license_download = (int) $this->get_download_id_by_license( $license_key );

		if ( defined( 'EDD_BYPASS_ITEM_ID_CHECK' ) && EDD_BYPASS_ITEM_ID_CHECK && true !== $bypass_constant ) {
			$license_match = true;
		} else {
			$license_match = (bool) ( $license_download === (int) $download_id );
		}

		return apply_filters( 'edd_sl_id_license_match', $license_match, $download_id, $license_download, $license_key );
	}

	/**
	 * Returns the name of the download ID
	 *
	 * @param int $license_id
	 * @since 3.4
	 * @return int
	 */
	public function get_download_name( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		return $license->get_download()->get_name();
	}

	/**
	 * Gets the download name for display with a license key.
	 * Prepends name for child licenses with --
	 * Appends the variation name if applicable.
	 *
	 * @param object \EDD_SL_License $license
	 * @return string
	 */
	public function get_license_download_display_name( $license ) {
		$license_title = $this->get_download_name( $license->ID );

		// Return the title if this is a single product license and does not have a variation/price ID.
		if ( ! $license->parent && ! $license->price_id ) {
			return $license_title;
		}

		// If this is a child license, prepend the title with a --
		if ( $license->parent ) {
			$license_title = '&#8212;&nbsp;' . $license_title;
		}

		// Return the title if there is definitely no price ID.
		if ( ! $license->price_id ) {
			return $license_title;
		}

		// Final checks to see if the price ID truly should be shown, or was erroneously assigned.
		$show_price_id = true;
		if ( $license->parent ) {
			$download_id   = $this->get_download_id( $license->parent );
			$download      = new EDD_SL_Download( $download_id );
			$bundled_items = $download->get_bundled_downloads();
			// Loose comparison because the ID is an integer, bundled items are strings.
			if ( in_array( $license->download_id, $bundled_items ) ) {
				$show_price_id = false;
			}
		}
		if ( $license->price_id && $show_price_id ) {
			$license_title .= sprintf(
				'<span class="edd_sl_license_price_option">&nbsp;&ndash;&nbsp;%s</span>',
				esc_html( edd_get_price_option_name( $license->get_download()->ID, $license->price_id ) )
			);
		}

		return $license_title;
	}

	/**
	 * Returns the download ID of a license key
	 *
	 * @since 2.7
	 * @param int $license_id
	 * @return int
	 */
	public function get_download_id( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		return $license->download_id;
	}

	/**
	 * Returns the user ID (if any) the license belongs to, if none is found in post meta
	 * it retrieves it from the payment and populates the post meta
	 *
	 * @access public
	 * @since  3.4.8
	 * @param  int $license_id
	 * @return int
	 */
	public function get_user_id( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		return $license->user_id;
	}

	/**
	 * Returns the price ID for a license key
	 *
	 * @since 3.3.
	 * @param int $license_id
	 *
	 * @return int
	 */
	public function get_price_id( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		return $license->price_id;
	}

	/**
	 * Returns the payment ID of a license key
	 *
	 * @since 3.4
	 * @param int $license_id
	 * @return int
	 */
	public function get_payment_id( $license_id = 0 ) {
		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		return $license->payment_id;
	}

	/**
	 * @param int $payment_id
	 *
	 * @return EDD_SL_License[]|bool
	 */
	public function get_licenses_of_purchase( $payment_id ) {

		$number = 99999;
		// If 0 is provided as the payment ID, limit the number of licenses returned.
		if ( 0 === $payment_id ) {
			$number = 10;
		}

		// Get licenses where this payment ID was the initial purchase.
		$licenses = self::$instance->licenses_db->get_licenses(
			array(
				'number'     => $number,
				'payment_id' => $payment_id,
				'parent'     => 0,
			)
		);

		if ( empty( $licenses ) ) {
			return false;
		}

		$licenses_to_return = array();

		foreach ( $licenses as $license ) {
			$licenses_to_return[] = $license;

			// If child licenses are present for bundles, they should be placed after their parent.
			$child_licenses = $license->get_child_licenses();
			if ( ! empty( $child_licenses ) ) {
				foreach ( $child_licenses as $child_license ) {
					$licenses_to_return[] = $child_license;
				}
			}
		}

		return $licenses_to_return;
	}

	/**
	 * @param int   $purchase_id
	 * @param int   $download_id
	 * @param mixed $cart_index
	 * @param bool  $allow_children If we should return child licenses if found on the payment containing a bundle
	 *
	 * @return EDD_SL_License|bool Returns license, if found. If not, returns false
	 */
	public function get_license_by_purchase( $purchase_id = 0, $download_id = 0, $cart_index = false, $allow_children = true ) {

		$args = array(
			'number' => 1,
		);

		if ( ! empty( $purchase_id ) ) {
			$args['payment_id'] = $purchase_id;
		}

		if ( ! empty( $download_id ) ) {
			$args['download_id'] = $download_id;
		}

		if ( false !== $cart_index ) {
			$args['cart_index'] = $cart_index;
		}

		if ( false === $allow_children ) {
			$args['parent'] = 0;
		}

		$licenses = self::$instance->licenses_db->get_licenses( $args );

		if ( ! empty( $licenses ) ) {
			$license = $licenses[0];
			return apply_filters( 'edd_sl_licenses_by_purchase', $license, $purchase_id, $download_id, $cart_index );
		}

		return false;
	}

	/**
	 * Retrieve all license keys for a user
	 *
	 * @param int                                                        $user_id The user ID to get licenses for
	 * @param bool                                                       $include_child_licenses If true (default) we will get all licenses including children of a bundle
	 *                                                                                           when false, the method will only return licenses without a parent
	 *
	 * @since 3.4
	 * @param  $user_id     int The ID of the user to filter by
	 * @param  $download_id int The ID of a download to filter by
	 * @param  $status      string The license status to filter by, or all
	 * @return array
	 */
	public function get_license_keys_of_user( $user_id = 0, $download_id = 0, $status = 'any', $include_child_licenses = true ) {

		if ( empty( $user_id ) ) {
			$user_id = get_current_user_id();
		}

		if ( empty( $user_id ) ) {
			return array();
		}

		$customer = new EDD_Customer( $user_id, true );

		$args = array(
			'number'      => 50,
			'customer_id' => $customer->id,
			'orderby'     => 'date_created',
			'order'       => 'DESC',
		);

		if ( ! empty( $download_id ) ) {
			$args['download_id'] = $download_id;
		}

		$status = strtolower( $status );
		if ( $status !== 'all' && $status !== 'any' ) {
			$args['status'] = $status;
		}

		if ( false === $include_child_licenses ) {
			$args['parent'] = 0;
		}

		/**
		 * Filters the arguments for the query to get the license keys of a user.
		 *
		 * @param array $args The arguments for get_posts
		 * @param int   $user_id The user this query is for.
		 */
		$args = apply_filters( 'edd_sl_get_license_keys_of_user_args', $args, $user_id );

		$license_keys = edd_software_licensing()->licenses_db->get_licenses( $args );

		// "License" was improperly capitalized. Filter corrected but typo maintained for backwards compatibility
		$license_keys = apply_filters( 'edd_sl_get_License_keys_of_user', $license_keys, $user_id );

		return apply_filters( 'edd_sl_get_license_keys_of_user', $license_keys, $user_id );
	}

	/**
	 * Given a license ID, return any child licenses it may have
	 *
	 * @since 3.4.8
	 * @param int $parent_license_id The parent license ID to look up
	 *
	 * @return array Array of child license objects.
	 */
	public function get_child_licenses( $parent_license_id = 0 ) {
		$license = $this->get_license( $parent_license_id );
		if ( false === $license ) {
			return array();
		}

		return $license->get_child_licenses();
	}

	/**
	 * @param int $license_id
	 *
	 * @return string
	 */
	public function get_license_status( $license_id ) {
		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		return $license->status;
	}

	/**
	 * Returns the status label
	 *
	 * @param int $license_id
	 *
	 * @since 2.7
	 * @return string
	 */
	public function license_status( $license_id ) {
		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		return $license->get_display_status();
	}

	/**
	 * @param int    $license_id
	 * @param string $status
	 */
	public function set_license_status( $license_id, $status = 'active' ) {
		if ( empty( $license_id ) ) {
			return;
		}

		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		$updated = $license->status = $status;

		return $updated;
	}

	/**
	 * @param int $license_id
	 * @param int $payment_id
	 * @param int $download_id
	 *
	 * @return string
	 */
	public function get_license_length( $license_id = 0, $payment_id = 0, $download_id = 0 ) {
		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		return $license->license_length();
	}

	/**
	 * @param int $license_id
	 *
	 * @return bool
	 */
	public function is_lifetime_license( $license_id ) {
		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		return $license->is_lifetime;
	}

	/**
	 * @param int $license_id
	 *
	 * @return bool|mixed|string
	 */
	public function get_license_expiration( $license_id ) {
		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		return $license->expiration;
	}

	/**
	 * @param int $license_id
	 * @param int $expiration
	 *
	 * @return void
	 */
	public function set_license_expiration( $license_id, $expiration ) {

		if ( empty( $license_id ) ) {
			return;
		}

		$license = $this->get_license( $license_id );

		if ( false == $license ) {
			return false;
		}

		// $expiration should be a valid timestamp
		$license->expiration = $expiration;
	}

	/**
	 * @param int $license_id
	 * @return void
	 */
	public function set_license_as_lifetime( $license_id ) {

		if ( empty( $license_id ) ) {
			return;
		}

		$license = $this->get_license( $license_id );

		if ( false === $license ) {
			return false;
		}

		$license->is_lifetime = true;
	}

	/**
	 * @param int $download_id
	 * @param int $license_id
	 *
	 * @return mixed|void
	 */
	public function get_license_limit( $download_id = 0, $license_id = 0 ) {
		// TODO: Set a deprecated notice when download_id isn't empty

		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		return $license->activation_limit;
	}

	/**
	 * Returns the license activation limit in a readable format
	 *
	 * @param int $license_id
	 * @since 2.7
	 * @return string|int
	 */
	public function license_limit( $license_id = 0 ) {
		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		return $license->license_limit();
	}

	/**
	 * @param int  $download_id
	 * @param null $price_id
	 *
	 * @return bool|int
	 */
	public function get_price_activation_limit( $download_id = 0, $price_id = null ) {
		$download = new EDD_SL_Download( $download_id );
		return $download->get_price_activation_limit( $price_id );
	}

	/**
	 * @param int $download_id
	 * @param int $price_id
	 *
	 * @return bool
	 */
	public function get_price_is_lifetime( $download_id = 0, $price_id = null ) {
		$download = new EDD_SL_Download( $download_id );
		return $download->is_price_lifetime( $price_id );
	}

	/**
	 * @param int $license_id
	 * @param int $download_id
	 *
	 * @return bool
	 */
	public function is_at_limit( $license_id = 0, $download_id = 0 ) {
		$license = $this->get_license( $license_id );
		if ( false === $license ) {
			return false;
		}

		return $license->is_at_limit();
	}

	/**
	 * Whether an order is a renewal order.
	 *
	 * @param int $order_id
	 * @return bool
	 */
	public function is_renewal( $order_id = 0 ) {
		return ! empty( edd_get_order_meta( $order_id, '_edd_sl_is_renewal', true ) );
	}

	/**
	 * Sanitize the item names to be able to compare them properly (else we get problems with HTML special characters created
	 * by WordPress like hyphens replaced by long dashes
	 *
	 * @param int    $download_id
	 * @param string $item_name
	 * @return boolean
	 * @since 2.5
	 */
	public function check_item_name( $download_id = 0, $item_name = 0, $license = null ) {
		$download = new EDD_SL_Download( $download_id );

		$match = false;

		if ( $download->ID > 0 ) {
			$tmp_name  = sanitize_title( urldecode( $item_name ) );
			$tmp_title = sanitize_title( $download->get_name() );

			$match = $tmp_title == $tmp_name;
		}

		return apply_filters( 'edd_sl_check_item_name', $match, $download_id, $item_name, $license );
	}

	/**
	 * @param $download_id
	 *
	 * @return bool|mixed
	 */
	public function get_download_version( $download_id ) {
		$download = new EDD_SL_Download( $download_id );

		if ( empty( $download->ID ) ) {
			return false;
		}

		return $download->get_version();
	}

	/**
	 * @param int $download_id Download (Post) ID
	 *
	 * @return bool|mixed
	 */
	public function get_beta_download_version( $download_id ) {
		$download = new EDD_SL_Download( $download_id );

		if ( empty( $download->ID ) ) {
			return false;
		}

		return $download->get_beta_version();
	}

	/**
	 * @param int    $download_id
	 * @param string $license_key
	 * @param string $url
	 * @param bool   $download_beta
	 *
	 * @return mixed|void
	 */
	public function get_encoded_download_package_url( $download_id = 0, $license_key = '', $url = '', $download_beta = false ) {
		$package_download = new EDD\SoftwareLicensing\Downloads\Package();

		return $package_download->get_encoded_download_package_url( $download_id, $license_key, $url, $download_beta );
	}

	/**
	 * @param int    $download_id
	 * @param string $license_key
	 * @param string $hash
	 * @param int    $expires
	 */
	public function get_download_package( $download_id, $license_key, $hash, $expires = 0 ) {
		$package = new EDD\SoftwareLicensing\Downloads\Package();
		return $package->get_download_package( $download_id, $license_key, $hash, $expires );
	}

	/**
	 * Force activation count increase
	 *
	 * This checks whether we should always count activations
	 *
	 * By default activations are tied to URLs so that a single URL is not counted as two separate activations.
	 * Desktop software, for example, is not tied to a URL so it can't be counted in the same way.
	 *
	 * @param int $license_id
	 * @access      private
	 * @since       1.3.9
	 * @return      bool
	 */
	public function force_increase( $license_id = 0 ) {
		$force_increase = edd_get_option( 'edd_sl_force_increase', false );

		return (bool) apply_filters( 'edd_sl_force_activation_increase', $force_increase, $license_id );
	}

	/**
	 * Add the /changelog enpoint
	 *
	 * Allows for the product changelog to be shown as plain text
	 *
	 * @access      public
	 * @since       1.7
	 */
	public function changelog_endpoint() {
		add_rewrite_endpoint( 'changelog', EP_PERMALINK );
	}

	/**
	 * Displays a changelog
	 *
	 * @access      public
	 * @since       1.7
	 */
	public function show_changelog() {

		global $wp_query;

		if ( ! isset( $wp_query->query_vars['changelog'] ) || ! isset( $wp_query->query_vars['download'] ) ) {
			return;
		}

		$download = get_page_by_path( $wp_query->query_vars['download'], OBJECT, 'download' );

		if ( ! is_object( $download ) || 'download' != $download->post_type ) {
			return;
		}

		$download = new EDD_SL_Download( $download->ID );

		$changelog = $download->get_changelog();

		if ( $changelog ) {
			echo $changelog;
		} else {
			_e( 'No changelog found', 'edd_sl' );
		}

		exit;
	}

	/**
	 * Prevent file downloads on expired license keys.
	 *
	 * @since 2.3
	 *
	 * @param int    $download_id
	 * @param string $email
	 */
	public function prevent_expired_downloads( $download_id, $email, $payment_id, $args ) {
		$can_download_response = $this->license_can_download( $download_id, $email, $payment_id, $args );

		if ( false === $can_download_response['success'] ) {
			$defaults = array(
				'message'  => __( 'You do not have a valid license for this download.', 'edd_sl' ),
				'title'    => __( 'No Valid License', 'edd_sl' ),
				'response' => 403,
			);

			$can_download_response = wp_parse_args( $can_download_response, $defaults );
			wp_die( $can_download_response['message'], $can_download_response['title'], $can_download_response['response'] );
		}
	}

	/**
	 * Return an array of data for if a user has the ability to be delivered a file via a download link.
	 *
	 * Triggers on the edd_process_verified_download hook in EDD Core.
	 *
	 * @since 3.6
	 *
	 * @param int    $download_id The download ID.
	 * @param string $email       The email address of the user.
	 * @param int    $payment_id  The payment ID.
	 * @param array  $args        The arguments passed to the function.
	 *
	 * @return array $args {
	 *     @type bool   $success If the download is available, true for yes, false for no.
	 *     @type string $message (Required for success => false) A message to display during wp_die
	 *     @type string $title (Required for success => false) A title to display in the browser <title> tag during wp_die
	 *     @type int    $response (Required for success => false) The HTTP response code to use for wp_die
	 * }
	 */
	public function license_can_download( $download_id, $email, $payment_id, $args ) {
		$can_download     = array( 'success' => true );
		$invalid_statuses = apply_filters( 'edd_sl_license_download_invalid_statuses', array( 'expired', 'disabled' ) );

		$query    = new \EDD\SoftwareLicensing\Database\Queries\License();
		$licenses = $query->query(
			array(
				'download_id' => $download_id,
				'payment_id'  => $payment_id,
			)
		);

		if ( count( $licenses ) === 1 ) {
			$license = $licenses[0];
			if ( 'expired' === $license->status ) {
				$can_download = array(
					'success'  => false,
					'message'  => __( 'Your license key for this purchase is expired. Renew your license key and you will be allowed to download your files again.', 'edd_sl' ),
					'title'    => __( 'Expired License', 'edd_sl' ),
					'response' => 401,
				);
			} elseif ( 'disabled' === $license->status ) {
				$can_download = array(
					'success'  => false,
					'message'  => __( 'Your license key for this purchase has been revoked.', 'edd_sl' ),
					'title'    => __( 'Revoked License', 'edd_sl' ),
					'response' => 401,
				);
			}
		} elseif ( count( $licenses ) > 1 ) {
			$has_access = false;
			foreach ( $licenses as $license ) {
				if ( ! in_array( $license->status, $invalid_statuses, true ) ) {
					$has_access = true;
					break;
				}
			}

			if ( false === $has_access ) {
				$can_download = array(
					'success'  => false,
					'message'  => __( 'You do not have a valid license for this download.', 'edd_sl' ),
					'title'    => __( 'No Valid License', 'edd_sl' ),
					'response' => 401,
				);
			}
		}

		// If the download is not available, check if the customer has a valid license for the download from another purchase.
		if ( ! $can_download['success'] && ! empty( $email ) ) {
			$customer = edd_get_customer_by( 'email', $email );
			if ( ! empty( $customer->id ) ) {
				$licenses = $query->query(
					array(
						'customer_id'    => $customer->id,
						'number'         => 99999,
						'download_id'    => $download_id,
						'status__not_in' => $invalid_statuses,
					)
				);
				if ( ! empty( $licenses ) ) {
					$can_download['success'] = true;
				}
			}
		}

		return apply_filters( 'edd_sl_license_can_download', $can_download, $args );
	}

	/**
	 * Removes the queries caused by `widgets_init` for remote API calls (and for generating the download)
	 *
	 * @return void
	 */
	public function reduce_query_load() {

		if ( ! isset( $_REQUEST['edd_action'] ) ) {
			return;
		}

		$actions = array(
			'activate_license',
			'deactivate_license',
			'get_version',
			'package_download',
			'check_license',
		);

		if ( in_array( $_REQUEST['edd_action'], $actions ) ) {
			remove_all_actions( 'widgets_init' );
		}
	}

	/**
	 * Updates license details when a payment is updated.
	 *
	 * @param int $payment_id
	 *
	 * @return void
	 */
	public function update_licenses_on_payment_update( $payment_id ) {

		$customer_id = edd_get_payment_customer_id( $payment_id );
		$customer    = edd_get_customer( $customer_id );
		if ( ! $customer ) {
			return;
		}

		$licenses = $this->get_licenses_of_purchase( $payment_id );
		if ( empty( $licenses ) ) {
			return;
		}

		foreach ( $licenses as $license ) {
			$license->update(
				array(
					'customer_id' => $customer->id,
					'user_id'     => $customer->user_id,
				)
			);
		}
	}

	/**
	 * Lowercases site URL's, strips HTTP protocols and strips www subdomains.
	 *
	 * @param string $url
	 *
	 * @return string
	 */
	public function clean_site_url( $url ) {

		$url = strtolower( $url );

		if ( apply_filters( 'edd_sl_strip_www', true ) ) {

			// strip www subdomain
			$url = str_replace( array( '://www.', ':/www.' ), '://', $url );

		}

		if ( apply_filters( 'edd_sl_strip_protocol', apply_filters( 'edd_sl_strip_protocal', true ) ) ) {
			// strip protocol
			$url = str_replace( array( 'http://', 'https://', 'http:/', 'https:/' ), '', $url );

		}

		if ( apply_filters( 'edd_sl_strip_port_number', true ) ) {

			$port = parse_url( $url, PHP_URL_PORT );

			if ( $port ) {

				// strip port number
				$url = str_replace( ':' . $port, '', $url );
			}
		}

		return sanitize_text_field( $url );
	}

	/**
	 * Looks up license keys by email that match the registering user.
	 *
	 * This is for users that purchased as a guest and then came
	 * back and created an account.
	 *
	 * @access      public
	 * @since       3.1
	 * @param      int $user_id the new user's ID
	 * @return      void
	 */
	public function add_past_license_keys_to_new_user( $user_id ) {

		$user     = get_user_by( 'id', $user_id );
		$customer = edd_get_customer_by( 'email', $user->user_email );
		if ( ! $customer ) {
			return;
		}
		$licenses = self::$instance->licenses_db->get_licenses(
			array(
				'customer_id' => $customer->id,
				'number'      => 99999,
			)
		);
		if ( empty( $licenses ) ) {
			return;
		}

		foreach ( $licenses as $license ) {
			// The license already has a user ID; don't change it.
			if ( ! empty( $license->user_id ) ) {
				continue;
			}

			$license->update(
				array(
					'user_id' => $user_id,
				)
			);
		}
	}

	/**
	 * Check if a URL is considered a local one.
	 *
	 * @since 3.2.7
	 *
	 * @param string $url         A URL that possibly represents a local environment.
	 * @param string $environment The current site environment. Default production.
	 *
	 * @return bool True if the URL is local, false otherwise.
	 */
	public function is_local_url( $url = '', $environment = 'production' ) {
		$is_local_url = false;

		if ( 'production' !== $environment ) {
			return (bool) apply_filters( 'edd_sl_is_local_url', true, $url, $environment );
		}

		// Trim it up.
		$url = strtolower( trim( $url ) );

		// Need to get the host...so let's add the scheme so we can use parse_url.
		if ( false === strpos( $url, 'http://' ) && false === strpos( $url, 'https://' ) ) {
			$url = 'http://' . $url;
		}

		$url_parts = wp_parse_url( $url );
		$host      = ! empty( $url_parts['host'] ) ? $url_parts['host'] : false;

		if ( ! empty( $url ) && ! empty( $host ) ) {

			if ( false !== ip2long( $host ) ) {
				if ( ! filter_var( $host, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE ) ) {
					$is_local_url = true;
				}
			} elseif ( 'localhost' === $host ) {
				$is_local_url = true;
			} else {
				$utility      = new \EDD\SoftwareLicensing\Utils\URL();
				$is_local_url = $utility->is_local_url( $url, $host );
			}
		}

		return (bool) apply_filters( 'edd_sl_is_local_url', $is_local_url, $url, $environment );
	}

	/**
	 * Get emails for a license
	 *
	 * This is currently only used for matching
	 * on renewals with the Enforced Matching setting enabled.
	 *
	 * @since 3.5
	 * @access public
	 * @param int $license_id The ID to get emails for
	 * @return array $emails The emails for this license
	 */
	public function get_emails_for_license( $license_id = 0 ) {
		$payment_id = $this->get_payment_id( $license_id );
		$order      = edd_get_order( $payment_id );
		$customer   = edd_get_customer( $order->customer_id );

		$emails   = $customer->emails;
		$emails[] = $customer->email;
		$emails[] = $order->email;
		$emails   = array_unique( $emails );

		return apply_filters( 'edd_sl_get_emails_for_license', $emails, $license_id );
	}

	/**
	 * Send the API response data as a JSON response, and define the JSON_REQUEST and WP_REDIS_DISABLE_COMMENT constants.
	 *
	 * @since 3.6.12
	 * @param array $response_data The data to send to the api.
	 */
	private function send_response( $response_data = array() ) {
		if ( ! defined( 'JSON_REQUEST' ) ) {
			define( 'JSON_REQUEST', true );
		}

		if ( ! defined( 'WP_REDIS_DISABLE_COMMENT' ) ) {
			define( 'WP_REDIS_DISABLE_COMMENT', true );
		}

		wp_send_json( $response_data );
	}

	/**
	 * Delete licenses for a specific download within an order.
	 *
	 * @since 3.9.0
	 *
	 * @param int $order_id The payment/order ID.
	 * @param int $download_id The download ID.
	 * @return bool $license_deleted Wether the delete operation was successful.
	 */
	private function delete_license_for_download( $order_id, $download_id ) {
		$order = edd_get_order( $order_id );
		if ( ! $order || ! $order instanceof \EDD\Orders\Order ) {
			return false;
		}
		if ( ! in_array( $order->status, array( 'complete', 'revoked' ), true ) ) {
			return false;
		}

		$licenses = $this->get_licenses_of_purchase( $order->id );
		if ( ! $licenses ) {
			return false;
		}

		$license_deleted = false;
		foreach ( $licenses as $license ) {
			if ( ! $this->is_download_id_valid_for_license( $download_id, $license->key, true ) ) {
				continue;
			}

			/*
			 * If this is not the initial payment for the license, don't delete it.
			 * Instead, roll back the expiration and remove the payment ID from the license payment IDs.
			 */
			if ( (int) $order_id !== (int) $license->payment_id ) {
				$this->rollback_license_for_order( $license, $order );
				continue;
			}

			// Delete the license with context.
			$context = array(
				'order_id'    => $order->id,
				'download_id' => $download_id,
			);
			$license_deleted = $license->delete( $context );
			break;
		}

		return $license_deleted;
	}

	/**
	 * Delete all licenses for an order.
	 *
	 * @since 3.9.0
	 *
	 * @param int $order_id The payment/order ID.
	 * @return bool $license_deleted Wether the delete operation was successful.
	 */
	private function delete_all_licenses_for_order( $order_id ) {

		$licenses = $this->get_licenses_of_purchase( $order_id );
		if ( ! $licenses ) {
			return false;
		}

		$license_deleted = false;
		foreach ( $licenses as $license ) {
			if ( empty( $license ) || ! $license instanceof \EDD\SoftwareLicensing\Licenses\License ) {
				continue;
			}

			// For order destruction, always delete the license.
			$context = array( 'order_id' => $order_id );
			$license_deleted = $license->delete( $context );
		}

		return $license_deleted;
	}

	/**
	 * Roll back license for a specific order.
	 *
	 * @since 3.9.0
	 *
	 * @param \EDD\SoftwareLicensing\Licenses\License $license The license object.
	 * @param \EDD\Orders\Order $order The order object.
	 * @return void
	 */
	private function rollback_license_for_order( $license, $order ) {
		// Validate license object
		if ( ! $license || ! $license instanceof \EDD\SoftwareLicensing\Licenses\License ) {
			return false;
		}

		// Validate order object
		if ( ! $order || ! $order instanceof \EDD\Orders\Order ) {
			return false;
		}

		$order_id = $order->id;

		// Roll back the expiration date on the license.
		$license->expiration = strtotime( '-' . $license->license_length(), $license->expiration );

		// Delete this payment ID from the license meta.
		edd_software_licensing()->license_meta_db->delete_meta( $license->id, '_edd_sl_payment_id', $order_id );

		// Delete the payment date from any meta (for upgrades and renewals).
		foreach ( $order->items as $item ) {
			$license_id = edd_get_order_item_meta( $item->id, '_option_license_id', true );
			if ( ! $license_id || (int) $license_id !== (int) $license->id ) {
				continue;
			}

			$action = false;
			if ( ! empty( edd_get_order_item_meta( $item->id, '_option_is_upgrade', true ) ) ) {
				$action = 'upgrade';
			} elseif ( ! empty( edd_get_order_item_meta( $item->id, '_option_is_renewal', true ) ) ) {
				$action = 'renewal';
			}

			if ( ! empty( $action ) && ! empty( $order->date_completed ) ) {
				$meta_key = '_edd_sl_' . $action . '_date';
				edd_software_licensing()->license_meta_db->delete_meta( $license->id, $meta_key, $order->date_completed );

				break; // We don't need to iterate on the items anymore.
			}
		}
	}
}
