<?php
/**
 * License Management
 *
 * @package     EDD\SoftwareLicensing\Admin\License
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Renders the main Licenses admin page.
 *
 * @since       1.0
 * @return      void
 */
function edd_sl_licenses_page() {
	// If we have an ID, we're viewing a specific license.
	if ( ! empty( filter_input( INPUT_GET, 'license_id', FILTER_VALIDATE_INT ) ) ) {
		$default_views  = edd_sl_license_views();
		$requested_view = isset( $_GET['view'] ) ? sanitize_text_field( $_GET['view'] ) : 'overview';

		if ( array_key_exists( $requested_view, $default_views ) && is_callable( $default_views[ $requested_view ] ) ) {
			edd_sl_render_license_view( $requested_view, $default_views );
		} else {
			edd_sl_render_license_view( 'overview', $default_views );
		}
		return;
	}

	edd_sl_licenses_list( filter_input( INPUT_GET, 'page_type', FILTER_SANITIZE_SPECIAL_CHARS ) );
}

/**
 * Register the views for license management
 *
 * @since  3.5
 * @return array Array of views and their callbacks
 */
function edd_sl_license_views() {
	$views = array(
		'overview'    => 'edd_sl_licenses_view',
		'logs'        => 'edd_sl_licenses_logs_view',
		'downloads'   => 'edd_sl_licenses_downloads_view',
		'activations' => 'edd_sl_licenses_activations_view',
	);
	$views = apply_filters( 'edd_sl_license_views', $views );
	if ( current_user_can( 'delete_licenses' ) ) {
		$views['delete'] = 'edd_sl_licenses_delete_view';
	}

	return $views;
}

/**
 * Register the tabs for license management
 *
 * @since  3.5
 * @return array Array of tabs for the customer
 */
function edd_sl_license_tabs() {
	$tabs = array(
		'overview'    => array(
			'dashicon' => 'dashicons-lock',
			'title'    => __( 'Details', 'edd_sl' ),
		),
		'logs'        => array(
			'dashicon' => 'dashicons-book',
			'title'    => __( 'Logs', 'edd_sl' ),
		),
		'downloads'   => array(
			'dashicon' => 'dashicons-download',
			'title'    => __( 'Downloads', 'edd_sl' ),
		),
		'activations' => array(
			'dashicon' => 'dashicons-admin-site',
			'title'    => __( 'Activations', 'edd_sl' ),
		),
	);

	$tabs = apply_filters( 'edd_sl_license_tabs', $tabs );
	if ( current_user_can( 'delete_licenses' ) ) {
		$tabs['delete'] = array(
			'dashicon' => 'dashicons-trash',
			'title'    => __( 'Delete', 'edd_sl' ),
		);
	}

	return $tabs;
}

/**
 * List table of licenses
 *
 * @since 3.5
 * @param string $active_tab The currently active tab. Default is 'licenses'.
 * @return void
 */
function edd_sl_licenses_list( $active_tab = 'licenses' ) {

	// Get the possible pages.
	$pages = EDD\SoftwareLicensing\Admin\Pages::get();

	// Reset page if not a registered page.
	if ( ! in_array( $active_tab, array_keys( $pages ), true ) ) {
		$active_tab = 'licenses';
	}

	// Get the label/name from the active tab.
	$name = $pages[ $active_tab ];

	// Get the action url from the active tab.
	$action_url = edd_get_admin_url(
		array(
			'page_type' => sanitize_key( $active_tab ),
			'page'      => 'edd-licenses',
		)
	);

	// Setup the list table class.
	switch ( $active_tab ) {
		case 'activations':
			$list_table_class = 'EDD\SoftwareLicensing\Admin\ActivationsListTable';
			break;

		case 'licenses':
		default:
			class_alias( \EDD\SoftwareLicensing\Admin\ListTable::class, 'EDD_SL_List_Table' );
			$list_table_class = 'EDD\SoftwareLicensing\Admin\ListTable';
			break;
	}

	// Initialize the list table.
	$list_table = new $list_table_class();
	$list_table->prepare_items();
	EDD\SoftwareLicensing\Admin\Pages::do_nav( $active_tab );
	?>

	<div class="wrap">
		<h1 class="wp-heading-inline"><?php echo esc_html( $name ); ?></h1>
		<hr class="wp-header-end">

		<?php edd_sl_show_errors(); ?>

		<form id="licenses-filter" method="get" action="<?php echo esc_url( $action_url ); ?>">
			<?php
			$list_table->advanced_filters();
			$list_table->display();
			?>
			<input type="hidden" name="post_type" value="download" />
			<input type="hidden" name="page" value="edd-licenses" />
			<input type="hidden" name="page_type" value="<?php echo esc_attr( $active_tab ); ?>" />
		</form>
	</div>
	<?php

	$redirect = get_transient( '_edd_sl_bulk_actions_redirect' );

	if ( false !== $redirect ) :
		delete_transient( '_edd_sl_bulk_actions_redirect' );
		$redirect = admin_url( 'edit.php?post_type=download&page=edd-licenses' );

		if ( ! empty( $_GET['s'] ) ) {
			$redirect = add_query_arg( 's', $_GET['s'], $redirect );
		}
		?>

	<script type="text/javascript">
	window.location = "<?php echo $redirect; ?>";
	</script>
		<?php
	endif;
}

/**
 * Renders the license view wrapper
 *
 * @since  3.5
 * @param  string $view      The View being requested
 * @param  array  $callbacks  The Registered views and their callback functions
 * @return void
 */
function edd_sl_render_license_view( $view, $callbacks ) {
	require_once 'views/license.php';
}

/**
 * View a license
 *
 * @since  3.5
 * @param  $license The License object being displayed
 * @return void
 */
function edd_sl_licenses_view( $license ) {
	require_once 'views/overview.php';
}

/**
 * View logs for a license
 *
 * @since  3.5
 * @param  EDD_SL_License $license The License object being displayed
 * @return void
 */
function edd_sl_licenses_logs_view( $license ) {
	require_once 'views/logs.php';
}

/**
 * View activations for a license
 *
 * @since  3.9.0
 * @param  EDD_SL_License $license The License object being displayed
 * @return void
 */
function edd_sl_licenses_activations_view( $license ) {
	require_once 'views/activations.php';
}

/**
 * Delete a license
 *
 * @since  3.5
 * @param  $license The License object being displayed
 * @return void
 */
function edd_sl_licenses_delete_view( $license ) {
	require_once 'views/delete.php';
}

/**
 * View downloads for a license
 *
 * @since 3.9.0
 * @param  EDD_SL_License $license The License object being displayed
 * @return void
 */
function edd_sl_licenses_downloads_view( $license ) {
	require_once 'views/downloads.php';
}

/**
 * Gets the license status admin badge.
 *
 * @since 3.9.0
 * @param string $status
 * @return string
 */
function edd_sl_get_license_status_badge( $status ) {
	$icon           = '';
	$license_status = $status;
	switch ( $status ) {

		case 'disabled':
			$status = 'warning';
			$icon   = 'no-alt';
			break;
		case 'expired':
			$icon = 'backup';
			break;
	}
	$statuses     = edd_sl_get_license_statuses();
	$status_badge = new \EDD\Utils\StatusBadge(
		array(
			'status' => $status,
			'label'  => array_key_exists( $license_status, $statuses ) ? $statuses[ $license_status ] : $license_status,
			'icon'   => $icon,
			'class'  => "edd-sl-admin-license-status-badge--{$status}",
		)
	);

	return $status_badge->get();
}
