<?php
/**
 * Provider.php
 *
 * @package   edd-git-download-updater
 * @copyright Copyright (c) 2021, Easy Digital Downloads
 * @license   GPL2+
 * @since     1.3
 */

namespace EDD\GitDownloadUpdater\Providers;

use EDD\GitDownloadUpdater\Exceptions\ResourceNotFoundException;

interface Provider {

	/**
	 * Unique ID for this provider.
	 *
	 * @since 1.3
	 *
	 * @return string
	 */
	public static function getId();

	/**
	 * Returns an ApiHandler object for this provider.
	 *
	 * @since 1.3
	 *
	 * @return ApiHandler
	 */
	public function getApiHandler();

	/**
	 * Returns a list of repositories, formatted like this:
	 *
	 * array(
	 *  'repo_owner' => array(
	 *      'https://source.com/url/to-repo' => 'repo name',
	 *  )
	 * )
	 *
	 * Each owner has an array of repos. The repos are keyed by the URL to
	 * the repo, and with the display name as the value.
	 *
	 * @since 1.3
	 *
	 * @return array
	 */
	public function getRepositories();

	/**
	 * Retrieves a list of tags available in the given repository.
	 * Should throw a `ResourceNotFound` exception in the event that no tags are available.
	 *
	 * @since 1.3
	 *
	 * @param string $repoPath Repository path in the format: {org}/{repo}
	 *
	 * @return array
	 * @throws ResourceNotFoundException
	 */
	public function getTags( $repoPath );

	/**
	 * Builds the URL to download an asset from the repository path and tag name.
	 *
	 * @since 1.3
	 *
	 * @param string $repoPath Repository path in the format: {org}/{repo}
	 * @param string $tag      Tag name.
	 *
	 * @return string
	 */
	public function buildAssetUrlFromRepoAndTag( $repoPath, $tag );

	/**
	 * Fetches a zip file from its URL.
	 *
	 * @since 1.3
	 *
	 * @param string $url Full URL (domain included) to the zip file.
	 *
	 * @return string Response body.
	 */
	public function fetchZipFromUrl( $url );

}
