<?php
/**
 * BitBucket.php
 *
 * @package   edd-git-download-updater
 * @copyright Copyright (c) 2021, Easy Digital Downloads
 * @license   GPL2+
 * @since     1.3
 */

namespace EDD\GitDownloadUpdater\Providers;

use EDD\GitDownloadUpdater\Exceptions\ApiException;
use EDD\GitDownloadUpdater\Exceptions\ConfigurationException;
use EDD\GitDownloadUpdater\Exceptions\MissingConnectionException;
use EDD\GitDownloadUpdater\Exceptions\ResourceNotFoundException;

class BitBucketProvider implements Provider {

	const API_URL = 'https://api.bitbucket.org/2.0';

	/**
	 * @var ApiHandler
	 */
	protected $apiHandler;

	public function __construct() {
		$this->apiHandler = $this->getApiHandler();
	}

	/**
	 * @inheritDoc
	 *
	 * @return string
	 */
	public static function getId() {
		return 'bitbucket';
	}

	/**
	 * @inheritDoc
	 *
	 * @return ApiHandler
	 */
	public function getApiHandler() {
		$handler = new ApiHandler();

		$username = edd_get_option( 'bb_username' );
		$password = edd_get_option( 'bb_app_password' );
		if ( ! empty( $username ) && ! empty( $password ) ) {
			$handler->withAuthorizationHeader(
				sprintf(
					'Basic %s',
					base64_encode( $username . ':' . $password )
				)
			);
		} elseif ( defined( 'EDD_GIT_BB_USER' ) && defined( 'EDD_GIT_BB_PASSWORD' ) ) {
			$handler->withAuthorizationHeader(
				sprintf(
					'Basic %s',
					base64_encode( EDD_GIT_BB_USER . ':' . EDD_GIT_BB_PASSWORD )
				)
			);
		}

		$handler->withApiUrl( self::API_URL )
				->withHeader( 'Connection', 'keep-alive' );

		return $handler;
	}

	/**
	 * Fetches all repositories from the API.
	 *
	 * @since 1.3
	 *
	 * @return array
	 * @throws ApiException|ConfigurationException|MissingConnectionException
	 */
	public function getRepositories() {
		$hasMore = true;
		$page    = 1;
		$repos   = array();

		while ( $hasMore ) {
			$response = $this->apiHandler->makeRequest(
				'user/permissions/repositories?per_page=100&page=' . urlencode( $page )
			);

			$hasMore = ! empty( $response->values );

			if ( is_array( $response->values ) ) {
				foreach ( $response->values as $repository ) {
					$repoSlug  = $repository->repository->full_name;
					$repoParts = explode( '/', $repoSlug );

					if ( empty( $repoParts[0] ) || empty( $repoParts[1] ) ) {
						continue;
					}

					$owner = $repoParts[0];

					if ( ! array_key_exists( $owner, $repos ) ) {
						$repos[ $owner ] = array();
					}

					if ( ! empty( $repository->repository->links->html->href ) ) {
						$repos[ $owner ][ $repository->repository->links->html->href ] = $repoParts[1];
					}
				}
			}

			++$page;
		}

		return $repos;
	}

	/**
	 * Retrieves a list of tags from the API.
	 *
	 * @since 1.3
	 *
	 * @param string $repoPath Path of the repository. Format: {org}/{repo}
	 *
	 * @return array
	 * @throws ResourceNotFoundException|ApiException|ConfigurationException|MissingConnectionException
	 */
	public function getTags( $repoPath ) {
		$tags = array();

		$url = 'repositories/' . $repoPath . '/refs/tags/?pagelen=100&sort=-name';

		while ( ! empty( $url ) ) {

			$response = $this->apiHandler->makeRequest( $url );

			if ( empty( $response->values ) || ! is_array( $response->values ) ) {
				throw new ResourceNotFoundException( esc_html__( 'No tags found for this repository.', 'edd-git-download-updater' ) );
			}

			$tags = array_merge( $tags, wp_list_pluck( $response->values, 'name' ) );

			// Check for the "next" URL in the response to paginate.
			$url = isset( $response->next ) ? str_replace( self::API_URL, '', $response->next ) : null;
		}

		return $tags;
	}

	/**
	 * @inheritDoc
	 *
	 * @param string $repoPath
	 * @param string $tag
	 *
	 * @return string
	 */
	public function buildAssetUrlFromRepoAndTag( $repoPath, $tag ) {
		return 'https://bitbucket.org/' . $repoPath . '/get/' . urlencode( $tag ) . '.zip';
	}

	/**
	 * @inheritDoc
	 *
	 * @param string $url
	 *
	 * @return string
	 * @throws ApiException|ConfigurationException|MissingConnectionException
	 */
	public function fetchZipFromUrl( $url ) {
		$this->apiHandler->withApiUrl( '' )->makeRequest( $url );

		$contentType = wp_remote_retrieve_header( $this->apiHandler->lastResponse, 'content-type' );
		if ( false === strpos( $contentType, 'application/zip' ) ) {
			throw new ApiException( 'Invalid content type: ' . $contentType, 401 );
		}

		return wp_remote_retrieve_body( $this->apiHandler->lastResponse );
	}
}
