<?php
/**
 * ApiHandler.php
 *
 * @package   edd-git-download-updater
 * @copyright Copyright (c) 2021, Easy Digital Downloads
 * @license   GPL2+
 * @since     1.3
 */

namespace EDD\GitDownloadUpdater\Providers;

use EDD\GitDownloadUpdater\Exceptions\ApiException;
use EDD\GitDownloadUpdater\Exceptions\ConfigurationException;
use EDD\GitDownloadUpdater\Exceptions\MissingConnectionException;

class ApiHandler {

	/**
	 * Headers that will be sent with the request.
	 *
	 * @since 1.3
	 *
	 * @var array
	 */
	protected $headers = array();

	/**
	 * Base API URL.
	 *
	 * @since 1.3
	 *
	 * @var string
	 */
	protected $apiBaseUrl;

	/**
	 * Last response from the API (unmodified).
	 *
	 * @since 1.3
	 *
	 * @var array|\WP_Error
	 */
	public $lastResponse;

	/**
	 * HTTP response code returned from the last request.
	 *
	 * @since 1.3
	 *
	 * @var int
	 */
	public $lastResponseCode;

	/**
	 * Adds a header.
	 *
	 * @since 1.3
	 *
	 * @param string $headerName
	 * @param string $headerValue
	 *
	 * @return $this
	 */
	public function withHeader( $headerName, $headerValue ) {
		$this->headers[ $headerName ] = $headerValue;

		return $this;
	}

	/**
	 * Adds an Authorization header.
	 *
	 * @since 1.3
	 *
	 * @param string $value
	 *
	 * @return $this
	 */
	public function withAuthorizationHeader( $value ) {
		return $this->withHeader( 'Authorization', $value );
	}

	/**
	 * Sets the base API URL.
	 *
	 * @since 1.3
	 *
	 * @param string $url
	 *
	 * @return $this
	 */
	public function withApiUrl( $url ) {
		$this->apiBaseUrl = ! empty( $url ) ? trailingslashit( $url ) : '';

		return $this;
	}

	/**
	 * Makes a request to the API.
	 *
	 * @param string     $endpoint Endpoint to query.
	 * @param string     $method   HTTP method.
	 * @param mixed|null $body     Body to send with the request.
	 *
	 * @return string Response body, JSON decoded.
	 * @throws ApiException|ConfigurationException|MissingConnectionException
	 */
	public function makeRequest( $endpoint, $method = 'GET', $body = null ) {
		if ( ! isset( $this->apiBaseUrl ) ) {
			throw new ConfigurationException( 'Missing API base URL.', 400 );
		}

		if ( empty( $this->headers['Authorization'] ) ) {
			throw new MissingConnectionException( 'Missing Authorization header.', 401 );
		}

		$this->lastResponse = wp_safe_remote_request(
			$this->apiBaseUrl . $endpoint,
			array(
				'method'    => $method,
				'sslverify' => false,
				'headers'   => $this->headers,
				'body'      => $body,
			)
		);

		$this->lastResponseCode = wp_remote_retrieve_response_code( $this->lastResponse );
		$responseBody           = is_wp_error( $this->lastResponse )
			? $this->lastResponse->get_error_message()
			: wp_remote_retrieve_body( $this->lastResponse );

		if ( ! $this->isValidResponse( $this->lastResponse ) ) {
			throw new ApiException(
				$this->lastResponse,
				sprintf(
				/* Translators: %d HTTP response code; %s API response */
					__( 'Invalid API response. HTTP Code: %d; Response Body: %s', 'edd-git-download-updater' ),
					$this->lastResponseCode,
					$responseBody
				),
				$this->lastResponseCode ? : 500
			);
		}

		return json_decode( $responseBody );
	}

	/**
	 * Determines if we have a valid response code.
	 *
	 * @param array|\WP_Error $response
	 *
	 * @return bool
	 */
	protected function isValidResponse( $response ) {
		if ( is_wp_error( $response ) ) {
			return false;
		}

		$responseCode = wp_remote_retrieve_response_code( $response );
		if ( ! is_integer( $responseCode ) ) {
			return false;
		}

		return $responseCode >= 200 && $responseCode < 300;
	}

}
