<?php
/**
 * Plugin Name: Easy Digital Downloads - Git Download Updater
 * Plugin URI: https://easydigitaldownloads.com/downloads/git-download-updater
 * Description: Update download files and changelogs directly from Bitbucket or GitHub.
 * Version: 1.3.2
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * Text Domain: edd-git-download-updater
 * Domain Path: /languages
 * Author: Easy Digital Downloads
 * Author URI: https://easydigitaldownloads.com
 */

if ( ! defined( 'EDD_GIT_PLUGIN_DIR' ) ) {
	define( 'EDD_GIT_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
}

if ( ! defined( 'EDD_GIT_PLUGIN_URL' ) ) {
	define( 'EDD_GIT_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}

if ( ! defined( 'EDD_GIT_VERSION' ) ) {
	define( 'EDD_GIT_VERSION', '1.3.2' );
}

class EDD_GIT_Download_Updater {

	/**
	 * @var EDD_GIT_Download_Updater|null
	 */
	private static $instance = null;

	/*
	 * Store our git Username. This will be used to login to the desired repo.
	 */
	public $username;

	/*
	 * Store our git Password. This wil be used to login to the desired repo.
	 */
	public $password;

	/**
	 * @var EDD_GIT_Download_Updater_Ajax
	 */
	public $ajax;

	/**
	 * @var EDD_GIT_Download_Updater_Admin
	 */
	public $admin;

	/**
	 * @var EDD_GIT_Download_Updater_Process_File
	 */
	public $process_file;

	/**
	 * @var EDD_GIT_Download_Updater_Repos
	 */
	public $repos;

	/**
	 * @var \EDD\GitDownloadUpdater\ProviderRegistry
	 */
	public $providerRegistry;

	/**
	 * Get things up and running.
	 *
	 * @since 1.0
	 * @return void
	 */
	public function __construct() {
	}

	/**
	 * Returns the single instance of this class.
	 *
	 * @since 1.3
	 *
	 * @return EDD_GIT_Download_Updater
	 */
	public static function instance() {
		if ( self::$instance instanceof EDD_GIT_Download_Updater ) {
			return self::$instance;
		}

		self::$instance = new self();
		self::$instance->boot();

		return self::$instance;
	}

	/**
	 * Boots the plugin.
	 *
	 * @since 1.3
	 *
	 * @return void
	 */
	private function boot() {
		global $post;

		// Load the plugin textdomain.
		load_plugin_textdomain( 'edd-git-download-updater', false, EDD_GIT_PLUGIN_DIR . 'languages/' );

		// Bail if the zip extension hasn't been loaded.
		if ( ! class_exists( 'ZipArchive' ) ) {
			$this->register_zip_archive_notice();
			return;
		}

		$this->registerProviders();

		// Include our ajax class.
		require_once EDD_GIT_PLUGIN_DIR . 'classes/ajax.php';
		$this->ajax = new EDD_GIT_Download_Updater_Ajax( $this );

		// Include our admin class.
		require_once EDD_GIT_PLUGIN_DIR . 'classes/admin.php';
		$this->admin = new EDD_GIT_Download_Updater_Admin( $this );

		// Include our file processing class.
		require_once EDD_GIT_PLUGIN_DIR . 'classes/process-file.php';
		$this->process_file = new EDD_GIT_Download_Updater_Process_File( $this );

		// Include our repo interaction class.
		require_once EDD_GIT_PLUGIN_DIR . 'classes/repos.php';
		$this->repos = new EDD_GIT_Download_Updater_Repos();

		add_action(
			'edd_extension_license_init',
			function ( \EDD\Extensions\ExtensionRegistry $registry ) {
				$registry->addExtension( __FILE__, 'Git Download Updater', 602998, EDD_GIT_VERSION );
			}
		);
	}

	/**
	 * Backwards compatibility for handling properties that have been moved from
	 * this class to the `EDD_GIT_Download_Updater_Process_File` class.
	 *
	 * @link https://github.com/awesomemotive/edd-git-download-updater/issues/129
	 *
	 * @since 1.3
	 *
	 * @param string $property
	 *
	 * @return mixed
	 */
	public function __get( $property ) {
		if ( property_exists( $this, $property ) ) {
			return $this->{$property};
		} elseif ( property_exists( $this->process_file, $property ) ) {
			return $this->process_file->{$property};
		}

		return null;
	}

	/**
	 * Backwards compatibility for handling properties that have been moved from
	 * this class to the `EDD_GIT_Download_Updater_Process_File` class.
	 *
	 * @link https://github.com/awesomemotive/edd-git-download-updater/issues/129
	 *
	 * @since 1.3
	 *
	 * @param string $property
	 *
	 * @return bool
	 */
	public function __isset( $property ) {
		if ( property_exists( $this, $property ) ) {
			return isset( $this->{$property} );
		} elseif ( property_exists( $this->process_file, $property ) ) {
			return isset( $this->process_file->{$property} );
		}

		return false;
	}

	/**
	 * Backwards compatibility for handling properties that have been moved from
	 * this class to the `EDD_GIT_Download_Updater_Process_File` class.
	 *
	 * @link https://github.com/awesomemotive/edd-git-download-updater/issues/129
	 *
	 * @since 1.3
	 *
	 * @param string $property
	 * @param mixed  $value
	 *
	 * @return void
	 */
	public function __set( $property, $value ) {
		if ( property_exists( $this, $property ) ) {
			$this->{$property} = $value;
		} elseif ( property_exists( $this->process_file, $property ) ) {
			$this->process_file->{$property} = $value;
		}
	}

	/**
	 * Registers our providers.
	 *
	 * @since 1.3
	 */
	private function registerProviders() { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.MethodNameInvalid
		$this->providerRegistry = new \EDD\GitDownloadUpdater\ProviderRegistry(); // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		$this->providerRegistry->addProvider( '\\EDD\\GitDownloadUpdater\\Providers\\GitHubProvider' ); // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		$this->providerRegistry->addProvider( '\\EDD\\GitDownloadUpdater\\Providers\\BitBucketProvider' ); // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
	}

	/**
	 * Register ZipArchive notice
	 *
	 * @since  1.0.3
	 * @return void
	 */
	private function register_zip_archive_notice() {
		add_action( 'admin_notices', array( $this, 'display_zip_archive_notice' ) );
	}

	/**
	 * Display an admin notice if the ZipArchive class is not available
	 *
	 * @since  1.0.3
	 * @return void
	 */
	public function display_zip_archive_notice() {
		?>
		<div class="notice notice-error">
			<p><?php esc_html_e( 'The ZipArchive PHP class is required to use the Easy Digital Downloads - Git Update Downloads extension.', 'edd-git-download-updater' ); ?></p>
		</div>
		<?php
	}
} // End EDD_GIT_Download_Updater class

/**
 * Boots the plugin.
 *
 * @return EDD_GIT_Download_Updater|null
 */
function edd_git_download_updater() {
	return EDD_GIT_Download_Updater::instance();
}

require_once __DIR__ . '/vendor/autoload.php';
\EDD\ExtensionUtils\v1\ExtensionLoader::loadOrQuit(
	__FILE__,
	'edd_git_download_updater',
	array(
		'php'                    => '7.4',
		'easy-digital-downloads' => '3.3.6',
		'wp'                     => '6.0',
	)
);

register_activation_hook(
	__FILE__,
	function () {
		if ( ! get_option( 'edd_git_download_updater_version' ) ) {
			update_option( 'edd_git_download_updater_version', EDD_GIT_VERSION );
		}
	}
);
