<?php
/**
 * Process our fetched file
 *
 * @package EDD Git Download Updater
 * @since  1.0
 */

use EDD\GitDownloadUpdater\Providers\Provider;

class EDD_GIT_Download_Updater_Process_File {

	/**
	 * The current instance of this class.
	 *
	 * @var EDD_GIT_Download_Updater
	 */
	private $instance;

	/**
	 * The download ID
	 *
	 * @var int
	 */
	public $download_id;

	/**
	 * The tag (version number) to process
	 *
	 * @var string
	 */
	public $version;

	/**
	 * The repository URL
	 *
	 * @var string
	 */
	public $url;

	/**
	 * The repository URL
	 *
	 * @var string
	 */
	public $repo_url;

	/**
	 * The file key for the EDD Files array
	 *
	 * @var int
	 */
	public $file_key;

	/**
	 * The folder name to be used when the archive is unpacked
	 *
	 * @var string
	 */
	public $original_foldername;

	/**
	 * The file name that should be used when downloading the archive
	 *
	 * @var string
	 */
	public $original_filename;

	/**
	 * The temporary directory where we are storing things while we make the proper folder and archive names.
	 *
	 * @var string
	 */
	public $tmp_dir;

	/**
	 * The final file name we're going to be using when making the new archive
	 *
	 * @var string
	 */
	public $file_name;

	/**
	 * The changelog for the product, read from the readme.txt from the downloaded tag
	 *
	 * @var string
	 */
	public $changelog = '';

	/**
	 * The file condition
	 *
	 * @var string
	 */
	public $condition;

	/**
	 * The URL and Path of the EDD file directory for uploaded files
	 *
	 * @var array
	 */
	public $edd_dir;

	/**
	 * The version we're going to assign to the Software Licensing version field
	 *
	 * @var string
	 */
	public $sl_version;

	/**
	 * A holder for any errors that occur
	 *
	 * @var array
	 */
	public $errors;

	/**
	 * The subdirectory of our unpacked original archive
	 *
	 * @var string
	 */
	public $sub_dir;

	/**
	 * The folder name that will be used when the final archive is unpacked
	 *
	 * @var string
	 */
	public $folder_name;

	/**
	 * Name of the repository.
	 *
	 * @var string
	 */
	private $repo_name;

	/**
	 * Path to the repository: {owner}/{repo_name}
	 *
	 * @var string
	 */
	private $repo_path;

	/**
	 * The git provider (object) to use.
	 *
	 * @var Provider
	 */
	private $provider;

	/**
	 * The git provider to use (either bitbucket or github)
	 *
	 * @var string
	 */
	public $source;

	/**
	 * EDD_GIT_Download_Updater_Process_File constructor.
	 *
	 * @param EDD_GIT_Download_Updater $instance The instance of this class.
	 */
	public function __construct( $instance ) {
		$this->instance = $instance;
	}

	/**
	 * Process our file
	 *
	 * @since  1.0
	 *
	 * @param int      $post_id     The download ID to work with.
	 * @param string   $version     The tag of the repo to download.
	 * @param string   $repo_url    The URL to the repository.
	 * @param int      $key         The file key for the EDD files array.
	 * @param string   $folder_name The folder name that should be used when the final archive is unpacked.
	 * @param string   $file_name   The file name for the archive when downloaded.
	 * @param string   $repo_owner  Owner of the repo.
	 * @param string   $repo_name   Name of the repository.
	 * @param Provider $provider    The git provider to fetch the file from.
	 *
	 * @return array
	 * @throws Exception When anything but a 200 request is received while processing the file.
	 */
	public function process( $post_id, $version, $repo_url, $key, $folder_name, $file_name, $repo_owner, $repo_name, Provider $provider ) {
		// OK, we're authenticated: we need to find and save the data.
		$this->download_id         = $post_id;
		$this->version             = $version;
		$this->repo_url            = $repo_url;
		$this->file_key            = $key;
		$this->original_filename   = $file_name;
		$this->original_foldername = $folder_name;
		$this->repo_name           = $repo_name;
		$this->repo_path           = $repo_owner . '/' . $repo_name;
		$this->provider            = $provider;

		// Include our zip archive utility.
		require_once EDD_GIT_PLUGIN_DIR . 'includes/flx-zip-archive.php';

		/*
		 * If `true`, then a custom asset has selected and we will NOT proceed with
		 * folder renaming, etc. If `false`, then the user is downloading the repo
		 * source code and we WILL proceed with renaming.
		 */
		$using_custom_asset = ! empty( $this->url );

		// Setup our initial variables.
		$this->set_url();
		$this->set_foldername( $folder_name );
		$this->set_tmp_dir();
		$this->set_edd_dir();
		$this->set_filename( $file_name );

		// Grab our zip file.
		$zip_path = $this->fetch_zip( true );

		if ( ! $zip_path ) {
			// If we bailed during the fetch_zip function, stop processing update.
			throw new \Exception( 'Failed to process zip file.', 500 );
		}

		if ( ! $using_custom_asset ) {
			// Unzip our file to a new temporary directory.
			$new_dir = $this->unzip( $zip_path );

			// Create our new zip file.
			$zip = $this->zip( $new_dir, $this->tmp_dir . $this->file_name );
		} else {
			$new_dir = $this->tmp_dir . $this->folder_name;
			$zip     = $this->tmp_dir . $this->file_name;

			// Extract readme.txt from custom asset if setting is enabled.
			if ( edd_get_option( 'edd_sl_readme_parsing', false ) ) {
				$new_dir = $this->extract_readme_from_custom_asset( $zip_path );
			}
		}

		// Move our temporary zip to the proper EDD folder.
		$new_zip           = $this->move_zip( $zip );
		$new_zip['readme'] = $this->copy_readme_file( $new_dir );

		// Remove our temporary files.
		$this->remove_dir( $this->tmp_dir );

		// Reset our temporary directory.
		$this->set_tmp_dir();

		// Update our file with the new zip location.
		$this->update_files( $new_zip['url'] );

		return $new_zip;
	}

	/**
	 * Update the download's changelog from our grabbed readme.txt
	 *
	 * @since 1.0
	 *
	 * @param string $new_dir The directory to look in to grab the readme.txt file for.
	 * @return void
	 */
	public function update_changelog( $new_dir ) {
		$path = trailingslashit( $new_dir ) . 'readme.txt';
		if ( ! file_exists( $path ) ) {
			return;
		}
		if ( ! defined( 'EDD_SL_PLUGIN_DIR' ) ) {
			return;
		}
		$file_contents = file_get_contents( $path );
		if ( ! $file_contents ) {
			return;
		}

		include_once EDD_SL_PLUGIN_DIR . 'includes/classes/class-sl-parser.php';
		if ( ! class_exists( 'EDD_SL_Readme_Parser' ) ) {
			return;
		}
		$parser  = new EDD_SL_Readme_Parser( $file_contents );
		$content = $parser->parse_data();
		if ( isset( $content['sections']['changelog'] ) ) {
			$changelog = wp_kses_post( $content['sections']['changelog'] );
			update_post_meta( $this->download_id, '_edd_sl_changelog', $changelog );
		}
	}

	/**
	 * Retrieves the changelog text from the readme.txt file.
	 *
	 * @since 1.2
	 * @param string $readme_url The URL for the readme.txt file.
	 * @return void
	 */
	private function set_changelog_text( $readme_url ) {
		if ( ! $readme_url ) {
			return;
		}

		$content = _edd_sl_readme_parse( $readme_url );
		if ( isset( $content['sections']['changelog'] ) ) {
			$this->changelog = wp_kses_post( $content['sections']['changelog'] );
		}
	}

	/**
	 * If Software Licensing is active, copies the repository readme file to the
	 * EDD directory.
	 *
	 * @since 1.2
	 * @param string $new_dir
	 * @return string|boolean
	 */
	private function copy_readme_file( $new_dir ) {
		if ( ! function_exists( '_edd_sl_readme_parse' ) ) {
			return false;
		}

		if ( ! edd_get_option( 'edd_sl_readme_parsing', false ) ) {
			return false;
		}

		$url = trailingslashit( $new_dir ) . 'readme.txt';
		if ( ! file_exists( $url ) ) {
			return false;
		}

		// Get upload path and URL for readme file.
		remove_filter( 'upload_dir', 'edd_set_upload_dir' );
		$upload_dir = wp_upload_dir();
		add_filter( 'upload_dir', 'edd_set_upload_dir' );
		$upload_path = trailingslashit( $upload_dir['path'] );
		$upload_url  = trailingslashit( $upload_dir['url'] );
		$readme_path = "{$upload_path}{$this->folder_name}-readme.{$this->version}.txt";
		$readme_url  = "{$upload_url}{$this->folder_name}-readme.{$this->version}.txt";

		// Copy readme file to EDD upload directory.
		$readme = copy( $url, $readme_path );
		if ( ! $readme ) {
			return false;
		}
		$this->set_changelog_text( $readme_url );

		$edd_settings = edd_get_settings();
		if ( empty( $edd_settings['edd_sl_readme_parsing'] ) ) {
			unlink( $readme_path );
			return false;
		}

		return $readme_url;
	}

	/**
	 * Update our download files post meta
	 *
	 * @since 1.0
	 *
	 * @param string $new_zip The new archive file for the EDD Files array.
	 *
	 * @return void
	 */
	public function update_files( $new_zip ) {
		$files                                       = array(
			$this->file_key => array(
				'git_version'     => $this->version,
				'git_url'         => $this->repo_url,
				'git_folder_name' => $this->original_foldername,
				'git_file_asset'  => $this->url,
				'file'            => $new_zip,
				'name'            => ! empty( $this->file_name ) ? $this->file_name : basename( $new_zip ),
				'condition'       => $this->condition,
				'attachment_id'   => 0,
			),
		);

		if ( 0 === strpos( $this->version, 'v' ) ) {
			$this->sl_version = substr( $this->version, 1 );
		} else {
			$this->sl_version = $this->version;
		}
	}

	/**
	 * Move our zip file to the EDD uploads directory
	 *
	 * @since 1.0
	 *
	 * @param string $zip The zip file to move.
	 *
	 * @return array $new_zip
	 */
	public function move_zip( $zip ) {
		// Get upload path and URL for zip file.
		$edd_dir     = trailingslashit( $this->edd_dir['path'] );
		$edd_url     = trailingslashit( $this->edd_dir['url'] );
		$upload_path = apply_filters( 'edd_git_upload_path', $edd_dir . $this->file_name );
		$upload_url  = apply_filters( 'edd_git_upload_url', $edd_url . $this->file_name );

		// Copy zip file to EDD upload directory.
		copy( $zip, $upload_path );

		// Delete temporary zip file.
		unlink( $zip );

		// Prepare zip file path and URL.
		$new_zip = array(
			'path' => $upload_path,
			'url'  => $upload_url,
		);

		/**
		 * Fires after zip file has been moved to EDD uploads directory.
		 *
		 * @since Unknown
		 *
		 * @param array  $new_zip {
		 *    Path and URL to zip file.
		 *
		 *    @param string $path Path to zip file.
		 *    @param string $url  URL to zip file.
		 * }
		 * @param string $repo_name Name of git repo.
		 */
		do_action( 'edd_git_zip_saved', $new_zip, $this->repo_name );

		return $new_zip;
	}

	/**
	 * Set our temporary directory. Create it if it doesn't exist.
	 *
	 * @since 1.0
	 * @return void
	 */
	public function set_tmp_dir() {
		$tmp_dir = wp_upload_dir();
		$tmp_dir = trailingslashit( $tmp_dir['basedir'] ) . 'edd-git-tmp/';
		$tmp_dir = apply_filters( 'edd_git_zip_path', $tmp_dir );

		if ( ! is_dir( $tmp_dir ) ) {
			mkdir( $tmp_dir );
		}

		// $tmp_dir will always have a trailing slash.
		$this->tmp_dir = trailingslashit( $tmp_dir );
	}

	/**
	 * Set our git URL. Also sets whether we are working from GitHub or Bitbucket.
	 *
	 * @since 1.0
	 */
	public function set_url() {
		if ( ! empty( $this->url ) ) {
			$url = $this->url;
		} else {
			$url = $this->provider->buildAssetUrlFromRepoAndTag( $this->repo_path, $this->version );
		}

		/**
		 * Filters the URL to the zip file.
		 *
		 * @param string $url
		 */
		$this->url = apply_filters( 'edd_git_repo_url', $url );
	}

	/**
	 * Set our clean zip file name
	 *
	 * @since 1.0
	 *
	 * @param string $file_name The file name to set.
	 *
	 * @return void
	 */
	public function set_filename( $file_name ) {
		$this->file_name = ! empty( $file_name ) ? $file_name : $this->repo_name . '-' . $this->version . '.zip';

		$this->file_name = apply_filters( 'edd_git_download_file_name', $this->file_name, $this->download_id, $this->file_key );
	}

	/**
	 * Set the name of our folder that should go inside our new zip.
	 *
	 * @since 1.0
	 *
	 * @param string $folder_name The folder name to set.
	 *
	 * @return void
	 */
	public function set_foldername( $folder_name ) {
		$this->folder_name = ! empty( $folder_name ) ? $folder_name : sanitize_title( $this->repo_name );
	}

	/**
	 * Grab the zip file from git and store it in our temporary directory.
	 *
	 * @since 1.0
	 * @since 1.3 Added `$allowExceptions` parameter. Defaults to `false` for backwards compat.
	 *
	 * @param bool $allowExceptions Whether to allow exceptions to be thrown instead of a `false` response.
	 *
	 * @return string|false Path to downloaded zip on success, false on failure.
	 * @throws \Exception
	 */
	public function fetch_zip( $allowExceptions = false ) {
		$zip_path = $this->tmp_dir . $this->file_name;

		try {
			$contents = $this->provider->fetchZipFromUrl( $this->url );

			$fp = fopen( $zip_path, 'w' );
			fwrite( $fp, $contents );

			do_action( 'edd_git_zip_fetched', $zip_path, $this->repo_name );

			return $zip_path;
		} catch ( \EDD\GitDownloadUpdater\Exceptions\ApiException $e ) {
			// Allow individual providers to parse more specific error messages from the response if supported.
			if ( method_exists( $this->provider, 'parseApiErrors' ) ) {
				$this->errors[ $this->file_key ] = array(
					'error' => $e->getCode(),
					'msg'   => call_user_func( array( $this->provider, 'parseApiErrors' ), $e ),
				);
			} else {
				// This will be a generic error message containing the raw API response.
				$this->errors[ $this->file_key ] = array(
					'error' => $e->getCode(),
					'msg'   => $e->getMessage(),
				);
			}

			return $this->returnFalseOrException( $allowExceptions, $e );
		} catch ( \Exception $e ) {
			$this->errors[ $this->file_key ] = array(
				'error' => $e->getCode(),
				'msg'   => $e->getMessage(),
			);

			return $this->returnFalseOrException( $allowExceptions, $e );
		}
	}

	/**
	 * Throws the exception if supported, otherwise returns `false`.
	 *
	 * @since 1.3
	 *
	 * @param bool       $allowExceptions
	 * @param \Exception $e
	 *
	 * @return false
	 * @throws \Exception
	 */
	private function returnFalseOrException( $allowExceptions, $e ) {
		if ( $allowExceptions ) {
			throw $e;
		} else {
			return false;
		}
	}

	/**
	 * Unzip our file into a new temporary folder.
	 *
	 * @since 1.0
	 *
	 * @param string $zip_path The file path.
	 *
	 * @return string $new_dir
	 */
	public function unzip( $zip_path ) {
		if ( is_dir( trailingslashit( $this->tmp_dir . $this->folder_name ) ) ) {
			$this->remove_dir( trailingslashit( $this->tmp_dir . $this->folder_name ) );
		}
		$zip = new ZipArchive();
		$zip->open( $zip_path );
		$zip->extractTo( $this->tmp_dir );
		$zip->close();
		$this->set_sub_dir( $this->tmp_dir );

		$new_dir = rename( $this->tmp_dir . $this->sub_dir, $this->tmp_dir . $this->folder_name );

		if ( ! $new_dir ) {
			return false;
		}

		$new_dir = $this->tmp_dir . $this->folder_name;
		$this->set_sub_dir( $this->tmp_dir );
		unlink( $this->tmp_dir . $this->file_name );

		/**
		 * Allow developers to add an action once the .zip file has been extracted.
		 *
		 * @since 1.3
		 * @param string $new_dir   The new directory.
		 * @param string $repo_name The name of the Git repository.
		 */
		do_action( 'edd_git_zip_extracted', $new_dir, $this->repo_name );

		return $new_dir;
	}

	/**
	 * Zip our directory and return the path to the zip file.
	 *
	 * @since 1.0
	 *
	 * @param string $dir The directory to compress.
	 * @param string $destination The destination for the compressed file.
	 *
	 * @return string $destination
	 */
	public function zip( $dir, $destination ) {

		// Don't forget to remove the trailing slash.

		$the_folder    = $dir;
		$zip_file_name = $destination;

		$za = new FlxZipArchive();

		$res = $za->open( $zip_file_name, ZipArchive::CREATE );

		if ( true === $res ) {
			$za->addDir( $the_folder, basename( $the_folder ) );
			$za->close();
		} else {
			echo 'Could not create a zip archive';
		}

		return $destination;
	}

	/**
	 * Delete tmp directory and all contents
	 *
	 * @since 1.0
	 *
	 * @param string $dir The directory to remove.
	 *
	 * @return void
	 */
	public function remove_dir( $dir ) {
		foreach ( scandir( $dir ) as $file ) {
			if ( '.' === $file || '..' === $file ) {
				continue;
			}

			$dir = trailingslashit( $dir );
			if ( is_dir( $dir . $file ) ) {
				$this->remove_dir( $dir . $file );
			} else {
				unlink( $dir . $file );
			}
		}
		rmdir( $dir );
	}

	/**
	 * Get our newly unzipped subdirectory name.
	 *
	 * @since 1.0
	 *
	 * @param string $tmp_dir The temp dir.
	 *
	 * @return void|array
	 */
	public function set_sub_dir( $tmp_dir ) {
		$dir_array = array();
		// Bail if we weren't sent a directory.
		if ( ! is_dir( $tmp_dir ) ) {
			return $dir_array;
		}

		$dh = opendir( $tmp_dir );
		if ( $dh ) {
			while ( ( $file = readdir( $dh ) ) !== false ) {
				if ( '.' === $file || '..' === $file ) {
					continue;
				}

				if ( false !== strpos( $file, $this->repo_name ) ) {
					if ( is_dir( $tmp_dir . '/' . $file ) ) {
						$this->sub_dir = $file;
						break;
					}
				}
			}
			closedir( $dh );
		}
	}

	/**
	 * Set our EDD uploads directory.
	 *
	 * @since 1.0
	 * @return void
	 */
	public function set_edd_dir() {
		add_filter( 'upload_dir', 'edd_set_upload_dir' );
		$upload_dir = wp_upload_dir();
		wp_mkdir_p( $upload_dir['path'] );
		$this->edd_dir = $upload_dir;
	}

	/**
	 * Extract readme.txt from a custom asset zip file without full unzip/rezip process.
	 *
	 * @since 1.3.2
	 * @param string $zip_path Path to the custom asset zip file.
	 * @return string|false Path to temporary directory containing readme.txt, or false on failure.
	 */
	private function extract_readme_from_custom_asset( $zip_path ) {
		if ( ! edd_get_option( 'edd_sl_readme_parsing', false ) ) {
			return false;
		}

		$temp_extract_dir = $this->tmp_dir . 'readme_extract/';

		// Create temporary extraction directory.
		if ( ! is_dir( $temp_extract_dir ) ) {
			mkdir( $temp_extract_dir, 0755, true );
		}

		$zip = new ZipArchive();
		if ( true !== $zip->open( $zip_path ) ) {
			return false;
		}

		// Look for readme.txt in the zip archive.
		$readme_found = false;
		for ( $i = 0; $i < $zip->numFiles; $i++ ) {
			$file_info = $zip->statIndex( $i );
			$filename  = basename( $file_info['name'] );

			// Check if this is a readme.txt file (case-insensitive).
			if ( strtolower( $filename ) === 'readme.txt' ) {
				// Extract only this file.
				$zip->extractTo( $temp_extract_dir, $file_info['name'] );
				$readme_found = true;

				// If the readme is in a subdirectory, move it to the root of our temp directory.
				$extracted_path = $temp_extract_dir . $file_info['name'];
				$target_path    = $temp_extract_dir . 'readme.txt';

				if ( $extracted_path !== $target_path && file_exists( $extracted_path ) ) {
					// Create any necessary parent directories.
					$target_dir = dirname( $target_path );
					if ( ! is_dir( $target_dir ) ) {
						mkdir( $target_dir, 0755, true );
					}

					// Move the file to the expected location.
					rename( $extracted_path, $target_path );

					// Clean up any empty subdirectories created during extraction.
					$this->cleanup_empty_directories( $temp_extract_dir, dirname( $file_info['name'] ) );
				}

				break;
			}
		}

		$zip->close();

		return $readme_found ? $temp_extract_dir : false;
	}

	/**
	 * Recursively remove empty directories.
	 *
	 * @since 1.3.2
	 * @param string $base_dir Base directory to work from.
	 * @param string $relative_path Relative path from base directory to clean up.
	 * @return void
	 */
	private function cleanup_empty_directories( $base_dir, $relative_path ) {
		if ( empty( $relative_path ) || '.' === $relative_path ) {
			return;
		}

		$full_path = $base_dir . $relative_path;

		// Only remove if directory is empty.
		if ( is_dir( $full_path ) && count( scandir( $full_path ) ) === 2 ) { // Only . and ..
			rmdir( $full_path );

			// Recursively clean up parent directories.
			$this->cleanup_empty_directories( $base_dir, dirname( $relative_path ) );
		}
	}
}
