<?php
/**
 * Handle Ajax requests from EDD Git Downloader
 *
 * @package EDD Git Downloader
 * @since 1.0
 */
class EDD_GIT_Download_Updater_Ajax {

	/**
	 * Holds our instance
	 *
	 * @var EDD_GIT_Download_Updater
	 */
	private $instance = '';

	/**
	 * EDD_GIT_Download_Updater_Ajax constructor.
	 *
	 * @param EDD_GIT_Download_Updater $instance The instance.
	 */
	public function __construct( $instance ) {

		$this->instance = $instance;

		// Add our ajax action for updating our file download section.
		add_action( 'wp_ajax_edd_change_use_git', array( $this, 'ajax_use_git' ) );

		// Add our ajax action for grabbing repos.
		add_action( 'wp_ajax_edd_git_fetch_repos', array( $this, 'ajax_repos' ) );

		// Add our ajax action for grabbing repo tags.
		add_action( 'wp_ajax_edd_git_get_tags', array( $this, 'ajax_tags' ) );

		add_action( 'wp_ajax_edd_git_load_tag_assets', array( $this, 'ajax_load_tag_assets' ) );

		// Add our ajax action for updating our file.
		add_action( 'wp_ajax_edd_git_update_file', array( $this, 'ajax_fetch_file' ) );

		// Add our ajax action for requesting an initial GitHub token.
		add_action( 'wp_ajax_edd_git_gh_request_token', array( $this, 'gh_request_token' ) );

		// Add our ajax action for getting a permanent GitHub access token.
		add_action( 'wp_ajax_edd_git_gh_set_oauth_key', array( $this, 'gh_set_oauth_key' ) );

		// Add our ajax action for disconnecting from GitHub.
		add_action( 'wp_ajax_edd_git_gh_disconnect', array( $this, 'gh_disconnect' ) );

	}

	/**
	 * Run all of the functions necessary to update our download.
	 *
	 * @since 1.0
	 *
	 * @return void
	 */
	public function ajax_fetch_file() {
		$post_id    = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;
		$version    = isset( $_POST['version'] ) ? sanitize_text_field( wp_unslash( $_POST['version'] ) ) : '';
		$repo_url   = isset( $_POST['repo_url'] ) ? esc_url_raw( wp_unslash( $_POST['repo_url'] ) ) : '';
		$key        = isset( $_POST['key'] ) ? absint( $_POST['key'] ) : 0;
		$provider   = isset( $_POST['provider'] ) ? wp_strip_all_tags( $_POST['provider'] ) : null;
		$repo_owner = isset( $_POST['repo_owner'] ) ? sanitize_text_field( wp_unslash( $_POST['repo_owner'] ) ) : '';
		$repo_name  = isset( $_POST['repo_name'] ) ? sanitize_text_field( wp_unslash( $_POST['repo_name'] ) ) : '';
		$asset_url  = ! empty( $_POST['asset_url'] ) ? esc_url_raw( wp_unslash( $_POST['asset_url'] ) ) : null;

		$post_id = wp_unslash( $post_id );

		$this->instance->process_file->condition = isset( $_POST['condition'] ) ? sanitize_text_field( wp_unslash( $_POST['condition'] ) ) : 'all';

		$folder_name = isset( $_POST['folder_name'] ) ? sanitize_file_name( wp_unslash( $_POST['folder_name'] ) ) : '';
		$file_name   = isset( $_POST['file_name'] ) ? sanitize_file_name( wp_unslash( $_POST['file_name'] ) ) : '';

		if ( ! current_user_can( 'edit_product', $post_id ) || empty( $post_id ) || empty( $version ) || empty( $repo_url ) || empty( $repo_owner ) || empty( $repo_name ) ) {
			wp_send_json(
				array(
					'post_id'  => $post_id,
					'errors'   => __( 'Unable to fetch the file.', 'edd-git-download-updater' ),
					'can_edit' => current_user_can( 'edit_product', $post_id ),
					'version'  => $version,
					'url'      => $repo_url,
				)
			);
		}

		if ( empty( $provider ) ) {
			wp_send_json( array(
				'errors' => __( 'Missing Git provider.', 'edd-git-download-updater' )
			) );
		}

		try {
			$provider = edd_git_download_updater()->providerRegistry->getProvider( $provider );
		} catch ( \EDD\GitDownloadUpdater\Exceptions\ResourceNotFoundException $e ) {
			wp_send_json( array(
				'errors' => __( 'Invalid Git provider.', 'edd-git-download-updater' )
			) );
		}

		if ( ! empty( $asset_url ) ) {
			$this->instance->process_file->url = $asset_url;
		}

		try {
			$new_zip = $this->instance->process_file->process(
				$post_id,
				$version,
				$repo_url,
				$key,
				$folder_name,
				$file_name,
				$repo_owner,
				$repo_name,
				$provider
			);

			$file_name = empty( $file_name ) ? basename( $new_zip['path'] ) : $file_name;

			// Return our changelog and version.
			wp_send_json(
				array(
					'file'       => $new_zip['url'],
					'sl_version' => $this->instance->process_file->sl_version,
					'changelog'  => $this->instance->process_file->changelog,
					'errors'     => $this->instance->process_file->errors,
					'name'       => $file_name,
					'readme'     => $new_zip['readme'],
				)
			);
		} catch ( \Exception $e ) {
			wp_send_json( array(
				'errors' => $this->instance->process_file->errors,
			) );
		}
	}

	/**
	 * Output our repo options
	 *
	 * @since  1.0
	 *
	 * @return void
	 */
	public function ajax_repos() {
		if ( ! current_user_can( 'edit_products' ) ) {
			die();
		}

		if ( filter_input( INPUT_POST, 'clear_repositories', FILTER_VALIDATE_BOOLEAN ) ) {
			delete_transient( 'edd_git_repos' );
		}
		$current_repo = isset( $_POST['current_repo'] ) ? sanitize_text_field( wp_unslash( $_POST['current_repo'] ) ) : '';
		$repos        = $this->instance->repos->fetch_repos();

		ob_start();
		$this->instance->admin->output_repo_options( $repos, $current_repo );
		$options = ob_get_clean();

		wp_send_json( array( 'options_html' => $options ) );
	}

	/**
	 * Output our tag options
	 *
	 * @since  1.0
	 *
	 * @return void
	 */
	public function ajax_tags() {
		if ( ! current_user_can( 'edit_products' ) ) {
			die();
		}

		$repo        = isset( $_REQUEST['repo'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['repo'] ) ) : '';
		$return_tags = $this->instance->repos->fetch_tags( $repo );

		wp_send_json( $return_tags );
	}

	/**
	 * Loads the assets for the chosen tag.
	 *
	 * @since 1.3
	 *
	 * @return void
	 */
	public function ajax_load_tag_assets() {
		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'edd-git-download-updater' ), 403 );
		}

		try {
			$provider = ! empty( $_POST['git_provider'] ) ? wp_strip_all_tags( wp_unslash( $_POST['git_provider'] ) ) : null;
			if ( empty( $provider ) ) {
				throw new \Exception( __( 'Missing Git provider.', 'edd-git-download-updater' ), 400 );
			}

			$repoPath = ! empty( $_POST['repo_path'] ) ? wp_strip_all_tags( wp_unslash( $_POST['repo_path'] ) ) : null;
			$tag      = ! empty( $_POST['tag'] ) ? wp_strip_all_tags( wp_unslash( $_POST['tag'] ) ) : null;
			if ( empty( $repoPath ) || empty( $tag ) ) {
				throw new \Exception( __( 'Missing repository path and/or tag.', 'edd-git-download-updater' ), 400 );
			}

			if ( ! edd_git_download_updater()->providerRegistry->providerSupports( $provider, EDD\GitDownloadUpdater\Providers\SelectTagAsset::class ) ) {
				throw new \Exception( __( 'Provider does not support customizing the tag asset.', 'edd-git-download-updater' ) );
			}

			$provider = edd_git_download_updater()->providerRegistry->getProvider( $provider );

			try {
				$assets = $provider->getAssetsFromRepoTag( $repoPath, $tag );
			} catch ( \EDD\GitDownloadUpdater\Exceptions\ResourceNotFoundException $e ) {
				$assets = array();
			}

			wp_send_json_success( $assets );
		} catch ( \Exception $e ) {
			wp_send_json_error( $e->getMessage(), 400 );
		}
	}

	/**
	 * Handle someone checking the "Use Git" checkbox by outputting our repo and tag html.
	 *
	 * @since  1.0
	 *
	 * @return void
	 */
	public function ajax_use_git() {
		if ( ! current_user_can( 'edit_products' ) || ( ! defined( 'DOING_AJAX' ) || ! DOING_AJAX ) || ! is_admin() ) {
			die();
		}

		$type    = empty( $_POST['product_type'] ) ? 0 : esc_attr( $_POST['product_type'] );
		$checked = isset( $_POST['checked'] ) && 'bundle' !== $type ? absint( $_POST['checked'] ) : 0;
		$post_id = isset( $_POST['post_id'] ) ? absint( $_POST['post_id'] ) : 0;

		if ( empty( $post_id ) || ! current_user_can( 'edit_product', $post_id ) ) {
			die();
		}

		// Update the product type.
		update_post_meta( $post_id, '_edd_product_type', $type );

		require_once EDD_PLUGIN_DIR . 'includes/admin/downloads/metabox.php';

		// If use git is enabled, update the post meta and load the correct section.
		if ( 1 === $checked ) {
			update_post_meta( $post_id, '_edd_download_use_git', $checked );
			$this->instance->admin->register_git_section();
		} else {
			// Otherwise, delete the git post meta and update the download files.
			delete_post_meta( $post_id, '_edd_download_use_git' );
			update_post_meta( $post_id, 'edd_download_files', array() );
		}

		ob_start();
		do_action( 'edd_meta_box_files_fields', $post_id );
		$html = ob_get_clean();

		wp_send_json( array( 'html' => $html ) );
	}

	/**
	 * Request our initial, temporary GitHub token.
	 *
	 * @since 1.1
	 * @return void
	 */
	public function gh_request_token() {
		if ( ! current_user_can( 'edit_products' ) ) {
			die();
		}

		$client_id     = isset( $_REQUEST['client_id'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['client_id'] ) ) : '';
		$client_secret = isset( $_REQUEST['client_secret'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['client_secret'] ) ) : '';

		// Bail if we didn't receive a client_id or client_secret.
		if ( '' == $client_id || '' == $client_secret ) {
			die();
		}

		// Save our client_id and client_secret.
		$edd_settings = get_option( 'edd_settings' );
		$edd_settings['gh_clientid'] = $client_id;
		$edd_settings['gh_clientsecret'] = $client_secret;
		update_option( 'edd_settings', $edd_settings );

		$redirect_uri = urlencode( admin_url( 'admin-ajax.php?action=edd_git_gh_set_oauth_key' ) );

		// Send user to GitHub for account authorization.

		$query = 'https://github.com/login/oauth/authorize';
		$query_args = array(
			'scope'        => 'repo',
			'client_id'    => $client_id,
			'redirect_uri' => $redirect_uri,
		);
		$query = add_query_arg( $query_args, $query );

		echo esc_url_raw( $query );

		die();
	}

	/**
	 * Finish up our GitHub oAuth. The user has just returned from GitHub.
	 *
	 * @since 1.1
	 * @return void
	 */
	public function gh_set_oauth_key() {
		if ( ! current_user_can( 'edit_products' ) ) {
			die();
		}

		// Get our client id and secret.
		$edd_options     = edd_get_settings();
		$gh_clientid     = isset( $edd_options['gh_clientid'] ) ? $edd_options['gh_clientid'] : '';
		$gh_clientsecret = isset( $edd_options['gh_clientsecret'] ) ? $edd_options['gh_clientsecret'] : '';
		$access_code     = ! empty( $_GET['code'] ) ? sanitize_text_field( $_GET['code'] ) : '';
		$redirect        = add_query_arg(
			array(
				'post_type' => 'download',
				'page'      => 'edd-settings',
				'tab'       => 'extensions',
				'section'   => 'edd-git',
			),
			admin_url( 'edit.php' )
		);

		if ( $access_code ) {
			$query    = add_query_arg(
				array(
					'client_id'     => $gh_clientid,
					'client_secret' => $gh_clientsecret,
					'code'          => $access_code,
				),
				'https://github.com/login/oauth/access_token'
			);
			$response = wp_safe_remote_get(
				$query,
				array(
					'sslverify' => true,
				)
			);
			parse_str( wp_remote_retrieve_body( $response ), $github_response );

			if ( ! empty( $github_response['access_token'] ) ) {
				$edd_options['gh_access_token'] = $github_response['access_token'];
				update_option( 'edd_settings', $edd_options );
			} else {
				$redirect = add_query_arg( array( 'authorize' => 'false' ), $redirect );
			}
		} else {
			$redirect = add_query_arg( array( 'authorize' => 'false' ), $redirect );
		}

		delete_transient( 'edd_git_repos' );

		wp_safe_redirect( $redirect );
		die();
	}

	/**
	 * Disconnect from GitHub.
	 * This will remove the token, but will NOT revoke access.
	 * To fully revoke access, visit your account at github.com.
	 *
	 * @since 1.1
	 * @return void
	 */
	public function gh_disconnect() {
		global $edd_options;

		if ( ! current_user_can( 'edit_products' ) ) {
			die();
		}

		unset( $edd_options['gh_access_token'] );
		delete_transient( 'edd_git_repos' );
		update_option( 'edd_settings', $edd_options );
		die();
	}


}
