<?php
/**
 * Outputs our admin page HTML
 *
 * @package EDD Git Downloader
 * @since  1.0
 */

class EDD_GIT_Download_Updater_Admin {
	/**
	 * @var EDD_GIT_Download_Updater
	 */
	public $instance = '';

	/**
	 * @var \EDD\GitDownloadUpdater\Admin\Upgrades
	 * @since 1.3
	 */
	protected $upgrades;

	/**
	 * EDD_GIT_Download_Updater_Admin constructor.
	 *
	 * @param EDD_GIT_Download_Updater $instance
	 */
	function __construct( $instance ) {

		$this->instance = $instance;
		$this->upgrades = new \EDD\GitDownloadUpdater\Admin\Upgrades();

		// Handle upgrades.
		add_action( 'admin_init', array( $this->upgrades, 'maybeUpgrade' ) );

		// Add our settings to the EDD Extensions tab
		add_filter( 'edd_settings_sections_extensions', array( $this, 'register_subsection' ) );
		add_filter( 'edd_settings_extensions', array( $this, 'edd_extensions_settings' ) );
		add_action( 'edd_meta_box_files_fields', array( $this, 'output_file_checkbox' ), 4, 2 );
		add_filter( 'edd_flyout_docs_link', array( $this, 'update_docs_link' ) );

		// Add our init action that adds/removes git download boxes.
		add_action( 'admin_head', array( $this, 'init' ) );

		// Save our Use Git setting.
		add_action( 'save_post', array( $this, 'save_post' ), 9999 );

		/* GitHub */

		// Add our GitHub description hook.
		add_action( 'edd_git_gh_desc', array( $this, 'gh_desc' ) );

		// Add our GitHub Authorization button hook.
		add_action( 'edd_git_gh_authorize_button', array( $this, 'gh_authorize_button' ) );

		// Add our JS to the New/Edit Download page.
		add_action( 'admin_print_scripts-post.php', array( $this, 'admin_js' ) );
		add_action( 'admin_print_scripts-post-new.php', array( $this, 'admin_js' ) );
		add_action( 'admin_print_scripts-download_page_edd-settings', array( $this, 'admin_js' ) );

		// Add our CSS to the New/Edit Download page.
		add_action( 'admin_print_styles-post.php', array( $this, 'admin_css' ) );
		add_action( 'admin_print_styles-post-new.php', array( $this, 'admin_css' ) );

		/* Bitbucket */

		// Add our Bitbucket description hook.
		add_action( 'edd_git_bb_desc', array( $this, 'bb_desc' ) );
	}

	/**
	 * Register the Git Updater subsection
	 *
	 * @since  1.0.3
	 * @param  array $sections The Existing subsections in the Extensions tab
	 * @return array           Sections with our subsection added
	 */
	public function register_subsection( $sections ) {

		// Note the array key here of 'ck-settings'
		$sections['edd-git'] = __( 'Git Updater', 'edd-git-download-updater' );

		return $sections;
	}

	/*
	 * Add our default git settings to the Extensions tab
	 *
	 * @since 1.0
	 * @return array $extensions
	 */

	public function edd_extensions_settings( $extensions ) {

		$git_settings = array(
			'gh_desc'             => array(
				'id'   => 'git_gh_desc',
				'name' => __( 'GitHub Status', 'edd-git-download-updater' ),
				'type' => 'hook',
			),
			'gh_clientid'         => array(
				'id'   => 'gh_clientid',
				'name' => __( 'Client ID', 'edd-git-download-updater' ),
				'type' => 'text',
			),
			'gh_clientsecret'     => array(
				'id'   => 'gh_clientsecret',
				'name' => __( 'Client Secret', 'edd-git-download-updater' ),
				'type' => 'password',
			),
			'gh_authorize_button' => array(
				'id'   => 'git_gh_authorize_button',
				'type' => 'hook',
			),
			'bb_desc'             => array(
				'id'   => 'git_bb_desc',
				'name' => __( 'Bitbucket Status', 'edd-git-download-updater' ),
				'type' => 'hook',
			),
			'bb_username'         => array(
				'id'   => 'bb_username',
				'name' => __( 'Bitbucket Username', 'edd-git-download-updater' ),
				'type' => 'text',
			),
			'bb_app_password'     => array(
				'id'   => 'bb_app_password',
				'name' => __( 'Bitbucket App Password', 'edd-git-download-updater' ),
				'type' => 'password',
			),
		);

		return array_merge( $extensions, array( 'edd-git' => $git_settings ) );
	}

	/**
	 * Output our "Use Git" checkbox.
	 * @since  1.0
	 * @param  integer $post_id
	 * @return void
	 */
	public function output_file_checkbox( $post_id = 0, $type = '' ) {
		$checked = get_post_meta( $post_id, '_edd_download_use_git', true );
		$classes = array( 'edd-form-group', 'edd-git-enable' );
		if ( ! empty( $type ) && 'default' !== $type ) {
			$classes[] = 'edd-hidden';
		}
		?>
		<div class="<?php echo esc_attr( implode( ' ', array_filter( $classes ) ) ); ?>" data-edd-supports-product-type="false">
			<div class="edd-form-group__control edd-toggle">
				<input type="hidden" value="0" name="_edd_download_use_git">
				<input type="checkbox" value="1" name="_edd_download_use_git" id="_edd_download_use_git" class="edd-form-group__input" <?php checked( $checked, 1 ); ?>>
				<label for="_edd_download_use_git" class="edd-form-group__label">
					<?php esc_html_e( 'Fetch download from a git repository.', 'edd-git-download-updater' ); ?>
				</label>
				<span class="spinner edd-git-enable__spinner"></span>
			</div>
		</div>
		<?php
	}

	/**
	 * Check to see if our field metabox sections should be removed.
	 * @since  1.0
	 * @return void
	 */
	public function init() {
		global $post;

		if ( isset( $post ) && 1 == get_post_meta( $post->ID, '_edd_download_use_git', true ) ) {
			$this->register_git_section();
		}
	}

	/**
	 * Remove default field metabox sections if we are using Git.
	 * @since  1.0
	 * @return void
	 */
	public function register_git_section() {
		// Remove the default EDD file editing section.
		remove_action( 'edd_meta_box_files_fields', 'edd_render_files_field', 20 );
		remove_action( 'edd_render_file_row', 'edd_render_file_row', 10, 3 );

		// Add our settings to the download editor.
		add_action( 'edd_meta_box_files_fields', array( $this, 'edd_files_fields' ), 20 );
		add_action( 'admin_print_footer_scripts-post.php', array( $this, 'output_git_modal' ), 20 );
		add_action( 'edd_render_file_row', array( $this, 'edd_render_file_row' ), 10, 3 );
	}

	/**
	 * Save our "Use Git" checkbox setting.
	 * @since  1.0
	 * @param  int  $post_id
	 * @return void
	 */
	public function save_post( $post_id ) {
		// Bail if we aren't saving a download
		if ( ! isset( $_POST['post_type'] ) or $_POST['post_type'] !== 'download' ) {
			return $post_id;
		}

		// verify if this is an auto save routine.
		// If our form has not been submitted, we don't want to do anything.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return $post_id;
		}

		// If this is a quick edit save, we don't want to do anything either.
		$inline_edit = filter_input( INPUT_POST, '_inline_edit', FILTER_DEFAULT );
		if ( $inline_edit && wp_verify_nonce( $inline_edit, 'inlineeditnonce' ) ) {
			return $post_id;
		}

		// Verify that this user can edit downloads
		if ( ! current_user_can( 'edit_product', $post_id ) ) {
			return $post_id;
		}

		$use_git = esc_html( $_POST['_edd_download_use_git'] );

		update_post_meta( $post_id, '_edd_download_use_git', $use_git );

		if ( $use_git ) {
			$files = $_POST['edd_download_files'];
			foreach ( $files as $key => $file ) {
				if ( empty( $file['name'] ) ) {
					$files[ $key ]['name'] = basename( $file['file'] );
				}
			}

			update_post_meta( $post_id, 'edd_download_files', $files );
		}
	}

	/*
	 * Include our JS.
	 *
	 * @since 1.1
	 * @return void
	 */
	public function admin_js() {
		if ( $this->can_enqueue() ) {
			global $post;
			if ( is_object( $post ) ) {
				$post_id = $post->ID;
			} else {
				$post_id = '';
			}
			$this->register_select2( $post_id );
			wp_enqueue_script( 'edd-git-updater', EDD_GIT_PLUGIN_URL . 'assets/build/js/admin.js', array( 'jquery', 'jquery-chosen', 'jquery-ui-core' ), EDD_GIT_VERSION, true );
			wp_localize_script(
				'edd-git-updater',
				'gitUpdater',
				array(
					'pluginURL'           => EDD_GIT_PLUGIN_URL,
					'useGit'              => get_post_meta( $post_id, '_edd_download_use_git', true ),
					'currentGitUrl'       => $this->instance->repos->get_current_repo_url( $post_id ),
					'currentTag'          => $this->instance->repos->get_current_tag( $post_id ),
					'currentAsset'        => $this->instance->repos->get_current_asset( $post_id ),
					'downloadButton'      => array(
						'dirty' => __( 'Fetch Now', 'edd-git-download-updater' ),
						'clean' => __( 'Fetched', 'edd-git-download-updater' ),
						'none'  => __( 'File Unavailable', 'edd-git-download-updater' ),
					),
					'invalidID'           => __( 'Please enter a valid Client ID', 'edd-git-download-updater' ),
					'invalidSecret'       => __( 'Please enter a valid Client Secret', 'edd-git-download-updater' ),
					'noTags'              => __( 'No tags available', 'edd-git-download-updater' ),
					'placeholderTags'     => __( 'Select a tag', 'edd-git-download-updater' ),
					'supportsSelectAsset' => edd_git_download_updater()->providerRegistry->getProvidersThatSupport( EDD\GitDownloadUpdater\Providers\SelectTagAsset::class ),
					'sourceCode'          => __( 'Release Source Code', 'edd-git-download-updater' ),
					'cancel'              => __( 'Cancel', 'edd-git-download-updater' ),
					'fetch_continue'      => __( 'Fetch and Continue', 'edd-git-download-updater' ),
					'defaultError'        => esc_html__( 'An unexpected error has occurred.', 'edd-git-download-updater' ),
				)
			);
		}
	}

	/*
	 * Include our CSS.
	 *
	 * @since 1.1
	 * @return void
	 */
	public function admin_css() {
		if ( $this->can_enqueue() ) {
			wp_enqueue_style( 'edd-git-updater', EDD_GIT_PLUGIN_URL . 'assets/css/admin.css', array( 'wp-jquery-ui-dialog' ), EDD_GIT_VERSION );
		}
	}

	/**
	 * If it isn't registered already, register the select2 script/style for back compat.
	 *
	 * @since 1.2
	 * @param int $post_id
	 */
	private function register_select2( $post_id ) {
		global $wp_scripts;

		$scripts          = array_merge( $wp_scripts->registered, $wp_scripts->queue );
		$select2_enqueued = false;

		foreach ( $scripts as $key => $script ) {

			$handle = empty( $script->handle ) ? '' : strtolower( $script->handle );
			$src    = empty( $script->src ) ? '' : strtolower( $script->src );

			if ( false !== strpos( $handle, 'select2' ) || false !== strpos( $src, 'select2' ) ) {
				$select2_enqueued = true;
				break;
			}
		}

		if ( false === $select2_enqueued ) {
			wp_register_script( 'jquery-select2', EDD_GIT_PLUGIN_URL . 'assets/build/js/select2.min.js', array( 'jquery' ), EDD_GIT_VERSION );
			wp_register_style( 'jquery-select2', EDD_GIT_PLUGIN_URL . 'assets/css/select2.min.css', null, EDD_GIT_VERSION );
		}
	}

	/**
	 * Checks to see if the plugin script/style should be enqueued.
	 *
	 * @since 1.2
	 * @return boolean
	 */
	private function can_enqueue() {

		if ( $this->is_git_settings_page() ) {
			return true;
		}

		// If we are on the download edit screen.
		global $pagenow, $typenow;
		if ( in_array( $pagenow, array( 'post.php', 'post-new.php' ), true ) && 'download' === $typenow ) {
			return true;
		}

		return false;
	}

	/*
	 * Output our GitHub Authorize Button
	 *
	 * @since 1.1
	 * @return void
	 */
	public function gh_authorize_button() {
		$edd_settings = get_option( 'edd_settings' );
		if ( ! empty( $edd_settings['gh_access_token'] ) ) {
			$connected    = '';
			$disconnected = 'display:none;';
		} else {
			$connected    = 'display:none;';
			$disconnected = '';
		}

		$html  = '<button style="' . esc_attr( $connected ) . '" id="edd-github-disconnect" class="button button-secondary edd-git-github-connected">' . __( 'Disconnect from GitHub', 'edd-git-download-updater' ) . '</button>';
		$html .= '<button style="' . esc_attr( $disconnected ) . '" id="edd-github-auth" class="button button-secondary edd-git-github-disconnected">' . __( 'Authorize with GitHub', 'edd-git-download-updater' ) . '</button>';

		echo $html;
	}

	/*
	 * Display our Bitbucket description
	 *
	 * @since 1.1
	 * @return void
	 */
	public function bb_desc() {

		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		$username = edd_get_option( 'bb_username' );
		$password = edd_get_option( 'bb_app_password' );
		if ( ! empty( $username ) && ! empty( $password ) ) {
			if ( $this->is_bitbucket_authorized() ) {
				$status_badge = new EDD\Utils\StatusBadge(
					array(
						'status'   => 'success',
						'label'    => __( 'Connected', 'edd-git-download-updater' ),
						'icon'     => 'yes',
						'dashicon' => true,
					)
				);
				echo $status_badge->get();
			} else {
				$status_badge = new EDD\Utils\StatusBadge(
					array(
						'status'   => 'error',
						'label'    => __( 'Not Connected', 'edd-git-download-updater' ),
						'icon'     => 'no',
						'dashicon' => true,
					)
				);
				echo $status_badge->get();
				echo '<p>';
				esc_html_e( 'Your Bitbucket account could not be authenticated. Please double check your credentials.', 'edd-git-download-updater' );
				echo '</p>';
			}
			return;
		}

		$html = '';

		if ( defined( 'EDD_GIT_BB_USER' ) && defined( 'EDD_GIT_BB_PASSWORD' ) ) {
			$authorization = $this->get_bitbucket_authorization();
			if ( empty( $authorization ) ) {
				$html  = sprintf(
					'<p>%1$s</p><ul><li>%2$s: <code>%3$s</code></li><li>%4$s: <code>*******************</code></li></ul>',
					__( 'Connected to Bitbucket.', 'edd-git-download-updater' ),
					__( 'Username', 'edd-git-download-updater' ),
					EDD_GIT_BB_USER,
					__( 'Password', 'edd-git-download-updater' )
				);
				$html .= '<p>' . sprintf(
					/* translators: 1. opening anchor tag; do not translate. 2. Closing anchor tag; do not translate.  */
					__( 'Please update your Bitbucket connection. %1$sCreate an app password%2$s and enter it here with your username. You may then remove your credentials from the wp-config.php file.', 'edd-git-download-updater' ),
					'<a href="https://support.atlassian.com/bitbucket-cloud/docs/app-passwords/" target="_blank">',
					'</a>'
				) . '</p>';
			} else {
				$html  = sprintf(
					'<p><strong>%1$s</strong></p><ul><li>%2$s: <code>%3$s</code></li><li>%4$s: <code>*******************</code></li></ul>',
					__( 'You are not connected to Bitbucket.', 'edd-git-download-updater' ),
					is_email( EDD_GIT_BB_USER ) ? __( 'Email', 'edd-git-download-updater' ) : __( 'Username', 'edd-git-download-updater' ),
					EDD_GIT_BB_USER,
					__( 'Password', 'edd-git-download-updater' )
				);
				$html .= $authorization;
			}
		}

		if ( empty( $username ) || empty( $password ) ) {
			$html .=
				'<p>' .
				sprintf(
					/* translators: 1. opening anchor tag; do not translate. 2. Closing anchor tag; do not translate.  */
					__( 'To connect to Bitbucket, please %1$screate an app password%2$s and enter it here with your username.', 'edd-git-download-updater' ),
					'<a href="https://bitbucket.org/account/settings/app-passwords/" target="_blank">',
					'</a>'
				) .
				'</p>';

			$html .=
				'<p>' .
				sprintf(
					'Learn more about %1$sapp passwords%2$s.',
					'<a href="https://easydigitaldownloads.com/docs/git-download-updater-setup-documentation/" target="_blank">',
					'</a>'
				)
				. '</p>';
		}

		echo $html;
	}

	/**
	 * Gets the text for the Bitbucket authorization status.
	 *
	 * @uses $this->is_bitbucket_authorized()
	 * @return string
	 */
	private function get_bitbucket_authorization() {
		if ( $this->is_bitbucket_authorized() ) {
			return '';
		}

		$output = sprintf(
			'<div class="notice inline notice-error"><p>%s</p>',
			__( 'Your Bitbucket account could not be authenticated. Please double check your Bitbucket credentials.', 'edd-git-download-updater' ),
		);
		if ( defined( 'EDD_GIT_BB_USER' ) && is_email( EDD_GIT_BB_USER ) ) {
			$output .= '<p>' . sprintf(
				/* translators: 1. opening strong tag; do not translate. 2. closing strong tag; do not translate. 3. opening anchor tag; do not translate. 4. closing anchor tag; do not translate. */
				__( '%1$sNote:%2$s Bitbucket no longer allows authentication with an email address and personal account password. You will need to %3$sset up an app password%4$s and update your settings here. Once you have done that, you can remove the constants from the wp-config.php file.', 'edd-git-download-updater' ),
				'<strong>',
				'</strong>',
				'<a href="https://docs.easydigitaldownloads.com/article/1240-git-download-updater---setup-documentation" target="_blank">',
				'</a>'
			) . '</p>';
		}

		$output .= '</div>';

		return $output;
	}

	/**
	 * Checks whether the Bitbucket connection is authorized.
	 *
	 * @since 1.3.1
	 * @return bool
	 */
	private function is_bitbucket_authorized() {
		$username = edd_get_option( 'bb_username' );
		$password = edd_get_option( 'bb_app_password' );
		$token    = false;
		if ( ! empty( $username ) && ! empty( $password ) ) {
			// phpcs: base64 encoding is okay here.
			$token = base64_encode( $username . ':' . $password );
		} elseif ( defined( 'EDD_GIT_BB_USER' ) && defined( 'EDD_GIT_BB_PASSWORD' ) ) {
			// phpcs: base64 encoding is okay here.
			$token = base64_encode( EDD_GIT_BB_USER . ':' . EDD_GIT_BB_PASSWORD );
		}
		if ( ! $token ) {
			return false;
		}
		$response = wp_safe_remote_get(
			'https://api.bitbucket.org/2.0/user/permissions/repositories',
			array(
				'headers' => array(
					'Authorization' => "Basic {$token}",
					'Connection'    => 'keep-alive',
				),
			)
		);

		$response_code = (int) wp_remote_retrieve_response_code( $response );

		return in_array( $response_code, array( 200, 404 ), true );
	}

	/*
	 * Output our GitHub description text
	 *
	 * @since 1.1
	 * @return void
	 */
	public function gh_desc() {
		$edd_settings = edd_get_settings();
		if ( ! empty( $edd_settings['gh_access_token'] ) ) {
			$status_badge = new EDD\Utils\StatusBadge(
				array(
					'status'   => 'success',
					'label'    => __( 'Connected', 'edd-git-download-updater' ),
					'icon'     => 'yes',
					'dashicon' => true,
				)
			);
			echo $status_badge->get();
			return;
		}

		printf( '<p>%s</p>', esc_html__( 'Updating from GitHub repositories requires a one-time application setup and authorization.', 'edd-git-download-updater' ) );
		printf( '<p>%s</p>', esc_html__( 'Follow these steps to generate your access credentials:', 'edd-git-download-updater' ) );
		echo '<ol>';
		echo '<li>';
		echo wp_kses(
			sprintf(
				/* translators:
				1. Opening anchor tag, do not translate
				2. Closing anchor tag, do not translate
				3. Opening strong tag, do not translate
				4. Closing strong tag, do not translate
				5. Opening code tag, do not translate
				6. Closing code tag, do not translate
				7. Website home URL
				*/
				__( '%1$sCreate an application%2$s with the %3$sMain URL%4$s and %3$sCallback URL%4$s both set to %5$s%7$s%6$s', 'edd-git-download-updater' ),
				'<a href="https://github.com/settings/applications/new" target="_blank">',
				'</a>',
				'<strong>',
				'</strong>',
				'<code>',
				'</code>',
				esc_url( get_home_url() )
			),
			array(
				'a'      => array(
					'href'   => true,
					'target' => true,
				),
				'strong' => true,
				'code'   => true,
			)
		);
		echo '</li>';
		echo '<li>';
		echo wp_kses(
			sprintf(
				/* translators:
				1. Opening strong tag, do not translate
				2. Closing strong tag, do not translate
				3. Opening anchor tag, do not translate
				4. Closing anchor tag, do not translate
				*/
				__( 'Copy the %1$sClient ID%2$s and %1$sClient Secret%2$s from your %3$sapplication details%4$s into the fields below.', 'edd-git-download-updater' ),
				'<strong>',
				'</strong>',
				'<a href="https://github.com/settings/applications" target="_blank">',
				'</a>'
			),
			array(
				'strong' => true,
				'a'      => array(
					'href'   => true,
					'target' => true,
				),
			)
		);
		echo '</li>';
		printf( '<li>%s</li>', esc_html__( 'Authorize with GitHub.', 'edd-git-download-updater' ) );
		echo '</ol>';
	}

	/**
	 * Output our field field metabox sections
	 * @since  1.0
	 * @param  integer $post_id
	 * @return void
	 */
	public function edd_files_fields( $post_id = 0 ) {
		$type             = edd_get_download_type( $post_id );
		$files            = edd_get_download_files( $post_id );
		$variable_pricing = edd_has_variable_prices( $post_id );
		$display          = $type == 'bundle' ? ' style="display:none;"' : '';
		$variable_display = $variable_pricing ? '' : 'display:none;';
		?>
		<div id="edd_download_files"<?php echo $display; ?>>
			<div id="edd_git_error">

			</div>
			<input type="hidden" id="edd_download_files" class="edd_repeatable_upload_name_field" value=""/>

			<div id="edd_file_fields" class="edd_meta_table_wrap">
				<div class="widefat edd_repeatable_table">
					<div class="edd-file-fields edd-repeatables-wrap">
						<?php
						$args = array(
							'name'            => '',
							'file'            => '',
							'condition'       => false,
							'attachment_id'   => false,
							'git_url'         => '',
							'git_folder_name' => '',
							'git_version'     => '',
							'git_file_asset'  => '',
						);
						if ( ! empty( $files ) && is_array( $files ) ) :
							$files = array_slice( $files, 0, 1 );
							foreach ( $files as $key => $value ) :
								$args = wp_parse_args( $value, $args );
								$args = apply_filters( 'edd_file_row_args', $args, $value );
								?>
								<div class="edd_repeatable_row" data-key="<?php echo esc_attr( $key ); ?>">
									<?php do_action( 'edd_render_file_row', 0, $args, $post_id ); ?>
								</div>
								<?php
							endforeach;
						else :
							$args = apply_filters( 'edd_file_row_args', $args, array() );
							?>
							<div class="edd_repeatable_row" data-key="0">
								<?php do_action( 'edd_render_file_row', 0, $args, $post_id ); ?>
							</div>
						<?php endif; ?>
						</div>
					</div>
				</div>
			</div>
		<?php
	}

	/**
	 * Render our file row
	 *
	 * @since   1.0
	 *
	 * @param int   $key
	 * @param array $args
	 * @param int   $post_id
	 *
	 * @return  void
	 */
	public function edd_render_file_row( $key, $args, $post_id ) {
		$defaults = array(
			'name'            => null,
			'file'            => null,
			'condition'       => null,
			'attachment_id'   => null,
			'git_url'         => null,
			'git_folder_name' => null,
			'git_version'     => null,
			'git_file_asset'  => null,
			'git_provider'    => null,
		);

		$args = wp_parse_args( $args, $defaults );

		$prices = edd_get_variable_prices( $post_id );

		$variable_pricing = edd_has_variable_prices( $post_id );
		$variable_display = $variable_pricing ? '' : ' style="display:none;"';

		$repos             = array();
		$current_tags      = array();
		$repo_slug         = '';
		$default_file_name = '';

		if ( ! empty( $args['git_url'] ) ) {
			$repo              = wp_parse_url( $args['git_url'] );
			$current_tags      = $this->instance->repos->fetch_tags( $args['git_url'] );
			$path              = $repo['path'];
			$repo_slug         = explode( '/', $path );
			$repo_slug         = $repo_slug[2];
			$default_file_name = $repo_slug . '-' . $args['git_version'] . '.zip';
		}
		?>
		<div class="edd-repeatable-row-header">
			<span class="edd-repeatable-row-title"><?php esc_html_e( 'File Downloads:', 'edd-git-download-updater' ); ?></span>
		</div>
		<input type="hidden" name="edd_download_files[<?php echo esc_attr( $key ); ?>][attachment_id]" value="0">
		<input type="hidden" id="edd_git_file" name="edd_download_files[<?php echo esc_attr( $key ); ?>][file]" value="<?php echo esc_attr( $args['file'] ); ?>">
		<div class="edd-repeatable-row-standard-fields edd-git-download-fields">
			<div class="edd-form-group edd-form-group--git-has-button">
				<div class="edd-form-group__control">
					<label for="edd-download-files-<?php echo esc_attr( $key ); ?>-git_url" class="edd-form-group__label"><?php esc_html_e( 'Git Repository', 'edd-git-download-updater' ); ?></label>
					<select id="edd-download-files-<?php echo esc_attr( $key ); ?>-git_url" name="edd_download_files[<?php echo esc_attr( $key ); ?>][git_url]" class="edd-form-group__input git-repo">
						<?php
						$this->output_repo_options( $repos, $args['git_url'] );
						?>
					</select>
				</div>
				<div class="edd-form-group__control">
					<label for="edd-download-files-<?php echo esc_attr( $key ); ?>-git_version" class="edd-form-group__label"><?php esc_html_e( 'Version Tag', 'edd-git-download-updater' ); ?></label>
					<select id="edd-download-files-<?php echo esc_attr( $key ); ?>-git_version" name="edd_download_files[<?php echo esc_attr( $key ); ?>][git_version]" class="edd-form-group__input git-tag">
					<?php
					foreach ( $current_tags as $tag ) {
						?>
						<option value="<?php echo esc_attr( $tag ); ?>" <?php selected( $tag == $args['git_version'] ); ?>><?php echo esc_html( $tag ); ?></option>
						<?php
					}
					?>
					</select>
					<div class="git-tag-spinner" style="display:none;">
						<span class="spinner" style="visibility:visible;display:block;"></span>
					</div>
				</div>
				<div class="edd-form-group edd-git-release-assets" style="display: none;">
					<div class="edd-form-group__control">
						<label for="_edd_download_release_asset_<?php echo esc_attr( $key ); ?>" class="edd-form-group__label">
							<?php esc_html_e( 'Asset File', 'edd-git-download-updater' ); ?>
						</label>
						<select
							id="_edd_download_release_asset_<?php echo esc_attr( $key ); ?>"
							class="edd-git-asset"
							name="edd_download_files[<?php echo esc_attr( $key ); ?>][git_file_asset]"
							data-selected="<?php echo esc_attr( $args['git_file_asset'] ); ?>"
						>
							<option value=""><?php esc_html_e( 'Loading assets...', 'edd-git-download-updater' ); ?></option>
						</select>
					</div>
				</div>
				<button class="button edd-git-fetch-repos edd-git-button button-secondary">
				<?php
				echo wp_kses(
					sprintf(
						/* translators: Repositories noun */
						__( 'Refresh %s', 'edd-git-download-updater' ),
						(
							'<span class="screen-reader-text">' .
							__( 'Tags', 'edd-git-download-updater' ) .
							'</span>'
						)
					),
					array(
						'span' => array(
							'class' => true,
						),
					)
				);
				?>
				</button>
			</div>

			<div class="edd-form-group git-file-name">
				<div class="edd-form-group__control">
					<?php
					$name = '';
					if ( ! empty( $args['git_url'] ) && is_array( $args['git_url'] ) ) {
						$tmp          = explode( '/', $args['git_url'] );
						$git_repo     = $tmp[4];
						$tag          = $args['git_version'];
						$default_name = "{$git_repo}-{$tag}.zip";

						if ( $args['name'] !== $default_name ) {
							$name = $args['name'];
						}
					}

					printf(
						'<label for="edd-download-files-%s-name" class="edd-form-group__label">%s</label>',
						esc_attr( $key ),
						esc_html__( 'File Name', 'edd-git-download-updater' )
					);
					echo EDD()->html->text(
						array(
							'name'        => 'edd_download_files[' . esc_attr( $key ) . '][name]',
							'id'          => 'edd-download-files-' . esc_attr( $key ) . '-name',
							'value'       => $args['name'],
							'placeholder' => $default_file_name,
							'class'       => 'edd-form-group__input edd_repeatable_name_field large-text',
						)
					);
					?>
				</div>
			</div>

			<div class="edd-form-group edd-form-group--git-has-button">
				<div class="edd-form-group__control git-folder-name"<?php echo ! empty( $args['git_file_asset'] ) ? ' style="display:none;"' : ''; ?>>
					<label for="edd-download-files-<?php echo esc_attr( $key ); ?>-git_folder_name" class="edd-form-group__label"><?php esc_html_e( 'Plugin Folder Name', 'edd-git-download-updater' ); ?></label>
					<?php
					echo EDD()->html->text(
						array(
							'name'        => 'edd_download_files[' . esc_attr( $key ) . '][git_folder_name]',
							'id'          => 'edd-download-files-' . esc_attr( $key ) . '-git_folder_name',
							'value'       => $args['git_folder_name'],
							'placeholder' => $repo_slug,
							'class'       => 'edd-form-group__input edd_repeatable_upload_field large-text',
						)
					);
					?>
				</div>
				<?php
				$text     = __( 'Fetched', 'edd-git-download-updater' );
				$disabled = '';
				if ( empty( $args['file'] ) ) {
					$text     = __( 'File Unavailable', 'edd-git-download-updater' );
					$disabled = ' disabled';
				}
				printf(
					'<button class="button button-secondary edd-git-update edd-git-button"%s><span class="edd-git-text">%s</span></button>',
					esc_attr( $disabled ),
					esc_html( $text )
				);
				?>
			</div>

			<div class="edd-form-group pricing"<?php echo $variable_display; ?>>
				<div class="edd-form-group__control">
					<?php
					$options = array();

					if ( $prices ) {
						foreach ( $prices as $price_key => $price ) {
							$options[ $price_key ] = $prices[ $price_key ]['name'];
						}
					}

					printf(
						'<label for="edd-download-files-%s-condition" class="edd-form-group__label">%s</label>',
						esc_attr( $key ),
						esc_html__( 'Price Assignment', 'edd-git-download-updater' )
					);
					echo EDD()->html->select(
						array(
							'name'             => 'edd_download_files[' . esc_attr( $key ) . '][condition]',
							'id'               => 'edd-download-files-' . esc_attr( $key ) . '-condition',
							'class'            => 'edd-form-group__input edd_repeatable_condition_field git-condition',
							'options'          => $options,
							'selected'         => $args['condition'],
							'show_option_none' => false,
						)
					);
					?>
				</div>
			</div>
		</div>
		<div class="edd-git-download-errors hidden"></div>

		<?php
		do_action( 'edd_download_file_table_row', $post_id, $key, $args );
	}

	/**
	 * Outputs the admin modal.
	 *
	 * @since 1.2
	 * @return void
	 */
	public function output_git_modal() {
		$screen = get_current_screen();
		if ( 'download' !== $screen->id ) {
			return;
		}
		?>
		<div id="edd-git-modal" title="<?php esc_html_e( 'Your Download is Not Ready', 'edd-git-download-updater' ); ?>"></div>
		<div class="edd-git-fetch-prompt" style="display:none;">
			<p>
				<?php esc_html_e( 'The git file has not been fetched. Would you like to fetch it first?', 'edd-git-download-updater' ); ?>
			</p>
		</div>
		<?php
	}

	/**
	 * Take our repo array and return our HTML options
	 * @since  1.0
	 * @param  array  $repos
	 * @param  string  $current_repo URL of our current repo.
	 * @return void
	 */
	public function output_repo_options( $repos, $current_repo = '' ) {
		?>
		<option value="" data-slug=""><?php esc_html_e( 'Select a repository', 'edd-git-download-updater' ); ?></option>
		<?php
		if ( ( ! defined( 'DOING_AJAX' ) || ! DOING_AJAX ) ) {
			?>
			<option value="<?php echo esc_attr( $current_repo ); ?>" <?php echo $current_repo ? 'selected' : ''; ?>><?php echo esc_html( $current_repo ); ?></option>
			<?php
			return;
		}

		foreach ( $repos as $source => $sourceRepos ) {
			foreach ( $sourceRepos as $repoOwner => $ownersRepos ) {
				?>
				<optgroup class="<?php echo esc_attr( $source ); ?>" label="<?php echo esc_html( $source . '/' . $repoOwner ); ?>">
				<?php
				foreach ( $ownersRepos as $repoUrl => $repoName ) {
					?>
					<option
						value="<?php echo esc_attr( $repoUrl ); ?>"
						<?php selected( $repoUrl, $current_repo ); ?>
						class="<?php echo esc_attr( $source ); ?>"
						data-provider="<?php echo esc_attr( $source ); ?>"
						data-slug="<?php echo esc_attr( $repoName ); ?>"
						data-repo-owner="<?php echo esc_attr( $repoOwner ); ?>"
					><?php echo esc_html( $repoName ); ?></option>
					<?php
				}
				?>
				</optgroup>
				<?php
			}
		}
	}

	/**
	 * Update the flyout menu docs link on the Git Download Updater settings page.
	 *
	 * @since 1.3.1
	 * @param string $link
	 * @return string
	 */
	public function update_docs_link( $link ) {

		if ( $this->is_git_settings_page() ) {
			return 'https://easydigitaldownloads.com/docs/git-download-updater-setup-documentation/';
		}

		return $link;
	}

	/**
	 * Checks whether the current page is the Git Download Updater settings page.
	 *
	 * @since 1.3.1
	 * @return bool
	 */
	private function is_git_settings_page() {
		if ( ! edd_is_admin_page( 'settings', 'extensions' ) ) {
			return false;
		}

		if ( ! empty( $_GET['section'] ) && 'edd-git' === $_GET['section'] ) {
			return true;
		}

		return count( edd_get_settings_tab_sections( 'extensions' ) ) < 3;
	}
}
