<?php
/**
 * GitHub.php
 *
 * @package   edd-git-download-updater
 * @copyright Copyright (c) 2021, Easy Digital Downloads
 * @license   GPL2+
 * @since     1.3
 */

namespace EDD\GitDownloadUpdater\Providers;

use EDD\GitDownloadUpdater\Exceptions\ApiException;
use EDD\GitDownloadUpdater\Exceptions\ConfigurationException;
use EDD\GitDownloadUpdater\Exceptions\MissingConnectionException;
use EDD\GitDownloadUpdater\Exceptions\ResourceNotFoundException;

class GitHubProvider implements Provider, SelectTagAsset, ParsesApiErrors {

	const API_URL = 'https://api.github.com';

	/**
	 * @var ApiHandler
	 */
	protected $apiHandler;

	public function __construct() {
		$this->apiHandler = $this->getApiHandler();
	}

	/**
	 * @inheritDoc
	 *
	 * @return string
	 */
	public static function getId() {
		return 'github';
	}

	/**
	 * Builds a GitHub API handler.
	 *
	 * @since 1.3
	 *
	 * @return ApiHandler
	 */
	public function getApiHandler() {
		$handler = new ApiHandler();

		$accessToken = edd_get_option( 'gh_access_token', '' );
		if ( ! empty( $accessToken ) ) {
			$handler->withAuthorizationHeader( "token {$accessToken}" );
		}

		$handler->withApiUrl( self::API_URL )
		        ->withHeader( 'Accept', 'application/vnd.github.v3+json' );

		return $handler;
	}

	/**
	 * Fetches all repositories from the API.
	 *
	 * @since 1.3
	 *
	 * @return array
	 * @throws ApiException|ConfigurationException|MissingConnectionException
	 */
	public function getRepositories() {
		$hasMore = true;
		$page    = 1;
		$repos   = array();

		while ( $hasMore ) {
			$response = $this->apiHandler->makeRequest(
				'user/repos?per_page=100&page=' . urlencode( $page )
			);

			$hasMore = ! empty( $response );

			if ( is_array( $response ) ) {
				foreach ( $response as $repo ) {
					if ( ! isset( $repo->owner->login ) ) {
						continue;
					}

					if ( ! array_key_exists( $repo->owner->login, $repos ) ) {
						$repos[ $repo->owner->login ] = array();
					}

					if ( isset( $repo->html_url ) && isset( $repo->name ) ) {
						$repos[ $repo->owner->login ][ $repo->html_url ] = $repo->name;
					}
				}
			}

			$page ++;
		}

		return $repos;
	}

	/**
	 * Retrieves a list of tags from the API.
	 *
	 * @since 1.3
	 *
	 * @param string $repoPath Path of the repository. Format: {org}/{repo}
	 *
	 * @return array
	 * @throws ResourceNotFoundException|ApiException|ConfigurationException|MissingConnectionException
	 */
	public function getTags( $repoPath ) {
		$response = $this->apiHandler->makeRequest(
			'repos/' . $repoPath . '/tags'
		);

		if ( ! is_array( $response ) || empty( $response ) || isset( $response->message ) ) {
			throw new ResourceNotFoundException( __( 'No tags found for this repository.', 'edd-git-download-updater' ) );
		}

		return wp_list_pluck( $response, 'name' );
	}

	/**
	 * @inheritDoc
	 *
	 * @param string $repoPath
	 * @param string $tagName
	 *
	 * @return array
	 * @throws ApiException|ConfigurationException|MissingConnectionException|ResourceNotFoundException
	 */
	public function getAssetsFromRepoTag( $repoPath, $tagName ) {
		// First get the release linked to this tag.
		try {
			$release = $this->apiHandler->makeRequest(
				'repos/' . $repoPath . '/releases/tags/' . urlencode( $tagName )
			);
		} catch ( ApiException $e ) {
			if ( 404 === $e->getCode() ) {
				/*
				 * If it's a 404 then that means there's no release associated with this tag, which
				 * then also means there are no assets.
				 */
				throw new ResourceNotFoundException( __( 'No valid assets found.', 'edd-git-download-updater' ) );
			} else {
				throw $e;
			}
		}

		if ( empty( $release->assets ) || ! is_array( $release->assets ) ) {
			throw new ResourceNotFoundException( __( 'No valid assets found.', 'edd-git-download-updater' ) );
		}

		$assets = array();

		foreach ( $release->assets as $asset ) {
			if ( $this->isValidAsset( $asset ) ) {
				$assets[] = array(
					'url'  => $asset->url,
					'name' => $asset->name,
				);
			}
		}

		if ( empty( $assets ) ) {
			throw new ResourceNotFoundException( __( 'No valid assets found.', 'edd-git-download-updater' ) );
		}

		return $assets;
	}

	/**
	 * Determines if an asset is one we can accept/download.
	 *
	 * @since 1.3
	 *
	 * @param object|array $asset
	 *
	 * @return bool
	 */
	private function isValidAsset( $asset ) {
		$asset        = (array) $asset;
		$requiredKeys = array( 'url', 'name', 'content_type' );
		if ( array_diff( $requiredKeys, array_keys( $asset ) ) ) {
			return false;
		}

		return in_array( $asset['content_type'], array( 'application/zip', 'application/x-zip-compressed' ) );
	}

	/**
	 * @inheritDoc
	 *
	 * @param string $repoPath
	 * @param string $tag
	 *
	 * @return string
	 */
	public function buildAssetUrlFromRepoAndTag( $repoPath, $tag ) {
		return self::API_URL . '/repos/' . $repoPath . '/zipball/' . urlencode( $tag );
	}

	/**
	 * @inheritDoc
	 *
	 * @param string $url
	 *
	 * @return string
	 * @throws ApiException|ConfigurationException|MissingConnectionException
	 */
	public function fetchZipFromUrl( $url ) {
		if ( false !== strpos( $url, '/releases/assets/' ) ) {
			$this->apiHandler->withHeader( 'Accept', 'application/octet-stream' );
		}

		$this->apiHandler->withApiUrl( '' )->makeRequest( $url );

		return wp_remote_retrieve_body( $this->apiHandler->lastResponse );
	}

	/**
	 * @inheritDoc
	 */
	public function parseApiErrors( ApiException $e ) {
		$responseBody = json_decode( wp_remote_retrieve_body( $e->getApiResponse() ) );

		if ( empty( $responseBody->message ) ) {
			return $e->getMessage();
		}

		if ( false !== strpos( $responseBody->message, 'has multiple possibilities' ) ) {
			return __( 'Unable to fetch zip file, as there is both a branch and tag with this name.', 'edd-git-download-updater' );
		}

		return $responseBody->message;
	}
}
