<?php
/**
 * ProviderRegistry.php
 *
 * @package   edd-git-download-updater
 * @copyright Copyright (c) 2021, Easy Digital Downloads
 * @license   GPL2+
 * @since     1.3
 */

namespace EDD\GitDownloadUpdater;

use EDD\GitDownloadUpdater\Exceptions\ResourceNotFoundException;

class ProviderRegistry extends \ArrayObject {

	/**
	 * @inheritDoc
	 *
	 * @param int|string $key
	 *
	 * @return false|mixed
	 * @throws ResourceNotFoundException
	 */
	#[\ReturnTypeWillChange]
	public function offsetGet( $key ) {
		if ( ! $this->offsetExists( $key ) ) {
			throw new ResourceNotFoundException( "No such provider {$key}." );
		}

		return parent::offsetGet( $key );
	}

	/**
	 * @inheritDoc
	 *
	 * @param int|string $key
	 * @param mixed      $value
	 */
	#[\ReturnTypeWillChange]
	public function offsetSet( $key, $value ) {
		if ( $this->offsetExists( $key ) ) {
			throw new \InvalidArgumentException( "The {$key} provider is already registered." );
		}

		$this->validateProvider( $value );

		parent::offsetSet( $key, $value );
	}

	/**
	 * Adds a new provider.
	 *
	 * @since 1.3
	 *
	 * @param string $className Provider class name.
	 */
	public function addProvider( $className ) {
		$this->validateProvider( $className );

		$this->offsetSet( $className::getId(), $className );
	}

	/**
	 * Validates a provider.
	 *
	 * @since 1.3
	 *
	 * @param string $className
	 *
	 * @throws \InvalidArgumentException
	 */
	private function validateProvider( $className ) {
		if ( ! class_exists( $className ) ) {
			throw new \InvalidArgumentException( "The {$className} class does not exist." );
		}

		$interfaces = class_implements( $className );
		if ( ! is_array( $interfaces ) || ! array_key_exists( Providers\Provider::class, $interfaces ) ) {
			throw new \InvalidArgumentException( "The {$className} class must implement the Provider interface." );
		}
	}

	/**
	 * Returns all registered providers.
	 *
	 * @since 1.3
	 *
	 * @return Providers\Provider[]
	 */
	public function getProviders() {
		$providers = array();
		foreach ( $this->getArrayCopy() as $key => $value ) {
			try {
				$providers[ $key ] = $this->getProvider( $key );
			} catch ( \Exception $e ) {

			}
		}

		return $providers;
	}

	/**
	 * @param string $offset
	 *
	 * @return Providers\Provider
	 * @throws ResourceNotFoundException
	 */
	public function getProvider( $offset ) {
		$className = $this->offsetGet( $offset );

		return new $className();
	}

	/**
	 * Determines if a given provider supports a feature.
	 *
	 * @param string $offset    ID of the provider.
	 * @param string $interface Interface name to check.
	 *
	 * @return bool
	 * @throws ResourceNotFoundException
	 */
	public function providerSupports( $offset, $interface ) {
		if ( ! $this->offsetExists( $offset ) ) {
			return false;
		}

		$providerClassName = $this->offsetGet( $offset );
		$interfaces        = class_implements( $providerClassName );

		return is_array( $interfaces ) && array_key_exists( $interface, $interfaces );
	}

	/**
	 * Returns the IDs of providers that support a given feature (Interface).
	 *
	 * @since 1.3
	 *
	 * @param string $featureInterface Interface name.
	 *
	 * @return string[] Array of provider IDs.
	 */
	public function getProvidersThatSupport( $featureInterface ) {
		$providers = array();
		foreach ( $this->getProviders() as $providerId => $provider ) {
			try {
				if ( $this->providerSupports( $providerId, $featureInterface ) ) {
					$providers[] = $providerId;
				}
			} catch ( \Exception $e ) {

			}
		}

		return $providers;
	}
}
