<?php
/**
 * Grab repos and tags from GitHub and Bitbucket
 *
 * @package EDD Git Download Updater
 *
 * @since  1.0
 */
class EDD_GIT_Download_Updater_Repos {
	/**
	 * Fetch our repos from either our cache or BB/GH
	 *
	 * @since  1.0
	 *
	 * @return array $repos
	 */
	public function fetch_repos() {
		$repos = get_transient( 'edd_git_repos' );
		if ( false !== $repos ) {
			return $repos;
		}

		$repos = array();
		foreach( edd_git_download_updater()->providerRegistry->getProviders() as $provider ) {
			try {
				$repos[ $provider::getId() ] = $provider->getRepositories();
			} catch ( \Exception $e ) {
				// @todo Improve error logging.
				edd_debug_log( sprintf( 'Git: Exception fetching repos: %s', $e->getMessage() ) );
			}
		}

		set_transient( 'edd_git_repos', $repos );

		return $repos;
	}

	/**
	 * Grab out tags from GH or BB
	 *
	 * @since  1.0
	 *
	 * @param  string $repo_url URL of our repo.
	 * @return array $tags
	 */
	public function fetch_tags( $repo_url ) {
		if ( false !== strpos( $repo_url, 'bitbucket.org' ) ) { // Bitbucket url.
			$replace = array( 'https://bitbucket.org/', 'api/2.0/repositories/' );
			$function = 'bb_get_tags';
		} else { // GitHub url.
			$replace = 'https://github.com/';
			$function = 'gh_get_tags';
		}

		$slug = str_replace( $replace, '', $repo_url );

		return $this->$function( $slug );
	}

	/**
	 * Fetch our tags from GitHub
	 *
	 * @since  1.0
	 *
	 * @param  string $tag_url URL of our repo.
	 * @return array   $tags
	 */
	public function gh_get_tags( $tag_url ) {
		try {
			/** @var \EDD\GitDownloadUpdater\Providers\GitHubProvider $gitHub */
			$gitHub = edd_git_download_updater()->providerRegistry->getProvider( 'github' );

			return $gitHub->getTags( $tag_url );
		} catch ( \EDD\GitDownloadUpdater\Exceptions\ResourceNotFoundException $e ) {
			return array(
				'error' => __( 'Could not find any tags for repository.', 'edd-git-download-updater' )
			);
		} catch ( \Exception $e ) {
			return array();
		}
	}

	/**
	 * Fetch our tags from Bitbucket
	 *
	 * @since  1.0
	 *
	 * @param  string $tag_url URL of our repo.
	 * @return array $tags
	 */
	public function bb_get_tags( $tag_url ) {
		try {
			/** @var \EDD\GitDownloadUpdater\Providers\GitHubProvider $bitbucket */
			$bitbucket = edd_git_download_updater()->providerRegistry->getProvider( 'bitbucket' );

			return $bitbucket->getTags( $tag_url );
		} catch ( \EDD\GitDownloadUpdater\Exceptions\ResourceNotFoundException $e ) {
			return array(
				'error' => __( 'Could not find any tags for repository.', 'edd-git-download-updater' )
			);
		} catch ( \Exception $e ) {
			return array();
		}
	}

	/**
	 * Return our current tag
	 *
	 * @since  1.0
	 *
	 * @param  integer $download_id The Download ID to get the current tag of.
	 * @return string   $version
	 */
	public function get_current_tag( $download_id ) {
		$version = false;
		$files = get_post_meta( $download_id, 'edd_download_files', true );
		if ( empty( $files ) ) {
			return $version;
		}

		foreach ( $files as $file ) {
			$version = isset( $file['git_version'] ) ? $file['git_version'] : '';

			if ( ! empty( $version ) ) {
				break;
			}
		}

		return $version;
	}

	/**
	 * Returns the currently selected asset.
	 *
	 * @since 1.3
	 *
	 * @param int $download_id
	 *
	 * @return string
	 */
	public function get_current_asset( $download_id ) {
		$asset = '';
		$files = get_post_meta( $download_id, 'edd_download_files', true );
		if ( empty( $files ) ) {
			return $asset;
		}

		foreach ( $files as $file ) {
			$asset = isset( $file['git_file_asset'] ) ? $file['git_file_asset'] : '';

			if ( ! empty( $asset ) ) {
				break;
			}
		}

		return $asset;
	}

	/**
	 * Return our current repo URL
	 *
	 * @since  1.0
	 *
	 * @param  integer $download_id The download ID to get the current repo URL for.
	 * @return string $url
	 */
	public function get_current_repo_url( $download_id ) {
		$url = false;
		$files = get_post_meta( $download_id, 'edd_download_files', true );
		if ( empty( $files ) ) {
			return $url;
		}

		foreach ( $files as $file ) {
			$url = isset( $file['git_url'] ) ? $file['git_url'] : '';

			if ( ! empty( $url ) ) {
				break;
			}
		}

		return $url;
	}

}
