<?php
/**
 * Filters.
 *
 * @package EDD_Custom_Prices
 * @since 1.5.9
 */

defined( 'ABSPATH' ) || exit;

/**
 * Fix download files when using variable pricing.
 * For now it will only return the files for the first variable option.
 *
 * @param array $files The files.
 * @param int   $id The download ID.
 * @param int   $variable_price_id The variable price ID.
 *
 * @return array
 */
function edd_cp_download_files( $files, $id, $variable_price_id ) {
	if ( ! edd_cp_has_custom_pricing( $id ) || -1 !== $variable_price_id ) {
		return $files;
	}
	remove_filter( 'edd_download_files', 'edd_cp_download_files' );
	$files = edd_get_download_files( $id, 1 );
	return $files;
}
add_filter( 'edd_download_files', 'edd_cp_download_files', 10, 3 );

/**
 * Hook into add to cart post
 *
 * @param int $download_id The ID of the download.
 */
function edd_cp_purchase_link_top( $download_id ) {

	$default_price = get_post_meta( $download_id, 'edd_cp_default_price', true );
	$min_price     = get_post_meta( $download_id, 'edd_cp_min', true );
	if ( function_exists( 'eddMultiCurrency' ) ) {
		try {
			$default_price = EDD_Multi_Currency\Utils\Currency::convert( $default_price, eddMultiCurrency( EDD_Multi_Currency\Checkout\CurrencyHandler::class )->getSelectedCurrency() );
			$min_price     = EDD_Multi_Currency\Utils\Currency::convert( $min_price, eddMultiCurrency( EDD_Multi_Currency\Checkout\CurrencyHandler::class )->getSelectedCurrency() );
		} catch ( \Exception $e ) {
			// Do nothing.
		}
	}

	$posted_custom_price = isset( $_POST['edd_cp_price'] ) ? sanitize_text_field( wp_unslash( $_POST['edd_cp_price'] ) ) : false;
	$custom_price        = ! empty( $posted_custom_price ) ? edd_format_amount( $posted_custom_price ) : '';
	$button_text         = get_post_meta( $download_id, 'cp_button_text', true );
	$currency_position   = edd_get_option( 'currency_position', 'before' );

	if ( empty( $custom_price ) && ! empty( $default_price ) ) {
		$custom_price = edd_format_amount( $default_price );
	}

	if ( edd_cp_has_custom_pricing( $download_id ) && ! edd_single_price_option_mode( $download_id ) ) {
		$wrapper_display = '';
		$download        = edd_get_download( $download_id );
		if ( edd_item_in_cart( $download_id, array() ) && ( ! $download->has_variable_prices() || ! $download->is_single_price_mode() ) ) {
			$wrapper_display = 'style="display:none;"';
		}
		?>
		<p class="edd-cp-container" <?php echo esc_html( $wrapper_display ); ?>>
			<label for="edd_cp_price" class="edd-cp-price-label"><?php esc_html_e( 'Name your price', 'edd_cp' ); ?></label>
			<?php if ( 'before' === $currency_position ) : ?>
				<?php echo esc_html( edd_currency_filter( '' ) ); ?> <input type="text" name="edd_cp_price" id="edd_cp_price" class="edd_cp_price" value="<?php echo esc_attr( $custom_price ); ?>" data-min="<?php echo $min_price; ?>" data-default-text="<?php echo esc_attr( $button_text ); ?>" />
			<?php else : ?>
				<input type="text" name="edd_cp_price" id="edd_cp_price" class="edd_cp_price" value="<?php echo esc_attr( $custom_price ); ?>" data-min="<?php echo esc_attr( $min_price ); ?>" data-default-text="<?php echo esc_attr( $button_text ); ?>" />
				<?php
					echo esc_html( edd_currency_filter( '' ) );
			endif;
			$min_no_format = floatval( $min_price );
			if ( ! empty( $min_no_format ) ) {
				echo ' <small>(min ' . esc_html( edd_currency_filter( edd_format_amount( $min_price ) ) ) . ')</small>';
			}
			?>
		</p>
		<?php
	}
}
add_filter( 'edd_purchase_link_top', 'edd_cp_purchase_link_top' );

/**
 * Update price before adding item to card
 *
 * @param array $cart_item The cart item.
 *
 * @return array
 */
function edd_cp_add_to_cart_item( $cart_item ) {
	remove_filter( 'edd_add_to_cart_item', 'edd_cp_add_to_cart_item' );

	if ( ! empty( $_POST['post_data'] ) || isset( $_POST['edd_cp_price'] ) ) {
		if ( ! empty( $_POST['post_data'] ) ) {
			$post_data = array();
			parse_str( $_POST['post_data'], $post_data );
			$custom_price = isset( $post_data['edd_cp_price'] ) ? $post_data['edd_cp_price'] : null;
		} else {
			$custom_price = $_POST['edd_cp_price'];
		}

		// Multi-option purchase mode.
		if ( ! empty( $_POST['price_ids'] ) && is_array( $custom_price ) && edd_single_price_option_mode( $cart_item['id'] ) ) {
			foreach ( $_POST['price_ids'] as $price_id ) {
				$original_price = edd_get_price_option_amount( $cart_item['id'], $price_id );
				$price_id       = absint( $price_id );

				// It's only a custom price if the price doesn't match the original.
				if ( isset( $custom_price[ $price_id ] ) && $custom_price[ $price_id ] !== $original_price ) {
					$cart_item['options']['custom_price'][ $price_id ] = edd_sanitize_amount( $custom_price[ $price_id ] );
				}
			}
		} elseif ( ( ! is_null( $custom_price ) && '' !== $custom_price ) && ( ( edd_has_variable_prices( $cart_item['id'] ) ) || ! edd_has_variable_prices( $cart_item['id'] ) ) ) {
			$cart_item['options']['custom_price'] = edd_sanitize_amount( $custom_price );
		}
	}

	$bonus_item = get_post_meta( $cart_item['id'], 'bonus_item', true );

	if ( ! empty( $bonus_item['product'] ) && ! empty( $custom_price ) ) {
		$bonus_item_price = $bonus_item['price'];
		if ( function_exists( 'eddMultiCurrency' ) ) {
			try{
				$bonus_item_price = EDD_Multi_Currency\Utils\Currency::convert( $bonus_item_price, eddMultiCurrency( EDD_Multi_Currency\Checkout\CurrencyHandler::class )->getSelectedCurrency() );
			} catch ( \Exception $e ) {
				// Do nothing.
			}
		}
		$bonus_item_price = (float) $bonus_item_price;
		$custom_price     = (float) $custom_price;
		$condition_met    = false;
		if ( 'equal_to' === $bonus_item['condition'] && $custom_price === $bonus_item_price ) {
			$condition_met = true;
		} elseif ( 'less_than' === $bonus_item['condition'] && $custom_price < $bonus_item_price ) {
			$condition_met = true;
		} elseif ( 'more_than' === $bonus_item['condition'] && $custom_price > $bonus_item_price ) {
			$condition_met = true;
		}

		if ( $condition_met ) {
			edd_add_to_cart(
				$bonus_item['product'],
				array(
					'price_id'        => null,
					'is_cp_bonus'     => true,
					'cp_bonus_parent' => $cart_item['id'],
				)
			);
		}
	}

	add_filter( 'edd_add_to_cart_item', 'edd_cp_add_to_cart_item' );

	return $cart_item;
}
add_filter( 'edd_add_to_cart_item', 'edd_cp_add_to_cart_item' );

/**
 * Update cart options before sending to cart
 *
 * @param int   $download_id The download ID.
 * @param array $options The options.
 *
 * @return array
 */
function edd_cp_pre_add_to_cart( $download_id, $options ) {
	remove_filter( 'edd_pre_add_to_cart', 'edd_cp_pre_add_to_cart', 10, 2 );

	if ( ! empty( $_POST['post_data'] ) || isset( $_POST['edd_cp_price'] ) ) {
		if ( ! empty( $_POST['post_data'] ) ) {
			$post_data = array();
			parse_str( $_POST['post_data'], $post_data );
			$custom_price = isset( $post_data['edd_cp_price'] ) ? $post_data['edd_cp_price'] : null;
		} else {
			$custom_price = $_POST['edd_cp_price'];
		}
		if ( ! is_null( $custom_price ) && ( ( edd_has_variable_prices( $download_id ) && $options['price_id'] == -1 ) || ! edd_has_variable_prices( $download_id ) ) ) {
			$options['custom_price'] = edd_sanitize_amount( $custom_price );
		}
	}
	add_filter( 'edd_pre_add_to_cart', 'edd_cp_pre_add_to_cart', 10, 2 );
	return $options;
}
add_filter( 'edd_pre_add_to_cart', 'edd_cp_pre_add_to_cart', 10, 2 );

/**
 * Update price if custom price exists and meets criteria
 *
 * @param float $price The price.
 * @param int   $item_id The item ID.
 * @param array $options The options.
 *
 * @return float
 */
function edd_cp_cart_item_price( $price, $item_id, $options = array() ) {
	if ( ( edd_cp_has_custom_pricing( $item_id ) && isset( $options['custom_price'] ) ) || ( isset( $options['is_cp_bonus'] ) && $options['is_cp_bonus'] ) ) {
		if ( isset( $options['is_cp_bonus'] ) ) {
			$price = 0;
		} else {
			$min_price    = get_post_meta( $item_id, 'edd_cp_min', true );
			$custom_price = $options['custom_price'];

			if ( edd_has_variable_prices( $item_id ) && edd_single_price_option_mode( $item_id ) ) {
				$price_id = isset( $options['price_id'] ) ? $options['price_id'] : false;

				if ( $price_id && isset( $custom_price[ $price_id ] ) ) {
					$variable_price = $custom_price[ $price_id ];

					if ( ( 0 !== $min_price && ( $variable_price >= $min_price ) ) ||
						( 0 === $min_price && is_numeric( $variable_price ) ) ) {
							$price = $variable_price;
					}
				}
			} elseif ( 0 !== $min_price && ( $custom_price >= $min_price ) ) {
				$price = $options['custom_price'];
			} elseif ( 0 === $min_price && is_numeric( $options['custom_price'] ) ) {
				$price = $options['custom_price'];
			}
		}
	}
	return $price;
}
add_filter( 'edd_cart_item_price', 'edd_cp_cart_item_price', 10, 3 );

/**
 * Filter the product item name in the cart.
 *
 * @since 1.5.8
 * @param string $title The cart item title.
 * @param int    $item_id The item ID.
 * @param array  $item The item array.
 *
 * @return string
 */
function edd_cp_modify_cart_item_name( $title, $item_id, $item ) {
	if ( ! empty( $item['options']['is_cp_bonus'] ) ) {
		return $title . ' ' . __( '*bonus item*', 'edd_cp' );
	}
	if ( edd_cp_has_custom_pricing( $item_id ) && ! empty( $item['options']['custom_price'] ) ) {
		return $title . ' ' . __( '(custom price)', 'edd_cp' );
	}

	return $title;
}
add_filter( 'edd_get_cart_item_name', 'edd_cp_modify_cart_item_name', 10, 3 );

/**
 * Filter price option name
 *
 * @param string $price_name The price name.
 * @param int    $download_id The download ID.
 * @param int    $payment_id The payment ID.
 * @param int    $price_id The price ID.
 *
 * @return string
 */
function edd_cp_get_price_option_name( $price_name, $download_id, $payment_id, $price_id ) {
	if ( $payment_id ) {
		$cart_items = edd_get_payment_meta_cart_details( $payment_id );
		$prices     = edd_get_variable_prices( $download_id );

		if ( $prices && is_array( $prices ) ) {
			if ( isset( $prices[ $price_id ] ) ) {
				return $price_name;
			}
		}

		if ( $cart_items ) {
			foreach ( $cart_items as $item ) {
				$item_id = $item['item_number']['id'];
				if ( $item_id == $download_id ) {
					$price_options = $item['item_number']['options'];
					if ( isset( $price_options['custom_price'] ) && edd_cp_has_custom_pricing( $item_id ) ) {
						$price_name = __( 'Custom price', 'edd_cp' );
					} elseif ( isset( $price_options['is_cp_bonus'] ) ) {
						$price_name = __( '*Bonus item*', 'edd_cp' );
					}

					// Fallback for buy now items with custom price.
					if ( 'direct' === edd_get_download_button_behavior( $download_id ) && edd_cp_has_custom_pricing( $item_id ) ) {
						$price_name = __( 'Custom price', 'edd_cp' );
					}
				}
			}
		}
	}

	return $price_name;
}
add_filter( 'edd_get_price_option_name', 'edd_cp_get_price_option_name', 10, 4 );

/**
 * Filter the purchase data before sending it to the direct gateway.
 *
 * @param array $purchase_data The purchase data.
 * @return array
 */
function edd_cp_straight_to_gateway_purchase_data( $purchase_data ) {
	$download_id  = isset( $_REQUEST['download_id'] ) ? absint( $_REQUEST['download_id'] ) : false;
	$custom_price = isset( $_REQUEST['edd_cp_price'] ) ? (float) $_REQUEST['edd_cp_price'] : false;
	if ( ! $download_id || ! $custom_price || ! edd_cp_has_custom_pricing( $download_id ) ) {
		return $purchase_data;
	}

	$min_price = get_post_meta( $download_id, 'edd_cp_min', true );
	if ( $min_price && $custom_price < $min_price ) {
		return $purchase_data;
	}

	$custom_price = edd_sanitize_amount( $custom_price );
	foreach ( $purchase_data['downloads'] as $d_key => $downloads ) {
		foreach ( $downloads['options'] as $o_key => $options ) {
			if ( is_array( $purchase_data['downloads'][ $d_key ]['options'][ $o_key ] ) ) {
				$purchase_data['downloads'][ $d_key ]['options'][ $o_key ]['amount'] = $custom_price;
			}

			if ( is_array( $purchase_data['cart_details'][0]['item_number']['options'][ $o_key ] ) ) {
				$purchase_data['cart_details'][0]['item_number']['options'][ $o_key ]['amount'] = $custom_price;
			}
		}
	}
	$purchase_data['cart_details'][0]['item_price'] = $custom_price;
	$purchase_data['cart_details'][0]['subtotal']   = $custom_price;
	$purchase_data['cart_details'][0]['name']       = $purchase_data['cart_details'][0]['name'] . ' - custom price';
	$purchase_data['subtotal']                      = $custom_price;
	$purchase_data['price']                         = $custom_price;

	return $purchase_data;
}
add_filter( 'edd_straight_to_gateway_purchase_data', 'edd_cp_straight_to_gateway_purchase_data' );

/**
 * Check if a custom priced product is removed, and remove the associated bonus item (if it exists)
 *
 * @param string $cart_key The cart key.
 * @param int    $item_id The item ID.
 */
function edd_cp_post_remove_from_cart( $cart_key, $item_id ) {

	if ( ! edd_cp_has_custom_pricing( $item_id ) ) {
		return;
	}

	$cart = edd_get_cart_contents();

	if ( ! empty( $cart ) ) {
		// Find the bonus item.
		foreach ( $cart as $key => $item ) {
			$has_bonus    = ! empty( $item['options']['is_cp_bonus'] );
			$item_id      = intval( $item_id );
			$bonus_parent = ! empty( $item['options']['cp_bonus_parent'] ) ? intval( $item['options']['cp_bonus_parent'] ) : false;

			if ( $has_bonus && $bonus_parent === $item_id ) {
				edd_remove_from_cart( $key );
			}
		}
	}
}
add_filter( 'edd_post_remove_from_cart', 'edd_cp_post_remove_from_cart', 10, 2 );

/**
 * Filter the purchase link defaults to allow custom button text if price is zero
 *
 * @param array $args The purchase link args.
 *
 * @return array
 */
function edd_cp_edd_purchase_link_args( $args ) {

	if ( ! edd_cp_has_custom_pricing( $args['download_id'] ) ) {
		return $args;
	}

	$button_text = get_post_meta( $args['download_id'], 'cp_button_text', true );

	if ( ! empty( $button_text ) ) {
		$args['price'] = false; // Prevents 'Free' from being added.
		$args['text']  = $button_text;
	}
	return $args;
}
add_filter( 'edd_purchase_link_args', 'edd_cp_edd_purchase_link_args' );

/**
 * Set is_free to false if the product is buy now.
 *
 * @param bool $is_free Whether the download is free.
 * @param int  $download_id The download ID.
 *
 * @return bool
 */
function edd_cp_is_free_download( $is_free, $download_id ) {

	// If the download is already not free, don't change it.
	if ( ! $is_free ) {
		return $is_free;
	}

	// If the download doesn't have custom pricing, don't change it.
	if ( ! edd_cp_has_custom_pricing( $download_id ) ) {
		return $is_free;
	}

	// If the download is not buy now, don't change it.
	if ( 'direct' !== edd_get_download_button_behavior( $download_id ) ) {
		return $is_free;
	}

	return false;
}
add_filter( 'edd_is_free_download', 'edd_cp_is_free_download', 10, 2 );

/**
 * Add custom price to variable prices array on gateway page.
 *
 * This fixes an issue when the price_id doesn't exist. The actual custom price is set later on edd_straight_to_gateway_purchase_data().
 *
 * @param array $prices The prices.
 * @param int   $download_id The download ID.
 *
 * @return array
 */
function edd_cp_get_variable_prices( $prices, $download_id ) {
	if ( edd_cp_has_custom_pricing( $download_id ) && doing_action( 'edd_straight_to_gateway' ) ) {
		$prices[-1] = array(
			'index'  => -1,
			'option' => 'Custom Price',
			'name'   => 'Custom Price',
			'amount' => 0,
		);
	}
	return $prices;
}
add_filter( 'edd_get_variable_prices', 'edd_cp_get_variable_prices', 10, 2 );
