<?php
/**
 * User functions
 *
 * @package     EDD\ContentRestriction\UserFunctions
 * @copyright   Copyright (c) 2013-2014, Pippin Williamson
 * @since       2.2.0
 */


// Exit if accessed directly
if( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Check to see if a user has access to a post/page
 *
 * @since       2.0
 * @param       int|bool $user_id Optional. The ID of the user to check. Default is false.
 * @param       array $restricted_to The array of downloads for a post/page
 * @param       int|false $post_id Optional. The ID of the object we are viewing. Default is false.
 * @return      array $return An array containing the status and optional message
 */
function edd_cr_user_can_access( $user_id = false, $restricted_to = array(), $post_id = false ) {

	$return = edd_cr_user_can_access_with_purchase( $user_id, $restricted_to, $post_id );

	// These filters remain only for backwards compatibility
	$return = array(
		'status' => apply_filters( 'edd_cr_user_can_access', $return['status'], $user_id, $restricted_to ),
		'message' => apply_filters( 'edd_cr_user_can_access_message', $return['message'], $user_id, $restricted_to )
	);

	// Filter the entire return array, so that the status and the message can be checked with the same filter call.
	$return = apply_filters( 'edd_cr_user_can_access_status_and_message', $return, $user_id, $restricted_to, $post_id  );

	return $return;
}


/**
 * Determine if a user has permission to view the currently viewed URL
 * Mainly for use in template files
 *
 * @since       2.1
 * @param       int $user_id The User ID to check (defaults to logged in user)
 * @param       int $post_id The Post ID to check access for (defaults to current post)
 * @return      bool If the current user has permission to view the current URL
 */
function edd_cr_user_has_access( $user_id = 0, $post_id = 0 ) {
	global $post;

	$user_id = empty( $user_id )                       ? get_current_user_id() : $user_id;
	$post_id = empty( $post_id ) && is_object( $post ) ? $post->ID             : $post_id;

	$has_access = true;

	if ( ! empty( $post_id ) ) {
		$is_post_restricted = edd_cr_is_restricted( $post_id );

		if ( $is_post_restricted ) {
			$user_has_access = edd_cr_user_can_access( $user_id, $is_post_restricted, $post_id );
			$has_access      = $user_has_access['status'] == false  ? false : true;
		}
	}

	return apply_filters( 'ecc_cr_user_has_access', $has_access );
}

/**
 * Check to see if a user has access to a post/page because of their user permissions
 *
 * @since       2.3
 * @param       int|bool $user_id Optional. The ID of the user to check. Default is false.
 * @param       array $restricted_to The array of downloads for a post/page
 * @param       int|false $post_id Optional. The ID of the object we are viewing. Default is false.
 * @return      array $return An array containing the status and optional message
 */
function edd_cr_user_can_access_with_permissions( $user_id = false, $restricted_to = array(), $post_id = false ) {

	$has_access = false;

	$return_array = array(
		'status' => $has_access,
		'message' => __( 'You do not have access because of your user permissions.', 'edd-cr' )
	);

	// If no user is given, use the current user
	if ( ! $user_id ) {
		$user_id = get_current_user_id();
	}

	// bbPress specific checks. Moderators can see everything
	if ( class_exists( 'bbPress' ) && current_user_can( 'moderate' ) ) {

		$return_array = array(
			'status' => true,
			'message' => __( 'You have access because of your bbPress user permissions.', 'edd-cr' )
		);

		return $return_array;
	}

	// Admins have full access
	if ( user_can( $user_id, 'manage_options' ) ) {

		$return_array = array(
			'status' => true,
			'message' => __( 'You have access because of your user permissions.', 'edd-cr' )
		);

		return $return_array;
	}

	// The post author can always access
	if ( $post_id ) {

		$post = get_post( $post_id );

		if ( $post->post_author == $user_id ) {
			$return_array = array(
				'status' => true,
				'message' => __( 'You have access because you are the post author.', 'edd-cr' )
			);

			return $return_array;
		}
	}

	// If this post is not restricted to anything
	if ( ! $restricted_to || ! is_array( $restricted_to ) ) {

		$return_array = array(
			'status' => true,
			'message' => __( 'You have access because this post is not restricted.', 'edd-cr' )
		);

		return $return_array;
	}

	foreach ( $restricted_to as $item => $data ) {

		if ( empty( $data['download'] ) ) {

			$return_array = array(
				'status' => true,
				'message' => __( 'You have access because this post is not restricted.', 'edd-cr' )
			);

			return $return_array;
		}

		// As a possible product author, you do not get access to the post if it is set to "any", because there's no specific author to use.
		if ( 'any' == $data['download'] ) {

			$return_array = array(
				'status' => false,
				'message' => __( 'You do not have access because of your user permissions.', 'edd-cr' )
			);

			return $return_array;
		}

		// Make sure the download ID setting is an integer
		$download_id = absint( $data['download'] );

		// The author of a download always has access
		if ( (int) get_post_field( 'post_author', $download_id ) === (int) $user_id ) {

			$return_array = array(
				'status' => true,
				'message' => __( 'You have access because you are the author of a product that is required to view this content.', 'edd-cr' )
			);

			return $return_array;

		}
	}

	return $return_array;
}

/**
 * Check to see if a user has access to a post/page because of a one-time purchase.
 *
 * @since       2.3
 * @param       int|bool $user_id Optional. The ID of the user to check. Default is false.
 * @param       array $restricted_to The array of downloads for a post/page
 * @param       int|false $post_id Optional. The ID of the object we are viewing. Default is false.
 * @return      array $return An array containing the status and optional message
 */
function edd_cr_user_can_access_with_purchase( $user_id = false, $restricted_to = array(), $post_id = false ) {

	// In the event that a single post ID was sent in, convert to an array.
	if ( ! is_array( $restricted_to ) ) {
		$restricted_to = array( $restricted_to );
	}

	$user_permissions_results = edd_cr_user_can_access_with_permissions( $user_id, $restricted_to, $post_id );

	// If the user has access because of their user permissions
	if ( $user_permissions_results['status'] ) {
		return $user_permissions_results;
	}

	// Set up some defaults
	$message          = __( 'This content is restricted to buyers.', 'edd-cr' );
	$has_access       = false;
	$restricted_count = count( $restricted_to );
	$products         = array();

	// If no user is given, use the current user
	if ( empty( $user_id ) ) {
		$user_id = get_current_user_id();
	}

	// If this post is not restricted to anything
	if ( ! $restricted_to || ! is_array( $restricted_to ) ) {
		$has_access = true;
	}

	// Loop through each product this content is restricted to
	foreach ( $restricted_to as $item => $data ) {

		// If there is no product in this restricted_to item for some strange reason
		if ( empty( $data['download'] ) ) {
			$has_access = true;
		}

		// If restricted to any product
		if ( 'any' === $data['download'] ) {

			// If this user has purchased something, they have access
			if ( edd_has_purchases( $user_id ) ) {
				$has_access = true;
				break;
			}
		}

		// Check for variable prices
		if ( ! $has_access ) {
			if ( edd_has_variable_prices( $data['download'] ) ) {
				if ( strtolower( $data['price_id'] ) !== 'all' && ! empty( $data['price_id'] ) ) {
					$products[] = '<a href="' . get_permalink( $data['download'] ) . '">' . get_the_title( $data['download'] ) . ' - ' . edd_get_price_option_name( $data['download'], $data['price_id'] ) . '</a>';

					if ( edd_has_user_purchased( $user_id, $data['download'], $data['price_id'] ) ) {
						$has_access = true;
					}
				} else {
					$products[] = '<a href="' . get_permalink( $data['download'] ) . '">' . get_the_title( $data['download'] ) . '</a>';

					if ( edd_has_user_purchased( $user_id, $data['download'] ) ) {
						$has_access = true;
					}
				}
			} else {
				$products[] = '<a href="' . get_permalink( $data['download'] ) . '">' . get_the_title( $data['download'] ) . '</a>';

				if ( ! empty( $user_id ) && edd_has_user_purchased( $user_id, $data['download'] ) ) {
					$has_access = true;
				}
			}
		}

		if ( ! $has_access && ! empty( $user_id ) && edd_get_option( 'edd_content_restriction_include_bundled_products', false ) ) {
			$purchased = edd_get_users_purchased_products( $user_id );

			foreach ( $purchased as $item ) {
				if ( edd_is_bundled_product( $item->ID ) ) {
					$bundled = (array) edd_get_bundled_products( $item->ID );

					if ( in_array( $data['download'], $bundled ) ) {
						$has_access = true;
						break;
					}
				}
			}
		}
	}

	if ( $restricted_count > 1 ) {
		$message      = edd_cr_get_multi_restriction_message();
		$product_list = '';

		if ( ! empty( $products ) ) {
			$product_list .= '<ul>';

			foreach ( $products as $id => $product ) {
				$product_list .= '<li>' . $product . '</li>';
			}

			$product_list .= '</ul>';
		}

		$message = str_replace( '{product_names}', $product_list, $message );
	} else {
		if ( 'any' === $data['download'] ) {
			$message = edd_cr_get_any_restriction_message();
		} else {
			$message = edd_cr_get_single_restriction_message();
			$message = str_replace( '{product_name}', $products[0], $message );
		}
	}

	// Override message if per-content message is defined
	$content_message = get_post_meta( $post_id, '_edd_cr_restricted_message', true );
	$message         = ( $content_message && $content_message !== '' ? $content_message : $message );

	if ( ! isset( $message ) ) {
		$message = __( 'This content is restricted to buyers.', 'edd-cr' );
	}

	// Set up the return array.
	$return = array(
		'status' => $has_access,
		'message' => $message
	);

	return $return;
}
