<?php
/**
 * Add Helper Functions and Template Overrides
 *
 * @package     EDD\ContentRestriction\Functions
 * @copyright   Copyright (c) 2013-2014, Pippin Williamson
 * @since       1.0.0
 */


// Exit if accessed directly
if( ! defined( 'ABSPATH' ) ) {
	exit;
}


/**
 * Get posts/pages restricted to the purchased files
 *
 * @since       1.3.0
 * @param       int $payment_id The ID of this payment
 * @return      array $posts The list of accessible posts
 */
function edd_cr_get_restricted_pages( $payment_id = 0 ) {
	$posts = array();

	if ( empty( $payment_id ) ) {
		return $posts;
	}

	$payment = edd_get_payment( $payment_id );

	if ( empty( $payment ) ) {
		return $posts;
	}

	// Loop through each item purchased in the cart.
	foreach ( $payment->cart_details as $purchased_product ) {

		$purchased_product_id = $purchased_product['id'];
		$purchased_product_price_id = false;

		// If this purchased item had a variable price that was purchased.
		if ( isset( $purchased_product['item_number']['options']['price_id'] ) ) {
			$purchased_product_price_id = $purchased_product['item_number']['options']['price_id'];
		}

		// Check if this product protects any posts/pages.
		$posts_protected_by_product = get_post_meta( $purchased_product_id, '_edd_cr_protected_post' );

		// If no posts are protected by this product, skip it and check the next purchased product instead.
		if ( ! $posts_protected_by_product ) {
			continue;
		}

		// Skip revisions and unpublished posts.
		foreach ( $posts_protected_by_product as $key => $post_id ) {
			if ( 'revision' === get_post_type( $post_id ) || 'publish' !== get_post_status( $post_id ) ) {
				unset( $posts_protected_by_product[ $key ] );
			}
		}

		// Loop through each post protected by this product.
		foreach ( $posts_protected_by_product as $protected_post_id ){

			// Attached to the protected post is the _edd_cr_restricted_to array, which tells us which products protect this post.
			$restricted_to = get_post_meta( $protected_post_id, '_edd_cr_restricted_to', true );

			// If this protected post does not have anything saved to the _edd_cr_restricted_to meta, skip this protected post.
			if ( empty( $restricted_to ) ) {
				continue;
			}

			// Loop through each product which protects this post.
			foreach ( $restricted_to as $restricted_to_product ) {

				// Check if the product required is the product that was purchased.
				if ( (int) $restricted_to_product['download'] !== (int) $purchased_product_id ) {
					continue;
				}

				// Check if the product required has a specific variable price that is required.
				if ( isset( $restricted_to_product['price_id'] ) && 'all' !== $restricted_to_product['price_id'] ) {

					// Check if that variable price matches the one purchased.
					if ( $purchased_product_price_id == $restricted_to_product['price_id'] ) {

						// Add the product with the variable price to the list.
						$posts[ $purchased_product_id . ':' . $purchased_product_price_id ][ $protected_post_id ] = get_the_title( $protected_post_id );

					}

				} else {

					// If no variable price is required or 'all' is defined as the variable price ID.
					$posts[ $purchased_product_id ][ $protected_post_id ] = get_the_title( $protected_post_id );

				}
			}

		}

	}

	return $posts;
}
