<?php
/**
 * Software Licensing Integration
 *
 * @package     EDD\ContentRestriction\Integrations\SoftwareLicensing
 * @copyright   Copyright (c) 2013-2016, Pippin Williamson
 * @since       2.2.0
 */


// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


/**
 * Add metabox field
 *
 * @since       2.2.0
 * @param       int $post_id The ID of the post we are editing
 * @return      void
 */
function edd_cr_add_sl_metabox_field( $post_id ) {

	$active_license = get_post_meta( $post_id, '_edd_cr_sl_require_active_license', true );
	echo '<p>';
		echo '<label for="edd_cr_sl_require_active_license">';
			echo '<input type="checkbox" name="edd_cr_sl_require_active_license" id="edd_cr_sl_require_active_license" value="1"' . checked( '1', $active_license, false ) . '/>&nbsp;';
			echo __( 'Require a valid license key?', 'edd-sl' );
			echo '<span class="edd-help-tip dashicons dashicons-editor-help" alt="f223" title="<strong>' . __( 'Require a valid license key?', 'edd-cr' ) . '</strong> ' . sprintf( __( 'Only customers with a valid license key will be able to view the content. This setting is only applied if the selected %s has licensing enabled.', 'edd-sl' ), edd_get_label_singular( true ) ) . '"></span>';
		echo '</label>';
	echo '</p>';
}
add_action( 'edd_cr_restricted_table_before', 'edd_cr_add_sl_metabox_field', 10, 1 );


/**
 * Update data on save
 *
 * @since       2.2.0
 * @param       int $post_id The ID of the post we are editing
 * @param       array $data The submitted data
 * @return      void
 */
function edd_cr_sl_metabox_save( $post_id, $data ) {

	if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		return;
	}

	if ( isset( $data['edd_cr_sl_require_active_license'] ) ) {
		update_post_meta( $post_id, '_edd_cr_sl_require_active_license', '1' );
	} else {
		delete_post_meta( $post_id, '_edd_cr_sl_require_active_license' );
	}
}
add_action( 'edd_cr_save_meta_data', 'edd_cr_sl_metabox_save', 10, 2 );


/**
 * Override access depending on license status
 *
 * @since       2.2.0
 *
 * @param       bool $has_access Whether or not the user has access
 * @param       int $user_id The ID of the user
 * @param       array $restricted_to The array of downloads for a post/page
 * @return      bool $has_access The updated access condition for the user
 */
function edd_cr_user_has_license( $has_access, $user_id, $restricted_to ) {

	// Only proceed if the setting is enabled.
	if ( ! get_post_meta( get_the_ID(), '_edd_cr_sl_require_active_license', true ) ) {
		return $has_access;
	}

	// There are no products which could grant access.
	if ( ! $restricted_to || ! is_array( $restricted_to ) ) {
		return $has_access;
	}

	$user_licenses = edd_software_licensing()->get_license_keys_of_user( $user_id );

	// Only proceed if the user has actually purchased a license
	if ( empty( $user_licenses ) ) {
		return $has_access;
	}

	foreach ( $user_licenses as $key => $license ) {
		// Expired/revoked licenses don't count.
		if ( in_array( $license->status, array( 'expired', 'revoked', 'disabled' ), true ) ) {
			unset( $user_licenses[ $key ] );
			continue;
		}

		// If the licensed download is a draft, it doesn't count.
		$download_status = get_post_status( $license->download_id );
		if ( 'draft' === $download_status ) {
			unset( $user_licenses[ $key ] );
		}
	}

	// Reset access to false.
	$has_access = false;

	// All of the licenses had an unacceptable status.
	if ( empty( $user_licenses ) ) {
		return $has_access;
	}

	foreach ( $restricted_to as $restricted_item ) {

		$download = new EDD_SL_Download( $restricted_item['download'] );

		// No need to check if licensing is not enabled on the download
		if ( ! $download->licensing_enabled() ) {
			continue;
		}

		foreach ( $user_licenses as $license ) {

			$license_download = (int) $license->download_id;

			// License does not belong to the product we're checking
			if ( $license_download !== (int) $restricted_item['download'] ) {
				continue;
			}

			// The license matches; variable downloads need to also check the price ID.
			if ( $download->has_variable_prices() ) {
				if ( ( 'all' === strtolower( $restricted_item['price_id'] ) || $license->price_id === (int) $restricted_item['price_id'] ) ) {
					$has_access = true;
					break;
				}
			} else {
				$has_access = true;
				break;
			}
		}

		// Stop checking restricted products if a qualifying license was found and access was granted.
		if ( $has_access ) {
			break;
		}
	}

	return $has_access;
}
add_filter( 'edd_cr_user_can_access', 'edd_cr_user_has_license', 10, 3 );
