<?php
/**
 * Redirect Actions
 *
 * @since 1.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;


/**
 * Sets up and stores a new redirect
 *
 * @since 1.0
 * @param array $data Redirect data.
 * @uses edd_store_redirect()
 * @return void
 */
function edd_csr_add_redirect( $data ) {
	if ( ! current_user_can( 'manage_shop_settings' ) ) {
		return;
	}
	if ( empty( $data['edd-redirect-nonce'] ) || ! wp_verify_nonce( $data['edd-redirect-nonce'], 'edd_redirect_nonce' ) ) {
		return;
	}
	// Setup the redirect details.
	$posted = array();

	foreach ( $data as $key => $value ) {
		if ( 'edd-redirect-nonce' !== $key && 'edd-action' !== $key && 'edd-redirect' !== $key ) {
			if ( is_string( $value ) || is_int( $value ) ) {
				$posted[ $key ] = strip_tags( addslashes( $value ) );
			} elseif ( is_array( $value ) ) {
				$posted[ $key ] = array_map( 'absint', $value );
			}
		}
	}

	if ( empty( $posted['download'] ) ) {
		edd_csr_process_redirects_change_redirect(
			'no_download_selected',
			array(
				'edd-action' => 'add_redirect',
			)
		);
	}

	if ( empty( $posted['page'] ) ) {
		edd_csr_process_redirects_change_redirect(
			'no_page_selected',
			array(
				'edd-action' => 'add_redirect',
			)
		);
	}

	// Attempt to store the redirect and redirect with a message.
	edd_csr_process_redirects_change_redirect(
		edd_csr_store_redirect( $posted )
			? 'redirect_added'
			: 'redirect_add_failed'
	);
}
add_action( 'edd_add_redirect', 'edd_csr_add_redirect' );

/**
 * Saves an edited redirect
 *
 * @since 1.0
 * @param array $data Redirect data.
 * @return void
 */
function edd_csr_edit_redirect( $data ) {
	if ( ! current_user_can( 'manage_shop_settings' ) ) {
		return;
	}

	if ( empty( $data['edd-redirect-nonce'] ) || ! wp_verify_nonce( $data['edd-redirect-nonce'], 'edd_redirect_nonce' ) ) {
		return;
	}

	// Setup the redirect details.
	$redirect = array();

	foreach ( $data as $key => $value ) {
		if ( 'edd-redirect-nonce' !== $key && 'edd-action' !== $key && 'redirect-id' !== $key && 'edd-redirect' !== $key ) {
			if ( is_string( $value ) || is_int( $value ) ) {
				$redirect[ $key ] = strip_tags( addslashes( $value ) );
			} elseif ( is_array( $value ) ) {
				$redirect[ $key ] = array_map( 'absint', $value );
			}
		}
	}

	if ( empty( $redirect['download'] ) ) {
		edd_csr_process_redirects_change_redirect(
			'no_download_selected',
			array(
				'edd-action'  => 'edit_redirect',
				'redirect-id' => $data['redirect-id'],
			)
		);
	}

	if ( empty( $redirect['page'] ) ) {
		edd_csr_process_redirects_change_redirect(
			'no_page_selected',
			array(
				'edd-action'  => 'edit_redirect',
				'redirect-id' => $data['redirect-id'],
			)
		);
	}

	// Attempt to update the redirect and redirect with a message.
	edd_csr_process_redirects_change_redirect(
		edd_csr_store_redirect( $redirect, $data['redirect-id'] )
			? 'redirect_updated'
			: 'redirect_update_failed'
	);
}
add_action( 'edd_edit_redirect', 'edd_csr_edit_redirect' );

/**
 * Listens for when a redirect delete button is clicked and deletes the
 * redirect
 *
 * @since 1.0
 * @param array $data Redirect data.
 * @uses edd_remove_redirect()
 * @return void
 */
function edd_csr_delete_redirect( $data ) {

	if ( ! current_user_can( 'manage_shop_settings' ) ) {
		return;
	}

	// Check if there was a nonce passed in.
	if ( ! isset( $_GET['edd-delete-redirect-nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['edd-delete-redirect-nonce'] ) ), 'edd_delete_redirect_nonce' ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		wp_die( esc_html_e( 'Something went wrong.', 'edd-csr' ), esc_html_e( 'Error', 'edd-csr' ) );
	}

	$redirect_id = $data['redirect'];

	// Delete the redirect and redirect with a message.
	edd_csr_process_redirects_change_redirect(
		edd_csr_remove_redirect( $redirect_id )
			? 'delete_redirect'
			: 'delete_redirect_failed'
	);
}
add_action( 'edd_delete_redirect', 'edd_csr_delete_redirect' );

/**
 * Activates Redirect Code
 *
 * Sets a redirect's status to active
 *
 * @since 1.0
 * @param array $data Redirect data.
 * @uses edd_csr_update_redirect_status()
 * @return void
 */
function edd_csr_activate_redirect( $data ) {
	if ( ! current_user_can( 'manage_shop_settings' ) ) {
		return;
	}
	// Check if there was a nonce passed in.
	if ( ! isset( $_GET['edd-activate-redirect-nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['edd-activate-redirect-nonce'] ) ), 'edd_activate_redirect_nonce' ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		wp_die( esc_html_e( 'Something went wrong.', 'edd-csr' ), esc_html_e( 'Error', 'edd-csr' ) );
	}

	$id = $data['redirect'];

	// Activate the redirect and send the user back to the list with a message.
	edd_csr_process_redirects_change_redirect(
		edd_csr_update_redirect_status( $id, 'edd-csr-active' )
			? 'activate_redirect'
			: 'activate_redirect_failed'
	);
}
add_action( 'edd_activate_redirect', 'edd_csr_activate_redirect' );

/**
 * Deactivate Redirect
 *
 * Sets a redirect's status to deactivated.
 *
 * @since 1.0
 * @param array $data Redirect data.
 * @uses edd_csr_update_redirect_status()
 * @return void
 */
function edd_csr_deactivate_redirect( $data ) {
	if ( ! current_user_can( 'manage_shop_settings' ) ) {
		return;
	}
	// Check if there was a nonce passed in.
	if ( ! isset( $_GET['edd-deactivate-redirect-nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['edd-deactivate-redirect-nonce'] ) ), 'edd_deactivate_redirect_nonce' ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		wp_die( esc_html_e( 'Something went wrong.', 'edd-csr' ), esc_html_e( 'Error', 'edd-csr' ) );
	}

	$id = $data['redirect'];
	// Activate the redirect and send the user back to the list with a message.
	edd_csr_process_redirects_change_redirect(
		edd_csr_update_redirect_status( $id, 'edd-csr-inactive' )
			? 'deactivate_redirect'
			: 'deactivate_redirect_failed'
	);
}
add_action( 'edd_deactivate_redirect', 'edd_csr_deactivate_redirect' );

/**
 * Process Redirects Change Redirect
 *
 * When modifying or adding a redirect, this function will redirect the user to the main redirects list table with a message.
 *
 * @since 1.2.0
 *
 * @param string $message_id The message ID to display.
 * @param array  $options    Additional options to pass to the redirect URL.
 *
 * @return void
 */
function edd_csr_process_redirects_change_redirect( $message_id = '', $options = array() ) {
	$args = array(
		'page'        => 'edd-redirects',
		'edd-message' => $message_id,
	);

	if ( ! empty( $options ) ) {
		$args = array_merge( $args, $options );
	}

	edd_redirect( edd_get_admin_url( $args ) );
}
