<?php
/**
 * Redirects Table Class
 * Based largely on existing code from the Easy Digital Downloads plugin
 *
 * @since 1.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * EDD_CSR_Table Class
 *
 * Renders the Redirects table on the Conditional Redirects page
 *
 * @since 1.0
 */
class EDD_CSR_Table extends WP_List_Table {
	/**
	 * Number of results to show per page
	 *
	 * @var string
	 * @since 1.0
	 */
	public $per_page = 30;

	/**
	 *
	 * Total number of redirects
	 *
	 * @var string
	 * @since 1.0
	 */
	public $total_count;

	/**
	 * Active number of redirects
	 *
	 * @var string
	 * @since 1.0
	 */
	public $active_count;

	/**
	 * Inactive number of redirects
	 *
	 * @var string
	 * @since 1.0
	 */
	public $inactive_count;

	/**
	 * Parameters passed into the table
	 *
	 * @var array
	 * @since 1.0
	 */
	public $params;

	/**
	 * Get things started
	 *
	 * @since 1.0
	 * @uses EDD_CSR_Table::get_redirect_counts()
	 * @see WP_List_Table::__construct()
	 * @return void
	 */
	public function __construct() {
		parent::__construct(
			array(
				'singular' => 'redirect',
				'plural'   => 'redirects',
				'ajax'     => false,
			)
		);

		$this->get_redirect_counts();
	}

	/**
	 * Show the search field
	 *
	 * @since 1.0
	 *
	 * @param string $text Label for the search box.
	 * @param string $input_id ID of the search box.
	 *
	 * @return void
	 */
	public function search_box( $text, $input_id ) {
		if ( empty( $this->get_param( 's' ) ) && ! $this->has_items() ) {
			return;
		}

		$input_id = sanitize_text_field( $input_id . '-search-input' );
		?>
		<input type="hidden" name="orderby" value="<?php echo esc_attr( $this->get_param( 'orderby' ) ); ?>" />
		<input type="hidden" name="order" value="<?php echo esc_attr( $this->get_param( 'order' ) ); ?>" />
		<p class="search-box">
			<label class="screen-reader-text" for="<?php echo esc_attr( $input_id ); ?>"><?php sanitize_text_field( $text ); ?>:</label>
			<input type="search" id="<?php echo esc_attr( $input_id ); ?>" name="s" value="<?php _admin_search_query(); ?>" />
			<?php submit_button( $text, 'button', false, false, array( 'ID' => 'search-submit' ) ); ?>
		</p>
		<?php
	}

	/**
	 * Retrieve the view types
	 *
	 * @since 1.0
	 * @return array $views All the views available
	 */
	public function get_views() {
		$base = admin_url( 'edit.php?post_type=download&page=edd-redirects' );

		$current        = $this->get_param( 'status' );
		$total_count    = '&nbsp;<span class="count">(' . $this->total_count . ')</span>';
		$active_count   = '&nbsp;<span class="count">(' . $this->active_count . ')</span>';
		$inactive_count = '&nbsp;<span class="count">(' . $this->inactive_count . ')</span>';

		$views = array(
			'all'      => sprintf( '<a href="%s"%s>%s</a>', esc_url( remove_query_arg( 'status', $base ) ), ( 'all' === $current || empty( $current ) ) ? ' class="current"' : '', __( 'All', 'edd-csr' ) . $total_count ),
			'active'   => sprintf( '<a href="%s"%s>%s</a>', esc_url( add_query_arg( 'status', 'edd-csr-active', $base ) ), 'edd-csr-active' === $current ? ' class="current"' : '', __( 'Active', 'edd-csr' ) . $active_count ),
			'inactive' => sprintf( '<a href="%s"%s>%s</a>', esc_url( add_query_arg( 'status', 'edd-csr-inactive', $base ) ), 'edd-csr-inactive' === $current ? ' class="current"' : '', __( 'Inactive', 'edd-csr' ) . $inactive_count ),
		);

		return $views;
	}

	/**
	 * Retrieve the table columns
	 *
	 * @since 1.0
	 * @return array $columns Array of all the list table columns
	 */
	public function get_columns() {
		$columns = array(
			'cb'       => '<input type="checkbox" />',
			// translators: %s is the singular label for a download.
			'download' => sprintf( esc_html__( '%s Purchased', 'edd-csr' ), edd_get_label_singular() ),
			'status'   => __( 'Status', 'edd-csr' ),
			'redirect' => __( 'Redirect', 'edd-csr' ),
		);

		return $columns;
	}

	/**
	 * Retrieve the table's sortable columns
	 *
	 * @since 1.0
	 * @return array Array of all the sortable columns
	 */
	public function get_sortable_columns() {
		return array(
			'ID'       => array( 'ID', true ),
			'download' => array( 'download', false ),
		);
	}

	/**
	 * This function renders most of the columns in the list table.
	 *
	 * @since 1.0
	 *
	 * @param array  $item Contains all the data of the redirect.
	 * @param string $column_name The name of the column.
	 *
	 * @return string Column Name
	 */
	public function column_default( $item, $column_name ) {
		switch ( $column_name ) {
			default:
				return $item[ $column_name ];
		}
	}



	/**
	 * Render the Download Column
	 *
	 * @since 1.0
	 * @param array $item Contains all the data of the redirect.
	 * @return string Data shown in the Name column
	 */
	public function column_download( $item ) {

		$redirect    = get_post( $item['ID'] );
		$row_actions = array();

		$base_url = admin_url( 'edit.php?post_type=download&page=edd-redirects' );

		$title = sprintf(
			'<a href="%s" class="row-title">%s</a>',
			esc_url(
				wp_nonce_url(
					add_query_arg(
						array(
							'edd-action' => 'edit_redirect',
							'redirect'   => $redirect->ID,
						),
						$base_url
					),
					'edd_edit_redirect_nonce',
					'edd-edit-redirect-nonce'
				)
			),
			get_the_title( $item['download'] )
		);

		$row_actions['edit'] = '<a href="' . esc_url(
			wp_nonce_url(
				add_query_arg(
					array(
						'edd-action' => 'edit_redirect',
						'redirect'   => $redirect->ID,
					),
					$base_url
				),
				'edd_edit_redirect_nonce',
				'edd-edit-redirect-nonce'
			)
		) . '">' . __( 'Edit', 'edd-csr' ) . '</a>';

		if ( 'edd-csr-active' === strtolower( $item['status'] ) ) {
			$row_actions['deactivate'] = '<a href="' . esc_url(
				wp_nonce_url(
					add_query_arg(
						array(
							'edd-action' => 'deactivate_redirect',
							'redirect'   => $redirect->ID,
						),
						$base_url
					),
					'edd_deactivate_redirect_nonce',
					'edd-deactivate-redirect-nonce'
				)
			) . '">' . __( 'Deactivate', 'edd-csr' ) . '</a>';
		} else {
			$row_actions['activate'] = '<a href="' . esc_url(
				wp_nonce_url(
					add_query_arg(
						array(
							'edd-action' => 'activate_redirect',
							'redirect'   => $redirect->ID,
						),
						$base_url
					),
					'edd_activate_redirect_nonce',
					'edd-activate-redirect-nonce'
				)
			) . '">' . __( 'Activate', 'edd-csr' ) . '</a>';
		}

		$row_actions['delete'] = '<a href="' . esc_url(
			wp_nonce_url(
				add_query_arg(
					array(
						'edd-action' => 'delete_redirect',
						'redirect'   => $redirect->ID,
					),
					$base_url
				),
				'edd_delete_redirect_nonce',
				'edd-delete-redirect-nonce'
			)
		) . '">' . __( 'Delete', 'edd-csr' ) . '</a>';

		$row_actions = apply_filters( 'edd_csr_redirect_row_actions', $row_actions, $redirect );

		return $title . $this->row_actions( $row_actions );
	}

	/**
	 * Render the Status Column
	 *
	 * @since 1.2.0
	 * @param array $item Contains all the data of the redirect.
	 * @return string Data shown in the Status column
	 */
	public function column_status( $item ) {
		$status      = 'edd-csr-active' === $item['status'] ? 'active' : 'inactive';
		$post_status = get_post_status_object( $item['status'] );

		$status_badge = new EDD\Utils\StatusBadge(
			array(
				'status' => $status,
				'label'  => $post_status->label,
			)
		);

		return $status_badge->get();
	}

	/**
	 * Render the checkbox column
	 *
	 * @since 1.0
	 * @param array $item Contains all the data for the checkbox column.
	 * @return string Displays a checkbox
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			/*$1%s*/ 'redirect',
			/*$2%s*/ $item['ID']
		);
	}

	/**
	 * Retrieve the bulk actions
	 *
	 * @since 1.0
	 * @return array $actions Array of the bulk actions
	 */
	public function get_bulk_actions() {
		$actions = array(
			'delete' => __( 'Delete', 'edd-csr' ),
		);

		return $actions;
	}

	/**
	 * Message to be displayed when there are no items
	 *
	 * @since 1.0
	 */
	public function no_items() {
		esc_html_e( 'No redirects found.', 'edd-csr' );
	}


	/**
	 * Process the bulk actions
	 *
	 * @since 1.0
	 * @return void
	 */
	public function process_bulk_action() {
		$ids = $this->get_param( 'redirect' );

		if ( empty( $ids ) ) {
			return;
		}

		foreach ( $ids as $id ) {
			if ( 'delete' === $this->current_action() ) {
				edd_csr_remove_redirect( $id );
			}
		}
	}

	/**
	 * Retrieve the redirect counts
	 *
	 * @since 1.0
	 * @return void
	 */
	public function get_redirect_counts() {
		$redirect_count       = wp_count_posts( 'edd_redirect' );

		$active   = 'edd-csr-active';
		$inactive = 'edd-csr-inactive';

		$this->active_count   = isset( $redirect_count->$active ) ? $redirect_count->$active : 0;
		$this->inactive_count = isset( $redirect_count->$inactive ) ? $redirect_count->$inactive : 0;
		$this->total_count    = $this->active_count + $this->inactive_count;
	}

	/**
	 * Retrieve all the data for all the redirects
	 *
	 * @since 1.0
	 * @return array $redirect_data Array of all the data for the redirects
	 */
	public function redirect_data() {
		$redirect_data = array();

		$redirects = edd_csr_get_redirects(
			array(
				'posts_per_page' => $this->per_page,
				'paged'          => $this->get_param( 'paged' ),
				'orderby'        => $this->get_param( 'orderby' ),
				'order'          => $this->get_param( 'order' ),
				'post_status'    => $this->get_param( 'status' ),
				'meta_key'       => $this->get_param( 'meta_key' ),
				's'              => $this->get_param( 's' ),
			)
		);

		if ( $redirects ) {
			foreach ( $redirects as $redirect ) {

				$redirect_to = edd_csr_get_redirect_page( $redirect->ID ) ? edd_csr_get_redirect_page( $redirect->ID ) : '';

				$redirect_data[] = array(
					'ID'       => $redirect->ID,
					'download' => edd_csr_get_redirect_download( $redirect->ID ),
					'redirect' => get_the_title( $redirect_to ),
					'status'   => $redirect->post_status,
				);
			}
		}

		return $redirect_data;
	}

	/**
	 * Setup the final data for the table
	 *
	 * @since 1.0
	 * @uses EDD_CSR_Table::get_columns()
	 * @uses EDD_CSR_Table::get_sortable_columns()
	 * @uses EDD_CSR_Table::process_bulk_action()
	 * @uses EDD_CSR_Table::redirect_data()
	 * @uses WP_List_Table::get_pagenum()
	 * @uses WP_List_Table::set_pagination_args()
	 * @return void
	 */
	public function prepare_items() {
		$per_page = $this->per_page;

		$columns = $this->get_columns();

		$hidden = array();

		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array( $columns, $hidden, $sortable );

		$this->process_bulk_action();

		$data = $this->redirect_data();

		$status = $this->get_param( 'status' );

		switch ( $status ) {
			case 'edd-csr-active':
				$total_items = $this->active_count;
				break;
			case 'edd-csr-inactive':
				$total_items = $this->inactive_count;
				break;
			default:
				$total_items = $this->total_count;
				break;
		}

		$this->items = $data;

		$this->set_pagination_args(
			array(
				'total_items' => $total_items,
				'per_page'    => $per_page,
				'total_pages' => ceil( $total_items / $per_page ),
			)
		);
	}

	/**
	 * Get the value of a parameter passed into the table.
	 *
	 * @since 1.2.0
	 *
	 * @param string $param The parameter to retrieve.
	 *
	 * @return mixed The value of the parameter or false if it doesn't exist.
	 */
	private function get_param( $param = '' ) {
		if ( ! is_null( $this->params ) && isset( $this->params[ $param ] ) ) {
			return $this->params[ $param ];
		}

		$this->parse_params();

		if ( isset( $this->params[ $param ] ) ) {
			return $this->params[ $param ];
		}

		return false;
	}

	/**
	 * Parse the parameters passed into the table.
	 *
	 * @since 1.2.0
	 *
	 * @return void
	 */
	private function parse_params() {
		$params = array();

		$params['status']   = isset( $_GET['status'] ) ? sanitize_text_field( wp_unslash( $_GET['status'] ) ) : array( 'edd-csr-active', 'edd-csr-inactive' ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$params['orderby']  = isset( $_GET['orderby'] ) ? sanitize_text_field( wp_unslash( $_GET['orderby'] ) ) : 'ID'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$params['order']    = isset( $_GET['order'] ) ? sanitize_text_field( wp_unslash( $_GET['order'] ) ) : 'DESC'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$params['paged']    = isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$params['s']        = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : null; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$params['meta_key'] = isset( $_GET['meta_key'] ) ? sanitize_text_field( wp_unslash( $_GET['meta_key'] ) ) : null; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		// Handle the redirect IDs parameter a little differently to sanitize the entire array.
		$params['redirect'] = isset( $_GET['redirect'] ) ? sanitize_text_field( wp_unslash( $_GET['redirect'] ) ) : array(); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! is_array( $params['redirect'] ) ) {
			$params['redirect'] = array( $params['redirect'] );
		}

		$params['redirect'] = array_map( 'absint', $params['redirect'] );

		$this->params = $params;
	}
}
