<?php
/**
 * Base Plugin Class
 *
 * @package   edd-conditional-success-redirects
 * @copyright Copyright (c) 2021, Easy Digital Downloads
 * @license   GPL2+
 */
class EDD_Conditional_Success_Redirects {

	/**
	 * The Singleton Instance
	 *
	 * @var EDD_Conditional_Success_Redirects
	 */
	private static $instance;

	/**
	 * The version of the plugin.
	 *
	 * @var string Current version of the plugin.
	 */
	public $version;

	/**
	 * The main plugin file.
	 *
	 * @var string Path to the main plugin file.
	 */
	public $file;

	/**
	 * The plugin basename.
	 *
	 * @var string Plugin basename.
	 */
	public $basename;

	/**
	 * The plugin directory.
	 *
	 * @var string Path to the plugin's directory.
	 */
	public $plugin_dir;

	/**
	 * The plugin URL.
	 *
	 * @var string URL to the plugin's directory.
	 */
	public $plugin_url;

	/**
	 * The plugin includes directory.
	 *
	 * @var string Path to the plugin's `includes/` directory.
	 */
	public $includes_dir;

	/**
	 * The plugin includes URL.
	 *
	 * @var string URL to the plugin's `includes/` directory.
	 */
	public $includes_url;

	/**
	 * Main Instance
	 *
	 * Ensures that only one instance exists in memory at any one
	 * time. Also prevents needing to define globals all over the place.
	 *
	 * @since 1.0
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof EDD_Conditional_Success_Redirects ) ) {
			self::$instance = new EDD_Conditional_Success_Redirects();
			self::$instance->setup_globals();
			self::$instance->includes();
			self::$instance->setup_actions();
			self::$instance->licensing();
			self::$instance->load_textdomain();
		}

		return self::$instance;
	}

	/**
	 * Globals
	 *
	 * @since 1.0
	 *
	 * @return void
	 */
	private function setup_globals() {

		$this->version = EDD_CONDITIONAL_SUCCESS_REDIRECTS_VERSION;

		// paths.
		$this->file       = EDD_CONDITIONAL_SUCCESS_REDIRECTS_FILE;
		$this->basename   = apply_filters( 'edd_csr_plugin_basenname', plugin_basename( $this->file ) );
		$this->plugin_dir = apply_filters( 'edd_csr_plugin_dir_path', plugin_dir_path( $this->file ) );
		$this->plugin_url = apply_filters( 'edd_csr_plugin_dir_url', plugin_dir_url( $this->file ) );

		// includes.
		$this->includes_dir = apply_filters( 'edd_csr_includes_dir', trailingslashit( $this->plugin_dir . 'includes' ) );
		$this->includes_url = apply_filters( 'edd_csr_includes_url', trailingslashit( $this->plugin_url . 'includes' ) );
	}

	/**
	 * Setup the default hooks and actions
	 *
	 * @since 1.0
	 *
	 * @return void
	 */
	private function setup_actions() {
		// Add sub-menu page.
		add_action( 'admin_menu', array( $this, 'add_redirect_options' ), 10 );

		// Alter the EDD Settings Page Title.
		add_filter( 'edd_settings_page_title', array( $this, 'settings_page_title' ), 10, 2 );

		do_action( 'edd_csr_setup_actions' );
	}

	/**
	 * Licensing
	 *
	 * @since 1.0
	 */
	private function licensing() {
		add_action(
			'edd_extension_license_init',
			function ( \EDD\Extensions\ExtensionRegistry $registry ) {
				$registry->addExtension( $this->file, 'Conditional Success Redirects', 84640, $this->version );
			}
		);
	}

	/**
	 * Loads the plugin language files
	 *
	 * @access public
	 * @since 1.1
	 * @return void
	 */
	public function load_textdomain() {

		// Set filter for plugin's languages directory.
		$lang_dir = dirname( plugin_basename( __FILE__ ) ) . '/languages/';
		$lang_dir = apply_filters( 'edd_csr_languages_directory', $lang_dir );

		// Traditional WordPress plugin locale filter.
		$locale = apply_filters( 'plugin_locale', get_locale(), 'edd-csr' );
		$mofile = sprintf( '%1$s-%2$s.mo', 'edd-csr', $locale );

		// Setup paths to current locale file.
		$mofile_local  = $lang_dir . $mofile;
		$mofile_global = WP_LANG_DIR . '/edd-conditional-success-redirects/' . $mofile;

		if ( file_exists( $mofile_global ) ) {
			// Look in global /wp-content/languages/edd-conditional-success-redirects/ folder.
			load_textdomain( 'edd-csr', $mofile_global );
		} elseif ( file_exists( $mofile_local ) ) {
			// Look in local /wp-content/plugins/edd-conditional-success-redirects/languages/ folder.
			load_textdomain( 'edd-csr', $mofile_local );
		} else {
			// Load the default language files.
			load_plugin_textdomain( 'edd-csr', false, $lang_dir );
		}
	}

	/**
	 * Include required files.
	 *
	 * @since 1.0
	 *
	 * @return void
	 */
	private function includes() {

		require $this->includes_dir . 'class-process-redirects.php';
		require $this->includes_dir . 'redirect-functions.php';

		do_action( 'edd_csr_include_files' );

		if ( ! is_admin() ) {
			return;
		}

		require $this->includes_dir . 'redirect-actions.php';
		require $this->includes_dir . 'admin-notices.php';
		require $this->includes_dir . 'post-types.php';

		do_action( 'edd_csr_include_admin_files' );
	}


	/**
	 * Add submenu page
	 *
	 * @since 1.0
	 */
	public function add_redirect_options() {
		add_submenu_page( 'edit.php?post_type=download', __( 'Success Redirects', 'edd-csr' ), __( 'Success Redirects', 'edd-csr' ), 'manage_shop_settings', 'edd-redirects', array( $this, 'redirects_page' ) );
	}


	/**
	 * Redirects page
	 *
	 * @since 1.0
	 */
	public function redirects_page() {

		$action = filter_input( INPUT_GET, 'edd-action', FILTER_SANITIZE_SPECIAL_CHARS );
		if ( in_array( $action, array( 'edit_redirect', 'add_redirect' ), true ) ) {
			require_once $this->includes_dir . '/views/editor.php';
		} else {
			require_once $this->includes_dir . 'class-redirects-table.php';
			$redirects_table = new EDD_CSR_Table();
			$redirects_table->prepare_items();
			?>
			<style>
				.column-download,.column-status { width: 350px; }
			</style>
			<div class="wrap">
				<h2><a href="<?php echo esc_url( add_query_arg( array( 'edd-action' => 'add_redirect' ) ) ); ?>" class="add-new-h2"><?php esc_html_e( 'Add New', 'edd-csr' ); ?></a></h2>
				<?php do_action( 'edd_csr_redirects_page_top' ); ?>
				<form id="edd-redirects-filter" method="get" action="<?php echo esc_url( admin_url( 'edit.php?post_type=download&page=edd-redirects' ) ); ?>">
					<?php $redirects_table->search_box( __( 'Search', 'edd-csr' ), 'edd-redirects' ); ?>

					<input type="hidden" name="post_type" value="download" />
					<input type="hidden" name="page" value="edd-redirects" />

					<?php $redirects_table->views(); ?>
					<?php $redirects_table->display(); ?>
				</form>
				<?php do_action( 'edd_csr_redirects_page_bottom' ); ?>
			</div>
			<?php
		}
	}

	/**
	 * Alter the EDD Settings Page Title
	 *
	 * @since 1.0
	 *
	 * @param string $title The current title.
	 * @param string $tab   The current tab.
	 *
	 * @return string
	 */
	public function settings_page_title( $title, $tab ) {
		if ( 'edd-redirects' === $tab ) {
			// Change the title depending on if we're on the list, adding or editing.
			if ( isset( $_GET['edd-action'] ) && 'edit_redirect' === $_GET['edd-action'] ) {
				$title = __( 'Edit Success Redirect', 'edd-csr' );
			} elseif ( isset( $_GET['edd-action'] ) && 'add_redirect' === $_GET['edd-action'] ) {
				$title = __( 'Add New Success Redirect', 'edd-csr' );
			} else {
				$title = __( 'Success Redirects', 'edd-csr' );
			}
		}

		return $title;
	}
}
