<?php
/**
 * All Access Pass Details for the Order Details screen.
 *
 * @package     EDD\AllAccess\Admin\Orders\Details
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.2.8
 */

namespace EDD\AllAccess\Admin\Orders\Details;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\EventManagement\SubscriberInterface;
use EDD\Utils\StatusBadge;

/**
 * All Access Pass Details
 *
 * @since 1.2.8
 */
class PassDetails implements SubscriberInterface {

	/**
	 * Get subscribed events.
	 *
	 * @since 1.2.8
	 * @return array
	 */
	public static function get_subscribed_events() {
		return array(
			'edd_get_order_details_sections' => array( 'add_pass_details_section', 10, 2 ),
		);
	}

	/**
	 * Add the Pass Details section to the Order Details screen.
	 *
	 * @since 1.2.8
	 * @param array             $sections The existing sections.
	 * @param \EDD\Orders\Order $order The order object.
	 * @return array
	 */
	public function add_pass_details_section( $sections, $order ) {
		if ( ! $order instanceof \EDD\Orders\Order ) {
			return $sections;
		}

		$order_contains_passes = false;
		foreach ( $order->items as $order_item ) {
			// Check if this product is an All Access product.
			if ( edd_all_access_enabled_for_download( $order_item->product_id ) ) {
				$order_contains_passes = true;
				break;
			}
		}

		if ( ! $order_contains_passes ) {
			return $sections;
		}

		$sections['pass-details'] = array(
			'id'       => 'pass-details',
			'label'    => __( 'Pass Details', 'edd-all-access' ),
			'icon'     => 'welcome-widgets-menus',
			'callback' => array( $this, 'render_pass_details' ),
		);

		return $sections;
	}

	/**
	 * Render the Pass Details section.
	 *
	 * @since 1.2.8
	 * @param \EDD\Orders\Order $order The order object.
	 * @return void
	 */
	public function render_pass_details( $order ) {
		$all_access_passes = array();
		foreach ( $order->items as $order_item ) {
			// Check if this product is an All Access product.
			$all_access_enabled = edd_all_access_enabled_for_download( $order_item->product_id );

			if ( ! $all_access_enabled ) {
				continue;
			}

			$price_id = is_numeric( $order_item->price_id ) ? $order_item->price_id : 0;

			$all_access_pass     = edd_all_access_get_pass( $order->id, $order_item->product_id, $price_id );
			$all_access_passes[] = array(
				'order_item'   => $order_item,
				'pass_details' => $all_access_pass,
			);
		}
		?>
		<div>
			<div class="order-passes">
				<table class="wp-list-table widefat striped">
					<thead>
						<th class="name column-primary"><?php esc_html_e( 'Product', 'edd-all-access' ); ?></th>
						<th class="column-status"><?php esc_html_e( 'Status', 'edd-all-access' ); ?></th>
						<th class="column-start-date"><?php esc_html_e( 'Start Date', 'edd-all-access' ); ?></th>
						<th class="column-end-date"><?php esc_html_e( 'End Date', 'edd-all-access' ); ?></th>
						<th class="column-actions"><?php esc_html_e( 'Actions', 'edd-all-access' ); ?></th>
					</thead>
					<tbody>
						<?php if ( empty( $all_access_passes ) ) : ?>
							<tr>
								<td colspan="5"><?php esc_html_e( 'No All Access Passes found in this order.', 'edd-all-access' ); ?></td>
							</tr>
						<?php else : ?>
							<?php foreach ( $all_access_passes as $pass ) : ?>
								<tr>
									<td class="column-primary">
										<?php echo esc_html( $pass['order_item']->product_name ); ?>
									</td>
									<td class="column-status">
										<?php
										if ( 'pending' === $order->status ) {
											$status = 'pending';
											$label  = __( 'Payment Pending', 'edd-all-access' );
										} else {
											$status = $pass['pass_details']->status;
											$label  = edd_all_access_get_status_label( $pass['pass_details']->status );
										}

										$status_badge = new StatusBadge(
											array(
												'status' => $status,
												'label'  => $label,
											)
										);
										echo $status_badge->get();
										?>
									</td>
									<td class="column-start-date">
										<?php
										if ( 'invalid' !== $status ) {
											echo esc_html( edd_all_access_visible_date( edd_all_access_get_date_time_format( false ), $pass['pass_details']->start_time ) );
										} else {
											echo '&mdash;';
										}
										?>
									</td>
									<td class="column-end-date">
										<?php
										if ( 'invalid' !== $status ) {
											echo esc_html( 'never' === $pass['pass_details']->expiration_time ? __( 'Never Expires', 'edd-all-access' ) : edd_all_access_visible_date( edd_all_access_get_date_time_format( false ), $pass['pass_details']->expiration_time ) );
										} else {
											echo '&mdash;';
										}
										?>
									</td>
									<td class="column-actions">
										<?php
										if ( 'invalid' !== $status ) {
											$link = edd_get_admin_url(
												array(
													'page'        => 'edd-all-access-pass',
													'payment_id'  => absint( $order->id ),
													'download_id' => absint( $pass['order_item']->product_id ),
													'price_id'    => is_numeric( $pass['order_item']->price_id ) ? $pass['order_item']->price_id : 0,
												)
											);
											printf(
												'<a href="%s">%s</a>',
												esc_url( $link ),
												esc_html__( 'View Details', 'edd-all-access' )
											);
										} else {
											echo '&mdash;';
										}
										?>
									</td>
								</tr>
							<?php endforeach; ?>
						<?php endif; ?>
					</tbody>
				</table>
			</div>
		</div>
		<?php
	}
}
