<?php
/**
 * Plugin Name:     Easy Digital Downloads - All Access
 * Plugin URI:      https://easydigitaldownloads.com/downloads/all-access/
 * Description:     Sell "All Access" memberships to your customers so they can download any product.
 * Version:         1.2.8
 * Requires PHP:    7.4
 * Requires at least: 5.8
 * Author:          Easy Digital Downloads
 * Author URI:      https://easydigitaldownloads.com
 * Text Domain:     edd-all-access
 *
 * @package         EDD\EddAllAccess
 * @author          Easy Digital Downloads
 * @copyright       Copyright (c) Easy Digital Downloads
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Plugin version.
define( 'EDD_ALL_ACCESS_VER', '1.2.8' );

// Plugin path.
define( 'EDD_ALL_ACCESS_DIR', plugin_dir_path( __FILE__ ) );

// Plugin URL.
define( 'EDD_ALL_ACCESS_URL', plugin_dir_url( __FILE__ ) );

// Plugin Root File.
define( 'EDD_ALL_ACCESS_FILE', __FILE__ );

if ( ! class_exists( 'EDD_All_Access' ) ) {

	/**
	 * Main EDD_All_Access class
	 *
	 * @since       1.0.0
	 */
	class EDD_All_Access {

		/**
		 * The one true EDD_All_Access
		 *
		 * @var         EDD_All_Access $instance The one true EDD_All_Access
		 * @since       1.0.0
		 */
		private static $instance;

		/**
		 * Holds the class containing the EDD Recurring Integration.
		 *
		 * @var EDD_All_Access_Recurring
		 */
		public static $edd_recurring;

		/**
		 * Holds the class containing the EDD Software Licensing Integration.
		 *
		 * @var EDD_All_Access_Software_Licensing
		 */
		public static $edd_software_licensing;

		/**
		 * Holds the class containing the EDD Content Restriction Integration.
		 *
		 * @var EDD_All_Access_Content_Restriction
		 */
		public static $edd_content_restriction;

		/**
		 * Holds the class containing the EDD Commissions Integration.
		 *
		 * @var EDD_All_Access_Commissions
		 */
		public static $edd_commissions;

		/**
		 * Holds the class containing the EDD Bookings Integration.
		 *
		 * @var EDD_All_Access_Bookings
		 */
		public static $edd_bookings;

		/**
		 * Holds the class containing the EDD Downloads As Services Integration.
		 *
		 * @var EDD_All_Access_Downloads_As_Services
		 */
		public static $edd_downloads_as_services;

		/**
		 * Holds the class containing the EDD Simple Shipping Integration.
		 *
		 * @var EDD_All_Access_Simple_Shipping
		 */
		public static $edd_simple_shipping;

		/**
		 * Holds the class containing the EDD Frontend Submissions Integration.
		 *
		 * @var EDD_All_Access_Fes
		 */
		public static $edd_fes;

		/**
		 * Holds the class containing the EDD Free Downloads Integration.
		 *
		 * @var EDD_All_Access_Free_Downloads
		 */
		public static $edd_free_downloads;

		/**
		 * Holds the class containing the EDD Custom Prices Integration.
		 *
		 * @var EDD_All_Access_Custom_Prices
		 */
		public static $edd_custom_prices;

		/**
		 * Holds the class containing the EDD Auto Register Integration.
		 *
		 * @var EDD_All_Access_Auto_register
		 */
		public static $edd_auto_register;

		/**
		 * Holds the class containing the Marketify THeme Integration.
		 *
		 * @var EDD_All_Access_Marketify
		 */
		public static $marketify_theme;

		/**
		 * Holds the class containing the Restrict Content Pro Integration.
		 *
		 * @var EDD_All_Access_Restrict_Content_Pro
		 */
		public static $restrict_content_pro;

		/**
		 * Holds the class for the object cache integrations.
		 *
		 * @var EDD_All_Access_Object_Cache
		 */
		public $object_cache;

		/**
		 * Get active instance
		 *
		 * @access      public
		 * @since       1.0.0
		 * @return      object self::$instance The one true EDD_All_Access
		 */
		public static function instance() {
			if ( ! self::$instance ) {
				self::$instance = new EDD_All_Access();
				new EDD\AllAccess\Core();

				self::$instance->includes();
				self::$instance->load_textdomain();
				self::$instance->hooks();

				// Set up integrated plugins.
				self::$edd_recurring             = new EDD_All_Access_Recurring();
				self::$edd_software_licensing    = new EDD_All_Access_Software_Licensing();
				self::$edd_content_restriction   = new EDD_All_Access_Content_Restriction();
				self::$edd_commissions           = new EDD_All_Access_Commissions();
				self::$edd_bookings              = new EDD_All_Access_Bookings();
				self::$edd_downloads_as_services = new EDD_All_Access_Downloads_As_Services();
				self::$edd_simple_shipping       = new EDD_All_Access_Simple_Shipping();
				self::$edd_fes                   = new EDD_All_Access_Fes();
				self::$edd_free_downloads        = new EDD_All_Access_Free_Downloads();
				self::$edd_custom_prices         = new EDD_All_Access_Custom_Prices();
				self::$edd_auto_register         = new EDD_All_Access_Auto_Register();
				self::$restrict_content_pro      = new EDD_All_Access_Restrict_Content_Pro();

				// Set up integrated themes.
				self::$marketify_theme = new EDD_All_Access_Marketify();

				// Setup the Object Cache Helper.
				self::$instance->object_cache = new EDD\AllAccess\Helpers\ObjectCacheHelper();

			}

			return self::$instance;
		}

		/**
		 * Include necessary files
		 *
		 * @access      private
		 * @since       1.0.0
		 * @return      void
		 */
		private function includes() {

			// Include scripts.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/enqueue-scripts.php';

			// Include misc functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/misc-functions.php';

			// Include deprecated functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/deprecated.php';

			// Upgrade Functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/upgrades.php';

			// Include receipt functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/receipts.php';

			// Include All Access status functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/all-access-status-functions.php';

			// Include Downloading functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/downloading-functions.php';

			// Include Download Form functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/download-form.php';

			// Include helper functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/helper-functions.php';

			// Include ajax callback functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/ajax-callbacks.php';

			// Include Customer Meta functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/admin/customers/customers.php';

			// Include All Access Single View page.
			require_once EDD_ALL_ACCESS_DIR . 'includes/admin/custom-pages/all-access-passes-page.php';

			// Include site-wide settings under downloads > settings > all-access.
			require_once EDD_ALL_ACCESS_DIR . 'includes/admin/settings/settings.php';

			// Include Post Meta options.
			require_once EDD_ALL_ACCESS_DIR . 'includes/admin/post-meta/all-access-metabox.php';
			require_once EDD_ALL_ACCESS_DIR . 'includes/admin/post-meta/prices-metabox.php';

			// Include Shortcodes.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/shortcodes.php';

			// EDD_All_Access_Pass object.
			require_once EDD_ALL_ACCESS_DIR . 'includes/class-edd-all-access-pass.php';

			// Tools added to EDD > Tools page.
			require_once EDD_ALL_ACCESS_DIR . 'includes/functions/tools.php';

			// Reports.
			require_once EDD_ALL_ACCESS_DIR . 'includes/admin/reports/reports.php';
			require_once EDD_ALL_ACCESS_DIR . 'includes/admin/reports/class-edd-aa-download-popularity-table.php';

			// Discount Code functions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/admin/discount-codes/discount-codes.php';

			// Integration functions to make this work with EDD Recurring.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-recurring.php';

			// Integration with EDD Software Licensing.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-software-licenses.php';

			// Integration with EDD Content Restriction.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-content-restriction.php';

			// Integration with EDD Commissions.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-commissions.php';

			// Integration with EDD Bookings.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-bookings.php';

			// Integration with EDD Downloads as Services.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-downloads-as-services.php';

			// Integration with EDD Simple Shipping.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-simple-shipping.php';

			// Integration with FES.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-fes.php';

			// Integration with Free Downloads.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-free-downloads.php';

			// Integration with Custom Prices.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-custom-prices.php';

			// Integration with Auto Register.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-auto-register.php';

			// Integration with Marketify Theme.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/theme-marketify.php';

			// Integration with Restrict Content Pro.
			require_once EDD_ALL_ACCESS_DIR . 'includes/integrations/plugin-restrict-content-pro.php';
		}

		/**
		 * Run action and filter hooks
		 *
		 * @access      private
		 * @since       1.0.0
		 * @return      void
		 */
		private function hooks() {
			// Handle licensing.
			add_action(
				'edd_extension_license_init',
				function ( \EDD\Extensions\ExtensionRegistry $registry ) {
					$registry->addExtension( __FILE__, 'All Access', 1005380, EDD_ALL_ACCESS_VER );
				}
			);

			// Register a dashboard page so we can view single All Access pass data.
			add_action( 'admin_menu', array( $this, 'all_access_passes_view_page' ), 10 );
		}

		/**
		 * Register our All Access single view page
		 *
		 * @since  1.0.0
		 * @return void
		 */
		public function all_access_passes_view_page() {
			add_submenu_page(
				'edit.php?post_type=download',
				__( 'All Access Pass', 'edd-all-access' ),
				__( 'All Access Pass', 'edd-all-access' ),
				'view_shop_reports',
				'edd-all-access-pass',
				'edd_all_access_pass_page'
			);
			add_action( 'admin_head', function() {
				remove_submenu_page( 'edit.php?post_type=download', 'edd-all-access-pass' );
			} );
		}

		/**
		 * Internationalization
		 *
		 * @access      public
		 * @since       1.0.0
		 * @return      void
		 */
		public function load_textdomain() {
			// Set filter for language directory.
			$lang_dir = EDD_ALL_ACCESS_DIR . '/languages/';
			$lang_dir = apply_filters( 'edd_all_access_languages_directory', $lang_dir );

			// Traditional WordPress plugin locale filter.
			$locale = apply_filters( 'plugin_locale', get_locale(), 'edd-all-access' );
			$mofile = sprintf( '%1$s-%2$s.mo', 'edd-all-access', $locale );

			// Setup paths to current locale file.
			$mofile_local  = $lang_dir . $mofile;
			$mofile_global = WP_LANG_DIR . '/edd-all-access/' . $mofile;

			if ( file_exists( $mofile_global ) ) {
				// Look in global /wp-content/languages/edd_all_access/ folder.
				load_textdomain( 'edd-all-access', $mofile_global );
			} elseif ( file_exists( $mofile_local ) ) {
				// Look in local /wp-content/plugins/edd_all_access/languages/ folder.
				load_textdomain( 'edd-all-access', $mofile_local );
			} else {
				// Load the default language files.
				load_plugin_textdomain( 'edd-all-access', false, $lang_dir );
			}
		}
	}
} // End if class_exists check

/**
 * The main function responsible for returning the one true EDD_All_Access
 * instance to functions everywhere
 *
 * @since       1.0.0
 * @return      \EDD_All_Access The one true EDD_All_Access
 */
function edd_all_access() {
	return EDD_All_Access::instance();
}

require_once __DIR__ . '/vendor/autoload.php';
\EDD\ExtensionUtils\v1\ExtensionLoader::loadOrQuit(
	__FILE__,
	'edd_all_access',
	array(
		'php'                    => '7.4',
		'easy-digital-downloads' => '3.2',
		'wp'                     => '5.8',
	)
);

/**
 * Admin notice used if EDD is not updated to 2.8 or later.
 *
 * @deprecated 1.2 In favour of ExtensionLoader class.
 *
 * @since       1.0.0
 */
function edd_all_access_edd_too_old_notice() {
	_deprecated_function( __FUNCTION__, '1.2' );
	?>
	<div class="notice notice-error">
	<p><?php echo esc_html( __( 'EDD All Access: Your version of Easy Digital Downloads must be updated to version 2.8 or later to use the All Access extension', 'edd-all-access' ) ); ?></p>
	</div>
	<?php
}

/**
 * Upon fresh activation, this function fires and prevents all previous upgrade routines from running as they are not needed on fresh installs.
 *
 * @since       1.0.0
 */
function edd_all_access_install() {

	$current_version = get_option( 'edd_all_access_version' );

	if ( ! $current_version ) {

		if ( defined( 'EDD_PLUGIN_DIR' ) ) {

			require_once EDD_PLUGIN_DIR . 'includes/admin/upgrades/upgrade-functions.php';

			// When new upgrade routines are added, mark them as complete on fresh install.
			$upgrade_routines = array(
				'aa_v1_reorganize_customer_meta',
				'aa_fix_utc_timezones',
			);

			foreach ( $upgrade_routines as $upgrade ) {
				edd_set_upgrade_complete( $upgrade );
			}
		}
	}

	add_option( 'edd_all_access_version', EDD_ALL_ACCESS_VER, '', false );
}
register_activation_hook( __FILE__, 'edd_all_access_install' );
