<?php
/**
 * All Access section.
 *
 * @package EDD\Admin\Downloads\Editor
 * @since 1.2.8
 */

namespace EDD\AllAccess\Admin\Downloads;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Admin\Downloads\Editor\Section;

/**
 * All Access section.
 *
 * @since 1.2.8
 */
class Metabox extends Section {

	/**
	 * Section ID.
	 *
	 * @since 1.2.8
	 * @var string
	 */
	protected $id = 'all_access';

	/**
	 * Section priority.
	 *
	 * @since 1.2.8
	 * @var int
	 */
	protected $priority = 60;

	/**
	 * Section icon.
	 *
	 * @since 1.2.8
	 * @var string
	 */
	protected $icon = 'welcome-widgets-menus';

	/**
	 * What the section supports.
	 *
	 * @since 1.2.8
	 * @var array
	 */
	protected $supports = array(
		'product-type' => array( 'all_access' ),
	);

	/**
	 * Get the section label.
	 *
	 * @since 1.2.8
	 * @return string
	 */
	public function get_label() {
		return __( 'All Access', 'edd-all-access' );
	}

	/**
	 * Render the section.
	 *
	 * @since 1.2.8
	 * @return void
	 */
	public function render() {
		$download = $this->item;
		// Check if the user can edit this specific download ID (post ID).
		if ( ! $download || ! current_user_can( 'edit_product', $download->ID ) ) {
			return;
		}

		$enabled = edd_all_access_enabled_for_download( $download->ID );
		$product = new \EDD\AllAccess\Models\AllAccessProduct( $download->ID );
		wp_nonce_field( 'edd_all_access_metabox', 'edd_download_all_access_metabox_nonce' );

		?>
		<div class="edd-form-group">
			<label for="edd_all_access_meta_all_access_categories" class="edd-form-group__label">
				<?php
				esc_html_e( 'To which product categories does the customer get "All Access"', 'edd-all-access' );
				$tooltip = new \EDD\HTML\Tooltip(
					array(
						'title'   => __( 'Limit by Category', 'edd-all-access' ),
						'content' => __( 'You can choose which products customers can download with this All Access pass. For example, if you want to sell an All Access pass to just a category called Photos, choose that category here. Note that the category must already exist for it to show up here. You can make product categories under Downloads > Categories.', 'edd-all-access' ),
					)
				);
				$tooltip->output();
				?>
			</label>
			<div class="edd-form-group__control">
				<?php
				$select = new \EDD\HTML\Select(
					array(
						'options'          => $this->get_categories(),
						'name'             => 'edd_all_access_meta[all_access_categories][]',
						'selected'         => $product->categories ?: 'all',
						'id'               => 'edd_all_access_meta_all_access_categories',
						'class'            => 'edd_all_access_meta_all_access_categories',
						'chosen'           => true,
						'placeholder'      => __( 'Type to search Categories', 'edd-all-access' ),
						'multiple'         => true,
						'show_option_all'  => false,
						'show_option_none' => false,
						'data'             => array( 'search-type' => 'no_ajax' ),
					)
				);
				$select->output();
				?>
			</div>
		</div>

		<div class="edd-form-group">
			<label for="edd_all_access_meta_all_access_duration_unit" class="edd-form-group__label">
				<?php
				esc_html_e( 'How long should "All Access" last?', 'edd-all-access' );
				$tooltip = new \EDD\HTML\Tooltip(
					array(
						'title'   => __( 'All Access Duration:', 'edd-all-access' ),
						'content' => __( 'You can set an expiration date for this All Access pass. Once a customer\'s All Access pass expires, they can no longer download products using that pass. If you want to make this renewable (like an ongoing membership), you will want to use the EDD Recurring extension so that this All Access pass is automatically repurchased by the customer once it expires.', 'edd-all-access' ),
					)
				);
				$tooltip->output();
				?>
			</label>
			<div class="edd-form-group__control">
				<input
					type="number"
					class="small-text"
					placeholder="1"
					id="edd_all_access_meta_all_access_duration_number"
					name="edd_all_access_meta[all_access_duration_number]"
					value="<?php echo esc_attr( $product->duration ); ?>"
					min="1"
					<?php if ( empty( $product->duration_unit ) || 'never' === $product->duration_unit ) : ?>
						style="display:none;"
					<?php endif; ?>
				/>
				<select name="edd_all_access_meta[all_access_duration_unit]" id="edd_all_access_meta_all_access_duration_unit">
					<?php
					foreach ( edd_all_access_get_duration_unit_options() as $time_period_slug => $output_string ) {
						?>
						<option value="<?php echo esc_attr( $time_period_slug ); ?>" <?php echo esc_attr( selected( $time_period_slug, $product->duration_unit, false ) ); ?>><?php echo esc_html( $output_string ); ?></option>
						<?php
					}
					?>
				</select>
			</div>
		</div>

		<div class="edd-form-group">
			<label for="edd_all_access_download_limit">
				<?php
				esc_html_e( 'Download Limit', 'edd-all-access' );
				$tooltip = new \EDD\HTML\Tooltip(
					array(
						'title'   => __( 'Download Limit', 'edd-all-access' ),
						'content' => __( 'How many downloads should the customer get? Leave blank or enter \'0\' for unlimited. Note: If a customer\'s account is expired, they won\'t be able to download - even if they have not hit this limit yet.', 'edd-all-access' ),
					)
				);
				$tooltip->output();
				?>
			</label>
			<div class="edd-form-group__control">
				<input type="number" class="small-text" name="edd_all_access_meta[download_limit]" id="edd_all_access_download_limit" value="<?php echo esc_attr( $product->download_limit ); ?>" min="0" />&nbsp;
				<span
					id="edd_all_access_unlimited_download_limit_note"
					<?php if ( ! empty( $product->download_limit ) ) : ?>
						style="display:none;"
					<?php endif; ?>
				>
					<?php esc_html_e( '(Unlimited downloads per day)', 'edd-all-access' ); ?>
				</span>
				<select
					name="edd_all_access_meta[download_limit_time_period]"
					id="edd_all_access_meta_download_limit_time_period"
					<?php if ( empty( $product->download_limit ) ) : ?>
						style="display:none;"
					<?php endif; ?>
				>
					<?php
					foreach ( edd_all_access_get_download_limit_periods() as $time_period_slug => $output_string ) {
						?>
						<option value="<?php echo esc_attr( $time_period_slug ); ?>" <?php echo esc_attr( selected( $time_period_slug, $product->download_limit_period, false ) ); ?>>
							<?php echo esc_html( str_replace( 'X', $product->download_limit, $output_string ) ); ?>
						</option>
						<?php
					}
					?>
				</select>
			</div>
		</div>

		<div class="edd-form-group">
			<label for="edd_all_access_number_of_price_ids">
				<?php
				esc_html_e( 'Total Price Variations (Optional):', 'edd-all-access' );
				$tooltip = new \EDD\HTML\Tooltip(
					array(
						'title'   => __( 'Price Variations', 'edd-all-access' ),
						'content' => __( 'How many price variations are there? Leave blank or enter \'0\' to include all price variations.', 'edd-all-access' ),
					)
				);
				$tooltip->output();
				?>
			</label>
			<div class="edd-form-group__control">
				<input type="number" class="small-text" name="edd_all_access_meta[number_of_price_ids]" id="edd_all_access_number_of_price_ids" value="<?php echo esc_attr( $product->number_price_ids ); ?>" min="0" />&nbsp;
				<p
					id="edd_all_access_included_price_ids_note"
					class="edd-form-group__help description"
					<?php if ( empty( $product->download_limit ) ) : ?>
						style="display:none;"
					<?php endif; ?>
				>
					<?php esc_html_e( 'Because this is set to 0, all price variations are included.', 'edd-all-access' ); ?>
				</p>
			</div>
			<div
				<?php if ( empty( $product->number_price_ids ) ) : ?>
					style="display:none;"
				<?php endif; ?>
				class="edd_all_access_included_price_ids_row"
			>
				<div class="edd_field_type_text">
					<p><strong><?php echo esc_html( __( 'Included Price Variations:', 'edd-all-access' ) ); ?></strong></p>
					<?php echo esc_html( __( 'Which price variations should be included in this All Access?', 'edd-all-access' ) ); ?>
					<ul id="edd_all_access_included_price_ids">
						<?php
						for ( $price_id = 1; $price_id <= $product->number_price_ids; $price_id++ ) {

							$variation_string = __( 'th Price Variation from each product', 'edd-all-access' );
							$variation_string = 1 === $price_id ? __( 'st Price Variation from each product', 'edd-all-access' ) : $variation_string;
							$variation_string = 2 === $price_id ? __( 'nd Price Variation from each product', 'edd-all-access' ) : $variation_string;
							$variation_string = 3 === $price_id ? __( 'rd Price Variation from each product', 'edd-all-access' ) : $variation_string;

							?>
							<li class="edd_all_access_included_price_id_li <?php echo esc_html( $price_id ); ?>">
								<label><input type="checkbox" name="edd_all_access_meta[included_price_ids][]" class="edd_all_access_included_price_id" value="<?php echo esc_attr( $price_id ); ?>" <?php echo esc_attr( ( in_array( $price_id, ( $product->included_price_ids ?: array() ), true ) ? ' checked ' : '' ) ); ?>/>
								<?php echo esc_html( $price_id . $variation_string ); ?></label>
							</li>
							<?php
						}
						?>
					</ul>
				</div>
			</div>
		</div>

		<div class="edd-form-group">
			<label for="edd_all_access_receipt_meta_show_link">
				<?php
				esc_html_e( 'Show Links in Receipts', 'edd-all-access' );
				$tooltip = new \EDD\HTML\Tooltip(
					array(
						'title'   => __( 'Show Links in Receipts', 'edd-all-access' ),
						'content' => sprintf(
							/* translators: the download list email tag */
							__( 'Would you like to output a custom link in the receipts your customers receive directing them to use their All Access Pass? Note: For email Receipts, you must be using the %s email tag.', 'edd-all-access' ),
							'{download_list}'
						),
					)
				);
				$tooltip->output();
				?>
			</label>
			<div class="edd-form-group__control">
				<select name="edd_all_access_receipt_meta[show_link]" id="edd_all_access_receipt_meta_show_link">
					<option value="show_link" <?php selected( $product->show_link_in_receipt ); ?>><?php esc_html_e( 'Show link in receipt', 'edd-all-access' ); ?></option>
					<option value="hide_link" <?php selected( ! $product->show_link_in_receipt ); ?>><?php esc_html_e( 'Hide link in receipt', 'edd-all-access' ); ?></option>
				</select>
			</div>
		</div>

		<div class="edd-form-group">
			<label for="edd_all_access_receipt_meta_link_message">
				<?php esc_html_e( 'Receipts: All Access Link Message:', 'edd-all-access' ); ?>
			</label>
			<div class="edd-form-group__control">
				<textarea name="edd_all_access_receipt_meta[link_message]" id="edd_all_access_receipt_meta_link_message" style="width:100%;"><?php echo esc_html( $product->receipt_link_message ); ?></textarea>
				<p class="edd-form-group__help description"><?php esc_html_e( 'What should the link in the receipt say to the user?', 'edd-all-access' ); ?></p>
			</div>
		</div>

		<div class="edd-form-group">
			<label for="edd_all_access_receipt_meta_link_url">
				<?php
				esc_html_e( 'Receipts Link URL', 'edd-all-access' )
				?>
			</label>
			<div class="edd-form-group__control">
				<input style="width:100%;" type="url" name="edd_all_access_receipt_meta[link_url]" id="edd_all_access_receipt_meta_link_url" value="<?php echo esc_attr( $product->receipt_link_url ); ?>" />
				<p class="edd-form-group__help description">
					<?php
					printf(
						/* translators: %1$s: opening anchor tag, %2$s: closing anchor tag */
						__( 'Which URL should the customer be directed to in the receipt? If you want to build your own custom page, %1$slearn how in this document%2$s.', 'edd-all-access' ),
						sprintf(
							/* translators: %s: the URL to the documentation */
							'<a href="%s" target="_blank">',
							edd_link_helper(
								'http://docs.easydigitaldownloads.com/article/1829-all-access-creating-all-access-products#creating-a-custom-page-of-products-the-customer-can-download-via-all-access',
								array(
									'utm_medium'   => 'edit-download',
									'utm_campaign' => 'all-access-receipt-link-url',
								)
							)
						),
						'</a>'
					);
					?>
				</p>
			</div>
		</div>
		<?php
	}

	/**
	 * Get the category options.
	 *
	 * @since 1.2.8
	 * @return array
	 */
	private function get_categories() {
		$categories = get_terms(
			apply_filters(
				'edd_category_dropdown',
				array(
					'taxonomy' => 'download_category',
				)
			)
		);
		$options    = array(
			'all' => __( 'All Products', 'edd-all-access' ),
		);

		foreach ( $categories as $category ) {
			$options[ absint( $category->term_id ) ] = esc_html( $category->name );
		}

		return $options;
	}
}
