<?php
/**
 * Integration functions to make All Access compatible with the EDD Simple Shipping extension
 *
 * @package     EDD\EDDAllAccess\Functions
 * @since       1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Integrates EDD All Access with the EDD Simple Shipping extension
 *
 * @since 1.0.0
 */
class EDD_All_Access_Simple_Shipping {

	/**
	 * Get things started
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function __construct() {

		if ( ! class_exists( 'EDD_Simple_Shipping' ) ) {
			return;
		}

		add_filter( 'edd_all_access_allow', array( $this, 'allow_simple_shipping' ), 10, 5 );

	}

	/**
	 * Don't allow All Access to change the download form shown to the user if the post is a Shipping product.
	 * You can't download a Physical Product post so having "access" to download it doesn't make sense here.
	 *
	 * @since       1.0.0
	 * @param       bool   $allowed Whether to allow All Access to change the download form for this product or not.
	 * @param       string $purchase_form The actual form output which is being filtered.
	 * @param       array  $args The arguments passed to the edd_purchase_download_form filter.
	 * @param       int    $download_id The ID of the product which is being shown.
	 * @param       int    $price_id The variable price ID of the product which is being shown.
	 * @return      bool   $allowed Whether to allow All Access to change the download form for this product or not.
	 */
	public function allow_simple_shipping( $allowed, $purchase_form, $args, $download_id, $price_id ) {

		$shipping_enabled = get_post_meta( $download_id, '_edd_enable_shipping', true );

		// If shipping is not enabled for this product, let it pass through as is.
		if ( ! $shipping_enabled ) {
			return $allowed;
		}

		// If we are checking for a specific price id to see if Shipping is enabled.
		if ( ! empty( $price_id ) || ( empty( $price_id ) && is_numeric( $price_id ) ) ) {
			$variable_prices_settings = get_post_meta( $download_id, 'edd_variable_prices', true );

			// Check if shipping is enabled for this variable price, turn All Access off.
			if ( isset( $variable_prices_settings[ $price_id ]['shipping'] ) ) {
				return false;
			}
		} else {
			// If no price ID was passed in but shipping is enabled, turn All Access off.
			return false;
		}

		return $allowed;
	}

}
