<?php
/**
 * Integration functions to make All Access compatible with Restrict Content Pro
 *
 * @package     EDD\EDDAllAccess\Functions
 * @since       1.0.2
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Integrates EDD All Access with the Restrict Content Pro extension
 *
 * @since 1.0.0
 */
class EDD_All_Access_Restrict_Content_Pro {

	/**
	 * Get things started
	 *
	 * @since  1.0.2
	 * @return void
	 */
	public function __construct() {

		if ( ! defined( 'RCP_PLUGIN_VERSION' ) ) {
			return;
		}

		add_action( 'edd_tools_tab_general', array( $this, 'edd_all_access_migrate_from_rcp' ) );

		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );

		add_action( 'wp_ajax_edd_all_access_do_ajax_rcp_migrate', array( $this, 'edd_all_access_do_ajax_rcp_migrate' ) );

	}

	/**
	 * Process All Access Payments via ajax
	 *
	 * @since 1.0.2
	 * @return void
	 */
	public function edd_all_access_do_ajax_rcp_migrate() {

		if ( ! isset( $_POST['form'] ) ) {
			die( esc_html( __( 'Invalid paramaters sent. Please try again.', 'edd-all-access' ) ) );
		}

		parse_str( sanitize_text_field( wp_unslash( $_POST['form'] ) ), $form );

		$form = (array) $form;

		if ( ! wp_verify_nonce( $form['edd_all_access_ajax_rcp_migrate'], 'edd_all_access_ajax_rcp_migrate' ) ) {
			die( esc_html( __( 'Nonce Failure. Please try again.', 'edd-all-access' ) ) );
		}

		$step = isset( $_POST['step'] ) ? absint( $_POST['step'] ) : 1;

		// Get all product/downloads which are All Access-enabled.
		$all_access_products = edd_all_access_get_all_access_downloads();

		$status       = 'active';
		$subscription = null;
		$offset       = 10 * $step;
		$number       = 10;

		// Get all Members in RCP.
		$rcp_members = rcp_get_members( $status, $subscription, $offset, $number );

		// If there are RCP members.
		if ( $rcp_members ) {

			// Loop through those members.
			foreach ( $rcp_members as $rcp_member ) {

				// Get the RCP Member.
				$rcp_member = new RCP_Member( $rcp_member->ID );

				$rcp_payment_date    = $rcp_member->get_joined_date();
				$rcp_subscription_id = $rcp_member->get_subscription_id();

				// Get the user.
				$user = get_userdata( $rcp_member->ID );

				// Check for discounts.
				$discounts = get_user_meta( $rcp_member->ID, 'rcp_user_discounts', true );
				if ( ! empty( $discounts ) && is_array( $discounts ) && ! $discounts instanceof stdClass ) {
					foreach ( $discounts as $key => $code ) {
						if ( ! is_string( $code ) ) {
							unset( $discounts[ $key ] );
						}
					}
					$discounts = implode( ' ', $discounts );
				}

				// We need to figure out which corresponding All Access Product to use based on which RCP product this is.
				$rcp_product_counter = 1;
				foreach ( $form['edd_aa_rcp_product'] as $rcp_product ) {

					// If the rcp member's subscription/product matches the one chosen in the admin, set the corresponding All Access product accordingly.
					if ( intval( $rcp_subscription_id ) === intval( $rcp_product ) ) {
						$corresponding_all_access_product_id = intval( $form[ 'edd_aa_rcp_corresponding_aa_' . $rcp_product_counter ] );
						break;
					}

					$rcp_product_counter++;
				}

				// Now we need to figure out which payment gateway was used in RCP for this payment.
				$rcp_members_payments = $rcp_member->get_payments();
				if ( $rcp_members_payments ) {
					$rcp_members_latest_payment = reset( $rcp_members_payments );
					$rcp_payment_type           = strtolower( $rcp_members_latest_payment->payment_type );
				} else {
					$rcp_members_latest_payment = null;
					$rcp_payment_type           = 'manual';
				}

				// Simplify variable name here.
				$rcp_payment = $rcp_members_latest_payment;

				// If we have already generated a payment for this RCP payment previously, don't duplicate it.
				$args = array(
					'post_type'      => 'edd_payment',
					'posts_per_page' => 1,
					'meta_query'     => array( //phpcs:ignore WordPress.DB.SlowDBQuery
						array(
							'key'   => '_edd_aa_migrated_rcp_member_id',
							'value' => $rcp_member->ID,
						),
					),
				);

				$rcp_payments_query = new WP_Query( $args );

				// We have already generated a payment for this RCP member's subscription so skip this member.
				if ( $rcp_payments_query->have_posts() ) {
					continue;
				}

				switch ( $rcp_payment_type ) {

					case 'web_accept':
					case 'paypal express one time':
					case 'recurring_payment':
					case 'subscr_payment':
					case 'recurring_payment_profile_created':
						$rcp_payment_gateway = 'paypal';
						break;

					case 'credit card':
					case 'credit card one time':
						$rcp_payment_gateway = 'stripe';
						break;

					case 'braintree credit card one time':
					case 'braintree credit card initial payment':
					case 'braintree credit card':
						$rcp_payment_gateway = 'braintree';
						break;

					default:
						$rcp_payment_gateway = 'manual';
						break;
				}

				// First we will make sure a customer exists for this user.
				$customer = new EDD_Customer( $rcp_member->ID, true );

				// Set up the names.
				$first_name = empty( $rcp_member->first_name ) ? __( 'First Name', 'edd-all-access' ) : $rcp_member->first_name;
				$last_name  = empty( $rcp_member->last_name ) ? __( 'Last Name', 'edd-all-access' ) : $rcp_member->last_name;

				// If this customer does not yet exist.
				if ( 0 === intval( $customer->id ) ) {

					$customer = new EDD_Customer();

					$customer_args = array(
						'user_id'        => $rcp_member->ID,
						'name'           => $first_name . ' ' . $last_name,
						'email'          => isset( $user->user_email ) ? $user->user_email : '',
						'payment_ids'    => '',
						'purchase_value' => '',
						'purchase_count' => '',
						'notes'          => 'Created via EDD All Access\'s migration tool for Restrict Content Pro.',
					);

					$success = $customer->create( $customer_args );

				}

				// Now we will create the EDD All Access payment to correspond with the RCP subscription.
				$edd_payment = new EDD_Payment();
				$edd_payment->add_download(
					$corresponding_all_access_product_id,
					array(
						'quantity'   => 1,
						'price_id'   => false,
						'item_price' => $rcp_payment->amount,
						'tax'        => 0.00,
						'fees'       => array(),
					)
				);
				$edd_payment->total             = $rcp_payment->amount;
				$edd_payment->subtotal          = $rcp_payment->amount;
				$edd_payment->tax               = 'draft';
				$edd_payment->number            = '';
				$edd_payment->mode              = 'live';
				$edd_payment->gateway           = $rcp_payment_gateway;
				$edd_payment->date              = $rcp_payment->date;
				$edd_payment->status            = 'complete';
				$edd_payment->email             = isset( $user->user_email ) ? $user->user_email : '';
				$edd_payment->first_name        = $first_name;
				$edd_payment->last_name         = $last_name;
				$edd_payment->customer_id       = $customer->ID;
				$edd_payment->user_id           = $rcp_payment->user_id;
				$edd_payment->discounts         = $discounts;
				$edd_payment->key               = '';
				$edd_payment->transaction_id    = $rcp_payment->transaction_id;
				$edd_payment->ip                = '';
				$edd_payment->currency          = '';
				$edd_payment->parent_payment_id = '';
				$edd_payment->line1             = '';
				$edd_payment->line2             = '';
				$edd_payment->city              = '';
				$edd_payment->state             = '';
				$edd_payment->zip               = '';
				$edd_payment->country           = '';

				// Save the payment.
				$edd_payment->save();

				// Add a peice of meta letting us know this payment was migrated from RCP.
				$edd_payment->update_meta( '_edd_aa_migrated_rcp_member_id', $rcp_member->ID );

				// Check if this was a recurring subscription payment or not.
				$subscription_profile_id = $rcp_member->get_merchant_subscription_id();

				if ( ! empty( $subscription_profile_id ) ) {

					// Allow other extensions (like EDD recurring) to carry out actions for subscriptions being migrated.
					do_action( 'edd_all_access_rcp_migrate_subscription', $subscription_profile_id, $edd_payment, $corresponding_all_access_product_id );

				}
			}
		}

		// Get the total number of RCP members left.
		$total_active_rcp_members = rcp_get_member_count();

		$total_rcp_members_left = $total_active_rcp_members - ( $step * 10 );

		// If no more Payments are left.
		if ( 0 === $total_rcp_members_left ) {
			$percentage = 100;
			$message    = __( 'All RCP Members Migrated to EDD All Access. ', 'edd-all-access' );
			echo wp_json_encode(
				array(
					'success' => true,
					'message' => $message,
				)
			);
			exit;

		} else {
			$percentage = ( ( $step * 10 ) / $total_active_rcp_members ) * 100;
			$step++;
			echo wp_json_encode(
				array(
					'step'                   => $step,
					'percentage'             => $percentage,
					'other'                  => $rcp_members,
					'total_rcp_members_left' => $total_rcp_members_left,
				)
			);
			exit;
		}
	}

	/**
	 * Enqueue JS Scripts for RCP integration in admin
	 *
	 * @since       1.0.0
	 * @return      void
	 */
	public function admin_enqueue_scripts() {
		wp_enqueue_script( 'admin_edd_all_access_rcp_js', EDD_ALL_ACCESS_URL . 'assets/js/admin/build/edd-aa-admin-rcp.js', array( 'jquery' ), true, EDD_ALL_ACCESS_VER );
	}

	/**
	 * Display the "Migrate RCP Members to All Access members" tool.
	 *
	 * @since       1.0.0
	 * @return      void
	 */
	public function edd_all_access_migrate_from_rcp() {

		if ( ! current_user_can( 'manage_shop_settings' ) ) {
			return;
		}

		?>
		<div class="postbox">
			<h3><span><?php esc_html_e( 'All Access - Migrate RCP Members to All Access Members', 'edd-all-access' ); ?></span></h3>
			<div class="inside recount-stats-controls">
				<p><?php echo esc_html__( 'Use this tool to take each active member you have in Restrict Content Pro and migrate them to be an Easy Digital Downloads Customer with an All Access Pass.', 'edd-all-access' ); ?></p>
				<form method="post" id="edd-tools-all-access-rcp-migrate-form" class="edd-all-access-rcp-migrate-form edd-import-export-form">
					<span>

					<?php
					wp_nonce_field( 'edd_all_access_ajax_rcp_migrate', 'edd_all_access_ajax_rcp_migrate' );

					// Make sure that the site has EDD Recurring in case any of the RCP subscriptions auto renew.
					if ( ! class_exists( 'EDD_Recurring' ) ) {
						// Translators: 1. The opening part of the to EDD Recurring Payments. 2. The closing part of the link.
						echo wp_kses_post( sprintf( __( 'You do not have the %1$s EDD Recurring Payments %2$s extension. It is required for this migration process because some of your Restrict Content Pro members may have auto recurring subscriptions.', 'edd-all-access' ), '<a href="https://easydigitaldownloads.com/downloads/recurring-payments" target="_blank">', '</a>' ) );

						echo wp_kses_post( '</span></form></div><!-- .inside --></div><!-- .postbox -->' );

						return;
					}

					// Get all of the Subcriptions (products) available in RCP.
					$rcp_products = rcp_get_subscription_levels();

					if ( empty( $rcp_products ) ) {
						echo esc_html( __( 'You don\'t have any RCP Subscription Products created so there is nothing to migrate', 'edd-all-access' ) );
						echo wp_kses_post( '</span></form></div><!-- .inside --></div><!-- .postbox -->' );
						return;
					}

					// Get all of the All Access products available in EDD.
					$all_access_products = edd_all_access_get_all_access_downloads();

					if ( empty( $all_access_products ) ) {
						echo wp_kses_post( __( 'You haven\'t created any All Access Products yet. Before you can migrate your RCP members to EDD All Access, you must create the All Access Products you want them to have. Once you have, you can complete the migration here. To learn how to create an All Access product, read', 'edd-all-access' ) . ' <a href="http://docs.easydigitaldownloads.com/article/1829-all-access-creating-all-access-products" target="_blank">' . __( 'this document', 'edd-all-access' ) . '</a>' );
						echo wp_kses_post( '</span></form></div><!-- .inside --></div><!-- .postbox -->' );

						return;
					}

					$aa_chosen_options = array();

					// Create an array with each AA product title so we can use it for the chosen js dropdown.
					foreach ( $all_access_products as $all_access_product ) {
						$aa_chosen_options[ $all_access_product ] = get_the_title( $all_access_product );
					}

					?>
					<table class="widefat edd_repeatable_table" width="100%" cellpadding="0" cellspacing="0">
						<thead>
							<tr>
								<th><?php esc_html_e( 'Migrate customers from this RCP subscription...', 'edd-all-access' ); ?></th>
								<th><?php esc_html_e( '...to this EDD All Access Product', 'edd-all-access' ); ?></th>
							</tr>
						</thead>
						<tbody>
							<?php

							$rcp_product_counter = 1;

							// Loop through each RCP product so we can output the All Access product the site owner wants to correlate to each.
							foreach ( $rcp_products as $rcp_product ) {
								?>
								<tr class="edd_variable_prices_wrapper edd_repeatable_row" data-key="1">
									<td><?php echo esc_html( $rcp_product->name ); ?><input type="hidden" name="edd_aa_rcp_product[<?php echo absint( $rcp_product_counter ); ?>]" value="<?php echo absint( $rcp_product_counter ); ?>"></td>
									<td>
									<?php

									echo EDD()->html->select(
										array(
											'options'     => $aa_chosen_options,
											'name'        => 'edd_aa_rcp_corresponding_aa_' . absint( $rcp_product_counter ),
											'selected'    => '',
											'id'          => 'edd_aa_rcp_corresponding_aa_' . absint( $rcp_product_counter ),
											'class'       => 'edd_all_access_meta_all_access_categories',
											'chosen'      => true,
											'placeholder' => esc_html( __( 'Type to search All Access Products', 'edd-all-access' ) ),
											'multiple'    => false,
											'show_option_all' => false,
											'show_option_none' => false,
											'data'        => array( 'search-type' => 'no_ajax' ),
										)
									);
									?>
										</td>
								</tr>
								<?php
								$rcp_product_counter++;
							}
							?>
						</tbody>
					</table>

					<br />

					<input type="submit" id="edd-all-access-tool-submit" value="<?php esc_html_e( 'Migrate RCP Members to EDD All Access', 'edd-all-access' ); ?>" class="button-secondary"/>

					<br />

					<span class="spinner"></span>

					</span>
				</form>
			</div><!-- .inside -->
		</div><!-- .postbox -->
		<?php
	}
}
