<?php
/**
 * Integration functions to make All Access compatible with the EDD Downloads As Services extension
 *
 * @package     EDD\EDDAllAccess\Functions
 * @since       1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Integrates EDD All Access with the EDD Downloads As Services extension
 *
 * @since 1.0.0
 */
class EDD_All_Access_Downloads_As_Services {

	/**
	 * Get things started
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function __construct() {

		if ( ! class_exists( 'EDD_Downloads_As_Services' ) ) {
			return;
		}

		add_filter( 'edd_all_access_allow', array( $this, 'allow_services' ), 10, 5 );

	}

	/**
	 * Don't allow All Access to change the download form shown to the user if the post is a Downloads As Services post.
	 * You can't download a "service" so having "access" to download it doesn't make sense here.
	 *
	 * @since       1.0.0
	 * @param       bool   $allowed Whether to allow All Access to change the download form for this product or not.
	 * @param       string $purchase_form The actual form output which is being filtered.
	 * @param       array  $args The arguments passed to the edd_purchase_download_form filter.
	 * @param       int    $post_id The ID of the product which is being shown.
	 * @param       int    $price_id  The variable price ID of the product which is being shown.
	 * @return      bool   $allowed Whether to allow All Access to change the download form for this product or not.
	 */
	public function allow_services( $allowed, $purchase_form, $args, $post_id, $price_id ) {

		if ( $this->is_service( $post_id, $price_id ) ) {
			return false;
		}

		return $allowed;

	}

	/**
	 * Is service - This method is duplicated from the EDD Downloads As Services main class and modified to handle price ids from the filter args.
	 *
	 * @param  int $item_id ID of download.
	 * @param  int $price_id Variable price ID of download.
	 * @return bool true if service, false otherwise.
	 */
	public function is_service( $item_id, $price_id ) {
		global $edd_options;

		// Get array of service categories.
		$service_categories = isset( $edd_options['edd_das_service_categories'] ) ? $edd_options['edd_das_service_categories'] : '';

		$term_ids = array();

		if ( ! empty( $service_categories ) ) {
			foreach ( $service_categories as $term_id => $term_name ) {
				$term_ids[] = $term_id;
			}
		}

		$is_service = get_post_meta( $item_id, '_edd_das_enabled', true );

		$download_files = edd_get_download_files( $item_id, $price_id );

		// If the service has a file attached, we still want to show it.
		if ( $is_service && $download_files ) {
			return false;
		}

		// Check if the service checkbox was clicked in the product's settings.
		if ( $is_service ) {
			return true;
		}

		// If the product has a service term assigned to it.
		if ( ! empty( $term_ids ) && has_term( $term_ids, 'download_category', $item_id ) ) {
			return true;
		}

		return false;
	}

}
