<?php
/**
 * Integration functions to make All Access compatible with the EDD Content Restriction extension
 *
 * @package     EDD\EDDAllAccess\Functions
 * @since       1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Integrates EDD All Access with the EDD Content Restriction extension
 *
 * @since 1.0.0
 */
class EDD_All_Access_Content_Restriction {

	/**
	 * Get things started
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function __construct() {

		if ( ! class_exists( 'EDD_Content_Restriction' ) ) {
			return;
		}

		add_action( 'admin_notices', array( $this, 'edd_all_access_cr_too_old_notice' ) );
		add_filter( 'edd_cr_user_can_access_status_and_message', array( $this, 'can_access_content_because_of_all_access' ), 10, 4 );

	}

	/**
	 * Make sure we are at the minimum version of Content Restriction - which is 2.0
	 *
	 * @since       1.0.0
	 */
	public function edd_all_access_cr_too_old_notice() {

		if ( defined( 'EDD_CONTENT_RESTRICTION_VER' ) && version_compare( EDD_CONTENT_RESTRICTION_VER, '2.0' ) <= 0 ) {
			?>
			<div class="notice notice-error">
				<p><?php echo esc_html( __( 'EDD All Access: Your version of EDD Content Restriction must be updated to version 2.0 or later to use the All Access extension in conjunction with Content Restriction.', 'edd-all-access' ) ); ?></p>
			</div>
			<?php
		}
	}

	/**
	 * Check if user has access to content
	 *
	 * @since  1.1
	 * @param  bool  $return Whether the user has access to the content or not.
	 * @param  int   $user_id The ID of the WordPress user in question.
	 * @param  array $restricted_to The ids of the product(s) that must be purchased in order to get access to the content in question.
	 * @param  int   $post_id The id of the WordPress post containing the locked content.
	 * @return bool
	 */
	public function can_access_content_because_of_all_access( $return, $user_id, $restricted_to, $post_id ) {

		if ( ! is_array( $restricted_to ) ) {
			return $return;
		}

		$restricted_to_download_ids = wp_list_pluck( $restricted_to, 'download' );
		$all_access_products        = edd_all_access_get_all_access_downloads();

		$status_because_of_all_access = array(
			'status'     => false,
			'message_id' => 'no_relevant_all_access_passes_found',
			'message'    => __( 'No relevant All Access Passes found', 'edd-all-access' ),
		);

		/*
		 * If they already have access, don't take it away because of All Access.
		 * In the context of Content Restriction's settings, All Access either:
		 * a) provides access that was never there (if active).
		 * b) doesn't provide access that was never there (if expired).
		 * It does NOT ever take away access that was there prior to this filter.
		 * Therefore, if they already have access through CR's checks, All Access is irrelevant
		 */
		if ( $return['status'] ) {
			return $return;
		}

		// Loop through each product this content/post requires to be unlocked.
		foreach ( $restricted_to as $restricted_to_product ) {

			if ( ! isset( $restricted_to_product['price_id'] ) ) {

				$restricted_to_product['price_id'] = 0;

			}

			// If the price ID doesn't matter.
			if ( 'all' === $restricted_to_product['price_id'] ) {

				// Get the variable prices attached to this product.
				$variable_prices = edd_get_variable_prices( $restricted_to_product['download'] );

				// Check each variable price to see they are covered by any All Access Pass the customer might have.
				foreach ( $variable_prices as $price_id => $price_data ) {

					$args = array(
						'download_id' => $restricted_to_product['download'],
						'price_id'    => $price_id,
					);

					$all_access = edd_all_access_check( $args );

					if ( $all_access['success'] ) {

						// If they have access, we don't need to do any more checking.
						return array(
							'status'  => true,
							'message' => __( 'You have access because of All Access Pass ID: ', 'edd-all-access' ) . $all_access['all_access_pass']->id,
						);

					}
				}
			} else {

				$args = array(
					'download_id' => $restricted_to_product['download'],
					'price_id'    => $restricted_to_product['price_id'],
				);

				$all_access = edd_all_access_check( $args );

				if ( $all_access['success'] ) {

					// If they have access, we don't need to do any more checking.
					return array(
						'status'  => true,
						'message' => __( 'You have access because of All Access Pass ID: ', 'edd-all-access' ) . $all_access['all_access_pass']->id,
					);

				}
			}
		}

		// If we got to here, it means All Access did not affect the content in any way, making All Access checks irrelevant.
		// Therefore, we can pass back the $return like nothing happened here.
		return $return;

	}

}
