<?php
/**
 * Integration functions to make All Access compatible with the EDD Commissions extension
 *
 * Integration functions to make All Access compatible with EDD Commissions.
 * The main change that All Access makes to Commissions is that Commissions on All Access purchases are not paid until the END of a customer's payment period.
 * It will also over-ride any commissions settings in the download's meta.
 *
 * To understand what this file does to change the default behaviour of EDD Commissions, read the following:
 * To set the stage, say you buy an All Access subscription to a photo site with hundreds of photographers set up as Vendors through something like FES.
 * For simplicity's sake in this example, lets say we have defined each Photographer's commission for their photo as 100% so they earn the entire value of their product.
 * Now, pretend you are a customer purchasing an "All Access" pass to this photography website - allowing you to download any photo you'd like.
 * If your All Access period is "1 month", costs $100, and during that time you downloaded 2 photos from 2 different photographers, at the end of that month, All Access checks
 * which photos you chose to download and splits the commission between those photographers. The split percentage is weighted based on the dollar value of the product. For example:
 * Photo 1 = $20
 * Photo 2 = $5
 * Total value of downloaded items = $25
 * Using the following formula, we can find which value percentage each photo is of the total:
 * 100 * ( $price_of_download / $total_value_of_downloaded_items ) = $percentage_of_total_value
 * Photo 1's percentage of the total is 80%
 * Photo 2's percentage of the total is 20%
 * Using those values and knowing that the All Access cost was $100, we'll pay a commission of $80 to the Photographer of Photo 1 and $20 to the one of Photo 2.
 * If a Photographer has multiple photos that were downloaded, we add all of their percentages together in a single commission.
 * If a single photo had multiple photographers (and thus multiple commission recievers), the $80 would then be split up using their commission rates defined in the product.
 * Any leftover money not commissioned-out will remain with the store. As a store owner, set your commission rates in each product accordingly.
 * Notes:
 * - Flat rate commissions do not work with All Access. All Access calculated commissions using percentages only.
 *
 * @package     EDD\EDDAllAccess\Functions
 * @since       1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Integrates EDD All Access with the EDD Commissions extension
 *
 * @since 1.0.0
 */
class EDD_All_Access_Commissions {

	/**
	 * Get things started
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function __construct() {

		if ( ! function_exists( 'edd_commissions' ) ) {
			return;
		}
		add_action( 'plugins_loaded', array( $this, 'run' ), 200 );
	}

	/**
	 * Run the Commissions integration. Set up as a separate function
	 * to allow a Commissions version check to be run after the Commissions
	 * version has been defined.
	 *
	 * @since 1.1.11
	 * @return void
	 */
	public function run() {

		add_action( 'admin_notices', array( $this, 'edd_all_access_commissions_too_old_notice' ) );

		if ( ! $this->is_commissions_version_acceptable() ) {
			return;
		}

		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );

		add_filter( 'eddc_commission_tabs', array( $this, 'edd_all_access_commissions_tab' ) );

		add_filter( 'eddc_commission_views', array( $this, 'edd_all_access_commissions_view' ) );

		add_action( 'eddc_metabox_after_enable', array( $this, 'edd_all_access_commissions_post_meta' ) );

		add_action( 'edd_all_access_expired', array( $this, 'edd_all_access_calculate_commissions' ), 10, 2 );

		add_filter( 'eddc_get_recipients', array( $this, 'edd_all_access_strip_recipients_from_aa_products' ), 10, 2 );

		add_filter( 'eddc_commissions_calculated', array( $this, 'edd_all_access_prevent_aa_commissions' ), 10, 2 );

		add_filter( 'eddc_commission_types', array( $this, 'edd_all_access_commission_types' ) );

		add_filter( 'eddc_format_rate', array( $this, 'edd_all_access_format_rate' ), 10, 3 );

		add_action( 'admin_enqueue_scripts', array( $this, 'edd_all_access_default_css' ) );
	}

	/**
	 * Check the current version of EDD Commissions and return early if it's less than 3.4.
	 *
	 * @since 1.1.11
	 * @return bool
	 */
	private function is_commissions_version_acceptable() {
		return defined( 'EDD_COMMISSIONS_VERSION' ) && version_compare( EDD_COMMISSIONS_VERSION, '3.4', '>=' );
	}

	/**
	 * When Commission rates are shown and need formatting, set them to show as a %.
	 *
	 * @since       1.0.0
	 * @param       string $formatted_rate  The formatted rate being shown.
	 * @param       int    $rate_only       The rate, but unformatted.
	 * @param       string $commission_type The type of commission this is.
	 * @return      string
	 */
	public function edd_all_access_format_rate( $formatted_rate, $rate_only, $commission_type ) {

		if ( 'all_access' === $commission_type ) {
			$formatted_rate = $rate_only . '%';
		}

		return $formatted_rate;
	}

	/**
	 * Add "All Access" as a commission type
	 *
	 * @since       1.0.0
	 * @param       array $commission_types The types of commissions available to choose from.
	 * @return      array
	 */
	public function edd_all_access_commission_types( $commission_types ) {
		$commission_types['all_access'] = __( 'All Access Calculates Split', 'edd-all-access' );

		return $commission_types;
	}

	/**
	 * Hide All Access Split option by default if All Access is not enabled.
	 *
	 * @since       1.0.0
	 * @return      void
	 */
	public function edd_all_access_default_css() {

		if ( ! wp_style_is( 'eddc-admin-styles' ) ) {
			return;
		}
		wp_enqueue_style( 'admin-eddaa-commissions-integration', EDD_ALL_ACCESS_URL . 'assets/css/admin/build/commissions.css', array( 'eddc-admin-styles' ), EDD_ALL_ACCESS_VER );
	}

	/**
	 * Make sure we are at the minimum version of EDD Commissions - which is 3.3.
	 *
	 * @since       1.0.0
	 * @return      void
	 */
	public function edd_all_access_commissions_too_old_notice() {

		if ( ! $this->is_commissions_version_acceptable() ) {
			?>
			<div class="notice notice-error">
				<p><?php echo esc_html( __( 'EDD All Access: Your version of EDD Commissions must be updated to version 3.4 or later to use the All Access extension in conjunction with Commissions.', 'edd-all-access' ) ); ?></p>
			</div>
			<?php
		}
	}

	/**
	 * Enqueue the commissions javascript for All Access in wp-admin.
	 *
	 * @since       1.0.0
	 * @return      void
	 */
	public function admin_enqueue_scripts() {
		if ( wp_script_is( 'eddc-admin-scripts' ) ) {
			wp_enqueue_script( 'admin_edd_all_access_commissions_js', EDD_ALL_ACCESS_URL . 'assets/js/admin/build/edd-aa-admin-commissions.js', array( 'jquery', 'eddc-admin-scripts' ), true, EDD_ALL_ACCESS_VER );
		}
	}

	/**
	 * In case the site owner entered Commissions Recipients in the settings for the actuall All Access Product, strip them out.
	 *
	 * @since       1.0.0
	 * @param       array $recipients  The users getting commissions.
	 * @param       int   $download_id The product for which users are getting commissions.
	 * @return      array $recipients
	 */
	public function edd_all_access_strip_recipients_from_aa_products( $recipients, $download_id ) {

		if ( edd_all_access_enabled_for_download( $download_id ) ) {

			// Get the "Type" that commissions have been set to use for this All Access product.
			$commissions_meta_settings = get_post_meta( $download_id, '_edd_commission_settings', true );
			$type                      = isset( $commissions_meta_settings['type'] ) ? $commissions_meta_settings['type'] : 'percentage';

			// If this All Access product has commissions set to use the All Access calculation.
			if ( 'all_access' === $type ) {

				// Strip out the recipients (if any were entered).
				$recipients = array();
			}
		}

		return $recipients;
	}

	/**
	 * Register a tab in the single commission view for All Access information.
	 *
	 * @since  1.0.0
	 * @param  array $tabs An array of existing tabs.
	 * @return array       The altered list of tabs
	 */
	public function edd_all_access_commissions_tab( $tabs ) {

		// This makes it so former commission recievers get the tab and new commission users with no sales see it.
		$tabs['edd-all-access-info'] = array(
			'dashicon' => 'dashicons-welcome-widgets-menus',
			'title'    => __( 'All Access Info', 'edd-all-access' ),
		);

		return $tabs;
	}

	/**
	 * Register a view in the single commission view for All Access information.
	 *
	 * @since  1.0.0
	 * @param  array $views An array of existing views.
	 * @return array        The altered list of views
	 */
	public function edd_all_access_commissions_view( $views ) {

		$views['edd-all-access-info'] = 'edd_all_access_commissions_single_view';

		return $views;
	}

	/**
	 * Add a note to the Commissions metabox breify explaining how commissions are handled if downloaded via All Access.
	 *
	 * @since       1.0.0
	 * @return      void
	 */
	public function edd_all_access_commissions_post_meta() {
		?>
		<p class="description" data-edd-requires-commissions="true">
			<strong><?php esc_html_e( 'All Access Note: ', 'edd-all-access' ); ?></strong>
			<?php
			echo wp_kses_post(
				sprintf(
					/* translators: %1$s is the opening <a> tag, %2$s is the closing </a> tag */
					__( 'Regardless of the commissions settings for this product, if this product is downloaded using an "All Access" pass that is set to split Commissions using All Access calculations, commissions will be calculated based on the total value of all the products the customer chose to download. %1$sSee this doc for more information.%2$s', 'edd-all-access' ),
					'<a href="https://easydigitaldownloads.com/docs/all-access-extension-integrations/#edd-commissions" target="_blank" rel="noopener">',
					'</a>'
				)
			);
			?>
		</p>
		<?php
	}

	/**
	 * Triggered when an All Access payment is expired, here we calculate any commissions that need to be awarded for that All Access product.
	 *
	 * @since       1.0.0
	 * @param       EDD_All_Access_Pass $freshly_expired_all_access_pass The All Access Pass Object that has just expired.
	 * @param       array               $args The array of args that were passed to the maybe_expire method.
	 * @return      void
	 */
	public function edd_all_access_calculate_commissions( $freshly_expired_all_access_pass, $args ) {

		if ( ! $this->is_commissions_version_acceptable() ) {
			return;
		}

		// If the All Access Pass has been upgraded, don't calculate commissions here. Instead we'll calculate it when the upgraded-to one expires.
		if ( 'upgraded' === $freshly_expired_all_access_pass->status ) {

			// Leave a note on the payment letting it know commissions will be calculated on the upgraded payment.
			$activation_note = sprintf( __( 'Commissions note: This All Access Pass has been upgraded. If commissions are enabled for that upgraded pass, All Access commissions will be calculated when that upgraded pass expires.', 'edd-all-access' ), get_the_title( $freshly_expired_all_access_pass->download_id ) );
			edd_insert_payment_note( $freshly_expired_all_access_pass->order->id, $activation_note );

			return;
		}

		// Store the freshly_expired_all_access pass in an object variable so we can use it in other filter functions.
		$this->freshly_expired_all_access_pass = $freshly_expired_all_access_pass;

		// Actually record the commissions (but only All Access ones. Normal, non "All Access" commissions are handled through the EDD Commissions eddc_record_commission function.
		$this->edd_all_access_record_commissions( $freshly_expired_all_access_pass );
	}

	/**
	 * Make the price that is being split include any prior pass purchases. This is so that upgraded passes include all pre-upgrade monies.
	 *
	 * @since       1.0.0
	 * @param       int    $price The price being split for commissions.
	 * @param       object $cart_item The EDD cart object on the payment being split for commissions.
	 * @return      int
	 */
	public function edd_all_access_commissions_total_price( $price, $cart_item ) {

		// If there were no prior All Access Passes.
		if ( ! is_array( $this->freshly_expired_all_access_pass->prior_all_access_passes ) || empty( $this->freshly_expired_all_access_pass->prior_all_access_passes ) ) {
			return $price;
		}

		// Get the calc base setting for this store.
		$calc_base = edd_get_option( 'edd_commissions_calc_base', 'subtotal' );

		// Loop through all prior All Access Passes attached to the expiring & upgraded pass.
		foreach ( $this->freshly_expired_all_access_pass->prior_all_access_passes as $prior_all_access_pass_id ) {

			// Get this All Access Pass's payment id.
			$aa_data     = explode( '_', $prior_all_access_pass_id );
			$payment_id  = intval( $aa_data[0] );
			$download_id = intval( $aa_data[1] );
			$price_id    = intval( $aa_data[2] );

			$prior_payment = edd_get_payment( $payment_id );

			// Loop through each item that was in the cart during this prior purchase.
			foreach ( $prior_payment->cart_details as $prior_cart_item ) {

				$prior_cart_item_download_id = $prior_cart_item['id'];
				$prior_cart_item_price_id    = edd_get_cart_item_price_id( $prior_cart_item );
				$prior_cart_item_price_id    = empty( $prior_cart_item_price_id ) ? 0 : $prior_cart_item_price_id;

				// If the cart item matches the prior All Access Pass.
				if ( intval( $prior_cart_item_download_id ) === intval( $download_id ) && intval( $prior_cart_item_price_id ) === intval( $price_id ) ) {

					switch ( $calc_base ) {
						case 'subtotal':
							$prior_price = $prior_cart_item['subtotal'];
							break;

						case 'total_pre_tax':
							$prior_price = $prior_cart_item['price'] - $prior_cart_item['tax'];
							break;

						default:
							$prior_price = $prior_cart_item['price'];
							break;
					}

					// Add the prior price to the current payment's price.
					$price = $price + $prior_price;

				}
			}
		}

		return $price;
	}

	/**
	 * Prevent commissions from being generated at the time of payment for All Access products only.
	 *
	 * @since       1.0.0
	 * @param       array  $commissions_calculated The array of commissions calculated by the eddc_calculate_payment_commissions function in EDD Commissions.
	 * @param       object $payment The EDD_Payment object for which the commissions are being calculated.
	 * @return      array
	 */
	public function edd_all_access_prevent_aa_commissions( $commissions_calculated, $payment ) {

		// Loop through all of the generated commissions coming-in to this function.
		foreach ( $commissions_calculated as $key => $commission_calculated ) {

			// If we are dealing with a commission for an All Access item that was purchased.
			if ( edd_all_access_enabled_for_download( $commission_calculated['download_id'] ) ) {

				// Get the "Type" that commissions have been set to use for this All Access product.
				$commissions_meta_settings = get_post_meta( $commission_calculated['download_id'], '_edd_commission_settings', true );
				$type                      = isset( $commissions_meta_settings['type'] ) ? $commissions_meta_settings['type'] : 'percentage';

				// If this All Access product has commissions set to use the All Access calculation.
				if ( 'all_access' === $type ) {

					// Remove this commission from the normal Commissions array because it's incorrect now that All Access is taking over.
					unset( $commissions_calculated[ $key ] );
				}
			}
		}

		return $commissions_calculated;
	}

	/**
	 * Record Commissions for any expired All Access products in a payment.
	 * This is a forked version of the eddc_record_commission function in the EDD Commissions function.
	 * The reason this had to be forked is because the original function only fires if the $old_status was pending.
	 * It also didn't allow us to set a "Commissions Already Generated" flag for each item in the cart
	 * which we need for All Access since multiple All Access items can be in the same payment but could expire on different dates.
	 *
	 * @access      private
	 * @since       1.0.0
	 * @param       object $freshly_expired_all_access_pass The EDD_All_Access_Pass object for which we are recording commissions.
	 * @return      void
	 */
	public function edd_all_access_record_commissions( $freshly_expired_all_access_pass ) {

		// If commissions are not enabled for this All Access product, leave a payment note and don't do anything else.
		if ( ! get_post_meta( $freshly_expired_all_access_pass->download_id, '_edd_commisions_enabled', true ) ) {

			$expiration_note = __( 'No All Access Commissions needed to be created because the expiring All Access Pass did not have commissions enabled at the time of expiration.', 'edd-all-access' );
			edd_insert_payment_note( $freshly_expired_all_access_pass->payment_id, $expiration_note );

			return;
		}

		// Get the "Type" that commissions have been set to use for this All Access product.
		$commissions_meta_settings = get_post_meta( $freshly_expired_all_access_pass->download_id, '_edd_commission_settings', true );
		$type                      = isset( $commissions_meta_settings['type'] ) ? $commissions_meta_settings['type'] : 'percentage';

		// If commissions are not set to use the All Access calculation, leave a note on the payment so the site owner can easily understand what has happened.
		if ( 'all_access' !== $type ) {

			$expiration_note = __( 'No All Access Commissions needed to be created because the expiring All Access Pass was not set to use All Access commissions calculations. ', 'edd-all-access' );
			edd_insert_payment_note( $freshly_expired_all_access_pass->payment_id, $expiration_note );

			return;
		}

		// If this purchased All Access Pass is not expired, don't do anything else. This is just an additional safety check for expiration.
		if ( 'expired' !== $freshly_expired_all_access_pass->status ) {
			return;
		}

		// If this purchased All Access product has already had its commissions calculated.
		if ( edd_get_payment_meta( $freshly_expired_all_access_pass->payment_id, '_edd_aa_has_commission_' . $freshly_expired_all_access_pass->id, true ) ) {
			return;
		}

		// Calculate the recipients and their rates for this expired All Access purchase.
		$recipients_and_rates = array();
		$downloaded_products  = array();

		// Add this All Access Pass to the list of Payments we need to check for file downloads.
		$all_access_pass_payments_to_check = array( $freshly_expired_all_access_pass->id );

		// Check if this All Access Pass has been upgraded. If so, include the payments containing all "prior" All Access Passes as well.
		if ( is_array( $freshly_expired_all_access_pass->prior_all_access_passes ) ) {
			foreach ( $freshly_expired_all_access_pass->prior_all_access_passes as $prior_all_access_pass_id ) {

				// Add this All Access Pass ID to the list of All Access Payments we need to check for file download logs.
				$all_access_pass_payments_to_check[] = $prior_all_access_pass_id;

			}
		}

		// EDD 3.0
		if ( function_exists( 'edd_get_file_download_logs' ) ) {
			$file_download_logs = edd_get_file_download_logs(
				array(
					'number'     => 999999,
					'meta_query' => array( //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
						array(
							'key'   => '_edd_log_all_access_pass_id',
							'value' => $all_access_pass_payments_to_check,
						),
					),
				)
			);
		} else {
			global $edd_logs;
			$log_query          = array(
				'log_type'       => 'file_download',
				'posts_per_page' => -1,
				'meta_query'     => array( //phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
					'relation' => 'AND',
					array(
						'key'     => '_edd_log_all_access_pass_id',
						'value'   => $all_access_pass_payments_to_check,
						'compare' => 'IN',
					),
				),
			);
			$file_download_logs = $edd_logs->get_connected_logs( $log_query );
		}

		if ( empty( $file_download_logs ) ) {

			// If there were no files downloaded by the customer.
			$expiration_note = __( 'No All Access Commissions needed to be created because the customer did not download any files.', 'edd-all-access' );
			edd_insert_payment_note( $freshly_expired_all_access_pass->payment_id, $expiration_note );
			return;
		}

		$total_value_of_downloaded_items = 0;

		// We need to find out how much each downloaded item costs and add them together.
		foreach ( $file_download_logs as $log ) {

			// Get the post object for the product that was downloaded.
			$product_id = empty( $log->product_id ) ? $log->post_parent : $log->product_id;

			// If commissions are not enabled for this product, skip it.
			if ( ! get_post_meta( $product_id, '_edd_commisions_enabled', true ) ) {
				continue;
			}

			// Get the price of this product.
			$variable_prices = edd_get_variable_prices( $product_id );

			// Get the value of the product by getting its price - whether variably-priced or not.
			if ( $variable_prices ) {
				$price_id          = isset( $log->price_id ) ? $log->price_id : get_post_meta( $log->ID, '_edd_log_price_id', true );
				$price_of_download = edd_get_price_option_amount( $product_id, $price_id );
				$price_key         = $product_id . '-' . $price_id;
			} else {
				$price_id          = 0;
				$price_of_download = edd_get_download_price( $product_id );
				$price_key         = $product_id;
			}

			// Only calculate if we have not already calculated the value for a product. This happens if the user has downloaded it more than once.
			if ( ! array_key_exists( $price_key, $downloaded_products ) ) {
				$downloaded_products[ $price_key ] = array(
					'recipients'  => eddc_get_recipients( $product_id ), // Note: We get the recipient rate later using eddc_get_recipient_rate.
					'price'       => $price_of_download,
					'download_id' => $product_id,
					'price_id'    => $price_id,
				);

				// Add the price of this product to the total value of all downloaded items in this file download log.
				$total_value_of_downloaded_items = $total_value_of_downloaded_items + $price_of_download;
			}
		}

		if ( 0 === $total_value_of_downloaded_items ) {
			$expiration_note = __( 'No All Access Commissions needed to be created because none of the products the customer downloaded had a price higher than $0.', 'edd-all-access' );
			edd_insert_payment_note( $freshly_expired_all_access_pass->payment_id, $expiration_note );
			return;
		}

		// Now that we have the total value of all downloaded items, we need to calculate what percentage of the total value each download is.
		foreach ( $downloaded_products as $price_key => $product_info ) {

			$price_of_download = $product_info['price'];

			// The formula which allows us to see what percentage each product deserves of the All Access sale.
			$products_percentage_of_aa_total = round( 100 * ( $price_of_download / $total_value_of_downloaded_items ), 2 );

			// Loop through each recipient this product has set up in Commissions.
			foreach ( $product_info['recipients'] as $recipient_id ) {

				// Get the saved commission rate for this recipient from the purchased product's meta.
				$saved_recipient_commission_rate = apply_filters( 'edd_all_access_commission_rate_override', eddc_get_recipient_rate( $product_info['download_id'], $recipient_id ), $product_info['download_id'], $recipient_id );

				// This formula allows us to see what percentage each user deserves of the All Access sale.
				$users_percentage_of_aa_total = $products_percentage_of_aa_total * ( $saved_recipient_commission_rate / 100 );

				// If this recipient already has a percentage logged for a different product, add their new percentage to this one.
				if ( array_key_exists( $recipient_id, $recipients_and_rates ) ) {
					$recipients_and_rates[ $recipient_id ]['rate'] = $recipients_and_rates[ $recipient_id ]['rate'] + $users_percentage_of_aa_total;

				} else {
					// Otherwise, create a new recipient.
					$recipients_and_rates[ $recipient_id ]['rate'] = $users_percentage_of_aa_total;
				}

				// Log the product that was downloaded. This data could be used to prove to a reciever which of their products were downloaded and, thus, earned them a commission.
				$recipients_and_rates[ $recipient_id ]['downloaded_products'][ $product_info['download_id'] ][ $product_info['price_id'] ] = array(
					'weight_value' => $price_of_download,
					'saved_rate'   => $saved_recipient_commission_rate,
				);

				// Log the total "weight" or "value" of this All Access Pass.
				$recipients_and_rates[ $recipient_id ]['total_weight'] = $total_value_of_downloaded_items;

			}
		}

		// Get the calc base setting for this store.
		$calc_base = edd_get_option( 'edd_commissions_calc_base', 'subtotal' );

		// Now lets find the price of this All Access Pass.
		foreach ( $freshly_expired_all_access_pass->payment->cart_details as $cart_item ) {

			$cart_item_download_id = $cart_item['id'];
			$cart_item_price_id    = edd_get_cart_item_price_id( $cart_item );
			$cart_item_price_id    = empty( $cart_item_price_id ) ? 0 : $cart_item_price_id;

			// If the cart item matches the prior All Access Pass.
			if ( intval( $cart_item_download_id ) === intval( $freshly_expired_all_access_pass->download_id ) && intval( $cart_item_price_id ) === intval( $freshly_expired_all_access_pass->price_id ) ) {

				$all_access_cart_item = $cart_item;

				switch ( $calc_base ) {
					case 'subtotal':
						$price = $cart_item['subtotal'];
						break;

					case 'total_pre_tax':
						$price = $cart_item['price'] - $cart_item['tax'];
						break;

					default:
						$price = $cart_item['price'];
						break;
				}
			}
		}

		// Total Price including any prior payments (if this is an upgraded pass).
		$total_price = $this->edd_all_access_commissions_total_price( $price, $all_access_cart_item );

		// Add the recipients and rates to the commissions_calculated array for this All Access item which is now expired and hasn't had its commissions generated yet.
		foreach ( $recipients_and_rates as $recipient_user_id => $recipient_data ) {

			$commissions_calculated[] = array(
				'recipient'                          => $recipient_user_id,
				'commission_amount'                  => $total_price / ( 100 / $recipient_data['rate'] ),
				'rate'                               => $recipient_data['rate'],
				'download_id'                        => $freshly_expired_all_access_pass->download_id,
				'payment_id'                         => $freshly_expired_all_access_pass->payment_id,
				'currency'                           => $freshly_expired_all_access_pass->order->currency,
				'has_variable_prices'                => edd_has_variable_prices( $freshly_expired_all_access_pass->download_id ),
				'price_id'                           => $freshly_expired_all_access_pass->price_id,
				'variation'                          => isset( $variation ) ? $variation : null,
				'cart_item'                          => $all_access_cart_item,
				'products_downloaded_via_all_access' => $recipient_data['downloaded_products'],
				'all_access_pass_id'                 => $this->freshly_expired_all_access_pass->id,
				'prior_all_access_passes'            => $this->freshly_expired_all_access_pass->prior_all_access_passes,
				'all_access_total_price'             => $total_price,
				'all_access_total_weight'            => $recipient_data['total_weight'],
			);
		}

		$payment_meta = $freshly_expired_all_access_pass->payment->get_meta();
		$user_info    = maybe_unserialize( $payment_meta['user_info'] );

		// If there are no commission recipients set up, trigger an action and return.
		if ( empty( $commissions_calculated ) ) {
			do_action( 'eddc_no_commission_recipients', $freshly_expired_all_access_pass->payment_id );

			// Try and figure out why no commissions needed to be created and give the site owner a good message why.

			// If commissions are not enabled for the expiring product.
			if ( ! get_post_meta( $freshly_expired_all_access_pass->download_id, '_edd_commisions_enabled', true ) ) {

				$expiration_note = __( 'No All Access Commissions needed to be created because the expiring All Access Pass did not have commissions enabled at the time of expiration.', 'edd-all-access' );

			} elseif ( edd_get_payment_meta( $freshly_expired_all_access_pass->payment_id, '_edd_aa_has_commission_' . $freshly_expired_all_access_pass->id, true ) ) {

				// If, for some reason, commissions were already recorded for this payment.
				$expiration_note = __( 'No All Access Commissions needed to be created because commissions were already calculated for this payment.', 'edd-all-access' );

			} elseif ( $freshly_expired_all_access_pass->time_period_still_valid ) {

				// If, for some reason, we got here but the time period was still valid, note that. Unlikely, but maybe possible. Good for potential future tracking.
				$expiration_note = __( 'No All Access Commissions needed to be created because the time period was still valid for the All Access Pass.', 'edd-all-access' );

			} elseif ( empty( $file_download_logs ) ) {

				// If there were no files downloaded by the customer.
				$expiration_note = __( 'No All Access Commissions needed to be created because the customer did not download any files.', 'edd-all-access' );

			} else {

				// If files were downloaded by the customer but none of them had commissions enabled, we'd be here.
				// It's safe to assume that is the case without further checking.
				$expiration_note = __( 'No All Access Commissions needed to be created because commissions were not enabled for any of the products the customer chose to download.', 'edd-all-access' );

			}

			edd_insert_payment_note( $freshly_expired_all_access_pass->payment_id, $expiration_note );

			return;
		}

		// Loop through each calculated commission and award commissions.
		foreach ( $commissions_calculated as $commission_calculated ) {

			$default_commission_calculated = array(
				'recipient'                          => 0,
				'commission_amount'                  => 0,
				'rate'                               => 0,
				'download_id'                        => 0,
				'payment_id'                         => 0,
				'currency'                           => null,
				'has_variable_prices'                => null,
				'price_id'                           => null,
				'variation'                          => null,
				'cart_item'                          => null,
				'products_downloaded_via_all_access' => array(),
				'all_access_pass_id'                 => 0,
				'prior_all_access_passes'            => array(),
				'all_access_total_price'             => 0,
				'all_access_total_weight'            => 0,
			);

			$commission_calculated = wp_parse_args( $commission_calculated, $default_commission_calculated );

			$commission_calculated['download_id'] = absint( $commission_calculated['download_id'] );

			$commission_post_data = array(
				'post_type'   => 'edd_commission',
				'post_title'  => $user_info['email'] . ' - ' . get_the_title( $freshly_expired_all_access_pass->download_id ),
				'post_status' => 'publish',
			);

			// Record a new commission.
			$commission              = new EDD_Commission();
			$commission->description = $user_info['email'] . ' - ' . get_the_title( $freshly_expired_all_access_pass->download_id );
			$commission->status      = 'unpaid';
			$commission->user_id     = $commission_calculated['recipient'];
			$commission->rate        = $commission_calculated['rate'];
			$commission->amount      = $commission_calculated['commission_amount'];
			$commission->currency    = $commission_calculated['currency'];
			$commission->download_id = (int) $freshly_expired_all_access_pass->download_id;
			$commission->payment_id  = $freshly_expired_all_access_pass->payment_id;
			$commission->type        = eddc_get_commission_type( $commission_calculated['download_id'] );

			// If we are dealing with a variation, then save variation info.
			if ( $commission_calculated['has_variable_prices'] && ! empty( $commission_calculated['variation'] ) ) {
				$commission->download_variation = $commission_calculated['variation'];
			}

			// If it's a renewal, save that detail.
			if ( ! empty( $commission_calculated['cart_item']['item_number']['options']['is_renewal'] ) ) {
				$commission->is_renewal = true;
			}

			$commission->save();

			$args = array(
				'user_id'  => $commission->user_id,
				'rate'     => $commission->rate,
				'amount'   => $commission->amount,
				'currency' => $commission->currency,
				'type'     => $commission->type,
			);

			$commission_info = apply_filters( 'edd_commission_info', $args, $commission->id, $commission->payment_id, $commission->download_id );
			$items_changed   = false;
			foreach ( $commission_info as $key => $value ) {
				if ( $value === $args[ $key ] ) {
					continue;
				}

				$commission->$key = $value;
				$items_changed    = true;
			}

			if ( $items_changed ) {
				$commission->save();
			}

			do_action( 'eddc_insert_commission', $commission->user_id, $commission->amount, $commission->rate, $commission->download_id, $commission->id, $commission->payment_id );

			// Now we'll add our custom meta for All Access to the commission record.
			$all_access_info = array(
				'all_access_pass_id'                 => $commission_calculated['all_access_pass_id'],
				'products_downloaded_via_all_access' => $commission_calculated['products_downloaded_via_all_access'],
				'prior_all_access_passes'            => $commission_calculated['prior_all_access_passes'],
				'all_access_total_price'             => $commission_calculated['all_access_total_price'],
				'all_access_total_weight'            => $commission_calculated['all_access_total_weight'],
			);

			// Add some custom meta to tell which of this vendor/creator's products where downloaded by this customer.
			$commission->update_meta( '_edd_all_access_info', $all_access_info );

		}

		// Set a flag so we can be sure to not double-apply commissions for this All Access item in this payment.
		edd_update_payment_meta( $freshly_expired_all_access_pass->payment_id, '_edd_aa_has_commission_' . $freshly_expired_all_access_pass->id, true );
	}
}

/**
 * View All Access Data about a single Commission
 *
 * @since  3.5
 * @param  EDD_Commission $commission The commission object being displayed.
 * @return void
 */
function edd_all_access_commissions_single_view( $commission ) {
	if ( ! $commission ) {
		echo wp_kses_post( '<div class="info-wrapper item-section">' . __( 'Invalid commission specified.', 'edd-all-access' ) . '</div>' );
		return;
	}

	$base            = admin_url( 'edit.php?post_type=download&page=edd-commissions&view=overview&commission=' . $commission->ID );
	$base            = wp_nonce_url( $base, 'eddc_commission_nonce' );
	$commission_id   = $commission->ID;
	$all_access_info = $commission->get_meta( '_edd_all_access_info', true );

	// If there is no All Access Data attached to this commission, this is a non-All Access commission (a normal commission).
	if ( empty( $all_access_info ) ) {

		?>
		<div id="customer-tables-wrapper" class="customer-section">

		<h3><?php echo esc_html( __( 'All Access Pass Data', 'edd-all-access' ) ); ?></h3>

		<p><?php echo esc_html( __( 'This commission is not for an All Access product.', 'edd-all-access' ) ); ?></p>
		</div>

		<?php
		return;
	}

	$user_data           = get_userdata( $commission->user_id );
	$all_access_pass_id  = $all_access_info['all_access_pass_id'];
	$downloaded_products = $all_access_info['products_downloaded_via_all_access'];
	$total_price         = $all_access_info['all_access_total_price'];
	$total_price_display = edd_currency_filter( edd_format_amount( $total_price ) );
	$total_weight        = $all_access_info['all_access_total_weight'];

	// Get this All Access Pass's payment id.
	$aa_data         = explode( '_', $all_access_pass_id );
	$payment_id      = intval( $aa_data[0] );
	$download_id     = intval( $aa_data[1] );
	$price_id        = intval( $aa_data[2] );
	$all_access_pass = edd_all_access_get_pass( $payment_id, $download_id, $price_id );

	do_action( 'eddc_all_access_commission_card_top', $commission_id );
	?>
	<div id="customer-tables-wrapper" class="customer-section">

		<h3><?php echo esc_html( __( 'All Access Pass Data', 'edd-all-access' ) ); ?></h3>
		<table class="wp-list-table widefat striped downloads">
			<thead>
				<tr>
					<th><?php echo esc_html( __( 'Total Cost of All Access', 'edd-all-access' ) ); ?></th>
					<th><?php echo esc_html( __( 'Total Weight of All Access', 'edd-all-access' ) ); ?></th>
				</tr>
			</thead>
			<tbody>
				<tr>
					<td><?php echo esc_html( $total_price_display ); ?></td>
					<td><?php echo esc_html( $total_weight ); ?></td>
				</tr>
			</tbody>
		</table>

		<h3>
		<?php

		if ( 'invalid' === $all_access_pass->status ) {
			echo esc_html( $user_data->display_name . '\'s ' . __( 'products downloaded using All Access', 'edd-all-access' ) );
		} else {
			echo wp_kses_post( $user_data->display_name . '\'s ' . __( 'products downloaded by ', 'edd-all-access' ) . '<a href="' . esc_url( admin_url( 'edit.php?post_type=download&page=edd-customers&view=overview&id=' . $all_access_pass->customer->id . '#wpbody-content' ) ) . '">' . $all_access_pass->customer->name . '</a> ' . __( 'using', 'edd-all-access' ) . ' <a href="' . esc_url( admin_url( 'post.php?post=' . $all_access_pass->download_id . '&action=edit' ) ) . '">' . get_the_title( $all_access_pass->download_id ) . '</a>' );
		}
		?>
		</h3>
		<table class="wp-list-table widefat striped downloads">
			<thead>
				<tr>
					<th style="text-align:center;"><?php echo esc_html( __( 'Product', 'edd-all-access' ) ); ?></th>
					<th style="text-align:center;"><?php echo esc_html( __( 'Weight of Product', 'edd-all-access' ) ); ?></th>
					<th style="text-align:center;"><?php echo esc_html( __( 'Weight %', 'edd-all-access' ) ); ?></th>
					<th style="text-align:center;"><?php echo esc_html( __( 'Actual Value', 'edd-all-access' ) ); ?></th>
					<th style="text-align:center;"><?php echo esc_html( __( 'Commission', 'edd-all-access' ) ); ?></th>
				</tr>
			</thead>
			<tbody style="text-align:center;">
				<?php if ( ! empty( $downloaded_products ) ) : ?>
					<?php
					foreach ( $downloaded_products as $downloaded_product_id => $prices ) {
						foreach ( $prices as $price_id => $product_data ) {

							$weight_percentage = round( ( $product_data['weight_value'] / $total_weight * 100 ), 2 );
							$value             = ( $product_data['weight_value'] / $total_weight ) * $total_price;
							$value_display     = edd_currency_filter( edd_format_amount( $value ) );
							$earnings_display  = edd_currency_filter( edd_format_amount( round( ( $product_data['saved_rate'] / 100 ) * $value, 2 ) ) );

							$prices = apply_filters( 'edd_purchase_variable_prices', edd_get_variable_prices( $downloaded_product_id ), $downloaded_product_id );
							?>
								<tr>
									<td>
										<a href="<?php echo esc_url( admin_url( 'post.php?action=edit&post=' . $downloaded_product_id ) ); ?>"><?php echo esc_html( get_the_title( $downloaded_product_id ) ); ?></a>
										<p style="font-size:80%;"><?php echo esc_html( isset( $prices[ $price_id ] ) ? edd_get_price_option_name( $downloaded_product_id, $price_id ) : '' ); ?></p>
									</td>

									<td>
										<strong><?php echo esc_html( $product_data['weight_value'] ); ?></strong>
									</td>

									<td>
										<strong><?php echo esc_html( $weight_percentage . '%' ); ?></strong>
										<p style="font-size:60%;"><?php echo esc_html( $product_data['weight_value'] . '/' . $total_weight . ' = ' . $weight_percentage . '%' ); ?></p>
									</td>

									<td>
										<strong><?php echo esc_html( $value_display ); ?></strong>
										<p style="font-size:60%;"><?php echo esc_html( $weight_percentage . '% ' . __( 'of', 'edd-all-access' ) . ' ' . $total_price_display . ' = ' . $value_display ); ?></p>
									</td>

									<td>
										<strong><?php echo esc_html( $earnings_display ); ?></strong>
										<p style="font-size:60%;"><?php echo esc_html( $product_data['saved_rate'] . '% ' . __( 'of', 'edd-all-access' ) . ' ' . $value_display . ' = ' . $earnings_display ); ?></p>
									</td>
								</tr>
								<?php
						}
					}
				else :
					?>
					<tr><td colspan="2"><?php esc_html_e( 'No downloaded products found.', 'edd-all-access' ); ?></td></tr>
				<?php endif; ?>
			</tbody>
		</table>

	</div>

	<?php
	do_action( 'eddc_commission_all_access_card_bottom', $commission_id );
}
