<?php
/**
 * Class ESSB_OpenGraph_SEO
 *
 * This class is part of the EasySocialShareButtons package.
 * It provides functionality for SEO plugin integration for the social media tags.
 *
 * Usage example:
 * <code>
 * $instance = new ESSB_OpenGraph_SEO();
 * </code>
 *
 * @package    EasySocialShareButtons
 * @since      10.5.4
 */
class ESSB_OpenGraph_SEO
{

    private $meta_details = null;

    private $seo_plugin_bind = false;
    private $should_generate_image_tag = false;

    /**
     * Constructor for the class.
     *
     * Initializes the object with the provided meta data.
     *
     * @param ESSB_FrontMetaDetails $meta Meta data to be used for initialization.
     */
    public function __construct($meta)
    {
        $this->meta_details = $meta;

        add_action('template_redirect', array($this, 'check_for_integrations'), 1);
    }

    /**
     * Checks and handles integrations with other plugins or services for social share optimization.
     *
     * This method is responsible for detecting and managing compatibility or integration
     * with third-party plugins or services that may affect Open Graph SEO functionality.
     *
     */
    public function check_for_integrations()
    {
        if (defined('WPSEO_VERSION')) {
            $this->yoastseo_integration();
        }

        if (defined('AIOSEO_VERSION')) {
            $this->allinoneseo_integration();
        }

        if (defined('RANK_MATH_VERSION')) {
            $this->rankmath_integration();
        }

        if (defined('SEOPRESS_VERSION')) {
            $this->seopress_integration();
        }
    }

    /**
     * Checks if integration with other plugins or services is enabled.
     *
     * This method determines whether the current module has integration
     * capabilities with external plugins or services required for social
     * share optimization.
     *
     * @return bool Returns true if integration is available, false otherwise.
     */
    public function has_integration() {
        return $this->seo_plugin_bind;
    }

    /**
     * Ensures that an image meta tag is present for Open Graph SEO optimization.
     *
     * This method checks if the required image tag is set for social sharing purposes.
     * It may add or validate the image meta tag to improve sharing previews on social networks.
     *
     * @return bool
     */
    public function require_image_tag() {
        return $this->should_generate_image_tag;
    }

    // Add missing method to avoid undefined method error
    public function yoastseo_integration()
    {
        $this->seo_plugin_bind = true;

        $current_social_title = $this->meta_details->title();
        $current_social_description = $this->meta_details->description();
        $current_social_image = $this->meta_details->image();

        $current_social_description = trim(apply_filters('essb_opengraph_desc', $current_social_description));
        $current_social_title = trim(apply_filters('essb_opengraph_title', $current_social_title));
        $current_social_image = trim(apply_filters('essb_opengraph_image', $current_social_image));

        //replace title
        add_filter('wpseo_opengraph_title', function ($title) use ($current_social_title) {
            return !empty($current_social_title) ? $current_social_title : $title;
        }, 10, 1);

        //replace description
        add_filter('wpseo_opengraph_desc', function ($description) use ($current_social_description) {
            return !empty($current_social_description) ? $current_social_description : $description;
        }, 10, 1);

        //remove original image tags
        if (!empty($current_social_image)) {
            $this->should_generate_image_tag = true;
            //depcrecated
            add_filter('wpseo_opengraph_image', '__return_false');

            //current
            add_filter('wpseo_twitter_image', '__return_false');
            add_filter('wpseo_frontend_presenter_classes', function ($filter) {
                if (($key = array_search('Yoast\WP\SEO\Presenters\Open_Graph\Image_Presenter', $filter)) !== false) {
                    unset($filter[$key]);
                }
                return $filter;
            });
        }

        //replace twitter title
        add_filter('wpseo_twitter_title', function ($title) use ($current_social_title) {
            return !empty($current_social_title) ? $current_social_title : $title;
        }, 10, 1);

        //replace twitter description
        add_filter('wpseo_twitter_description', function ($description) use ($current_social_description) {
            return !empty($current_social_description) ? $current_social_description : $description;
        }, 10, 1);
    }

    // Add missing method to avoid undefined method error
    public function allinoneseo_integration()
    {
        $this->seo_plugin_bind = true;

        $current_social_title = $this->meta_details->title();
        $current_social_description = $this->meta_details->description();
        $current_social_image = $this->meta_details->image();

        $current_social_description = trim(apply_filters('essb_opengraph_desc', $current_social_description));
        $current_social_title = trim(apply_filters('essb_opengraph_title', $current_social_title));
        $current_social_image = trim(apply_filters('essb_opengraph_image', $current_social_image));

        if (!empty($current_social_image)) {
            $this->should_generate_image_tag = true;
        }

        add_filter('aioseo_facebook_tags', function ($meta) use ($current_social_title, $current_social_description, $current_social_image) {

            //og title
            if (isset($meta['og:title']) || !empty($current_social_title)) {
                $meta['og:title']  = !empty($current_social_title) ? $current_social_title : $meta['og:title'];
            }

            //og description
            if (isset($meta['og:description']) || !empty($current_social_description)) {
                $meta['og:description'] = !empty($current_social_description) ? $current_social_description : $meta['og:description'];
            }

            //remove original image tags
            if (!empty($current_social_image)) {
                unset($meta['og:image'], $meta['og:image:secure_url'], $meta['og:image:width'], $meta['og:image:height']);
            }

            return $meta;
        }, 10, 1);

        add_filter('aioseo_twitter_tags', function ($meta) use ($current_social_title, $current_social_description, $current_social_image) {

            //twitter title
            if (isset($meta['twitter:title']) || !empty($current_social_title)) {
                $meta['twitter:title'] = !empty($current_social_title) ? $current_social_title : $meta['twitter:title'];
            }

            //twitter description
            if (isset($meta['twitter:description']) || !empty($current_social_description)) {
                $meta['twitter:description'] = !empty($current_social_description) ? $current_social_description : $meta['twitter:description'];
            }

            //remove original image tag
            if (!empty($current_social_image)) {                
                unset($meta['twitter:image']);
            }

            return $meta;
        }, 10, 1);
    }

    // Add missing method to avoid undefined method error
    public function rankmath_integration()
    {
        $this->seo_plugin_bind = true;

        $current_social_title = $this->meta_details->title();
        $current_social_description = $this->meta_details->description();
        $current_social_image = $this->meta_details->image();

        $current_social_description = trim(apply_filters('essb_opengraph_desc', $current_social_description));
        $current_social_title = trim(apply_filters('essb_opengraph_title', $current_social_title));
        $current_social_image = trim(apply_filters('essb_opengraph_image', $current_social_image));

        add_filter("rank_math/opengraph/facebook/og_title", function($content) use ($current_social_title) {
			return !empty($current_social_title) ? $current_social_title : $content;
		});

		//facebook description
		add_filter("rank_math/opengraph/facebook/og_description", function($content) use ($current_social_description) {
			return !empty($current_social_description) ? $current_social_description : $content;
		});

		//remove original image tags
		if(!empty($current_social_image)) {
			add_filter('rank_math/opengraph/facebook/og_image', '__return_false');
			add_filter('rank_math/opengraph/facebook/og_image_secure_url', '__return_false');
			add_filter('rank_math/opengraph/facebook/og_image_width', '__return_false');
			add_filter('rank_math/opengraph/facebook/og_image_height', '__return_false');
			add_filter('rank_math/opengraph/facebook/og_image_alt', '__return_false');
			add_filter('rank_math/opengraph/facebook/og_image_type', '__return_false');
			add_filter('rank_math/opengraph/twitter/twitter_image', '__return_false');
            $this->should_generate_image_tag = true;
		}

		//twitter title
		add_filter("rank_math/opengraph/twitter/twitter_title", function($content) use ($current_social_title) {
			return !empty($current_social_title) ? $current_social_title : $content;
		});

		//twitter description
		add_filter("rank_math/opengraph/twitter/twitter_description", function($content) use ($current_social_description) {
			return !empty($current_social_description) ? $current_social_description : $content;
		});
    }

    // Add missing method to avoid undefined method error
    public function seopress_integration()
    {
        $this->seo_plugin_bind = true;

        $current_social_title = $this->meta_details->title();
        $current_social_description = $this->meta_details->description();
        $current_social_image = $this->meta_details->image();

        $current_social_description = trim(apply_filters('essb_opengraph_desc', $current_social_description));
        $current_social_title = trim(apply_filters('essb_opengraph_title', $current_social_title));
        $current_social_image = trim(apply_filters('essb_opengraph_image', $current_social_image));

        //replace title
		add_filter('seopress_social_og_title', function($html) use ($current_social_title) {
			if(!empty($current_social_title)) {
				$html = '<meta property="og:title" content="' . esc_attr(sanitize_text_field($current_social_title)) . '" />' . PHP_EOL;
			}
			return $html;
		}, 10, 1);

		//replace description
		add_filter('seopress_social_og_desc', function($html) use ($current_social_description) {
			if(!empty($current_social_description)) {
				$html = '<meta property="og:description" content="' . esc_attr(sanitize_text_field($current_social_description)) . '" />' . PHP_EOL;
			}
			return $html;
		}, 10, 1);

		//remove original image tags
		if(!empty($current_social_image)) {
            $this->should_generate_image_tag = true;
			add_filter('seopress_social_og_thumb', '__return_false');
			add_filter('seopress_social_twitter_card_thumb', '__return_false');
		}
    }
}
