<?php
/**
 * Exports Actions
 *
 * These are actions related to exporting data from Easy Digital Downloads.
 *
 * @package     EDD
 * @subpackage  Admin/Export
 * @copyright   Copyright (c) 2018, Easy Digital Downloads, LLC
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       2.4
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Process the download file generated by a batch export.
 *
 * @since 2.4
 * @throws \Exception If the exporter is not available.
 */
function edd_process_batch_export_download() {
	if ( ! wp_verify_nonce( $_REQUEST['nonce'], 'edd-batch-export' ) ) {
		wp_die( esc_html__( 'Nonce verification failed.', 'easy-digital-downloads' ), esc_html__( 'Error', 'easy-digital-downloads' ), array( 'response' => 403 ) );
	}

	\EDD\Admin\Exports\Loader::bootstrap();

	try {
		if ( ! empty( $_REQUEST['exporter_id'] ) ) {
			$exporter = \EDD\Admin\Exports\Registry::instance()->get_item( $_REQUEST['exporter_id'] );

			if ( ! empty( $exporter['class_path'] ) ) {
				require_once $exporter['class_path'];
			}

			$class = $exporter['class'];
		} else {
			$class = $_REQUEST['class'];
			do_action( 'edd_batch_export_class_include', $class );
		}
	} catch ( \Exception $e ) {
		wp_die( $e->getMessage(), esc_html__( 'Error', 'easy-digital-downloads' ), array( 'response' => 500 ) );
	}

	if ( ! class_exists( $class ) ) {
		wp_die( esc_html__( 'Exporter class not found.', 'easy-digital-downloads' ), esc_html__( 'Error', 'easy-digital-downloads' ), array( 'response' => 500 ) );
	}

	if ( is_subclass_of( $class, '\\EDD\\Admin\\Exports\\Exporters\\Exporter' ) || is_subclass_of( $class, '\\EDD\\Admin\\Exports\\Legacy\\BatchExport' ) ) {
		$export = new $class();
		$export->export();
	} else {
		wp_die( esc_html__( 'Exporter class not found.', 'easy-digital-downloads' ), esc_html__( 'Error', 'easy-digital-downloads' ), array( 'response' => 500 ) );
	}
}
add_action( 'edd_download_batch_export', 'edd_process_batch_export_download' );

/**
 * Export all the customers to a CSV file.
 *
 * Note: The WordPress Database API is being used directly for performance
 * reasons (workaround of calling all posts and fetch data respectively)
 *
 * @since 1.4.4
 * @return void
 */
function edd_export_all_customers() {
	require_once EDD_PLUGIN_DIR . 'includes/admin/reporting/class-export-customers.php';

	$customer_export = new EDD_Customers_Export();

	$customer_export->export();
}
add_action( 'edd_email_export', 'edd_export_all_customers' );

/**
 * Exports all the downloads to a CSV file using the EDD_Export class.
 *
 * @since 1.4.4
 * @return void
 */
function edd_export_all_downloads_history() {
	require_once EDD_PLUGIN_DIR . 'includes/admin/reporting/class-export-download-history.php';

	$file_download_export = new EDD_Download_History_Export();

	$file_download_export->export();
}
add_action( 'edd_downloads_history_export', 'edd_export_all_downloads_history' );

/**
 * Add a hook allowing extensions to register a hook on the batch export process
 *
 * @since  2.4.2
 * @return void
 */
function edd_register_batch_exporters() {
	if ( is_admin() ) {
		do_action( 'edd_register_batch_exporter' );
	}
}
add_action( 'plugins_loaded', 'edd_register_batch_exporters', 99 );

add_action(
	'edd_register_batch_exporter',
	function () {
		/**
		* Class Alias for the `EDD\Admin\Exports` classes.
		*
		* @deprecated 3.3.8
		*/
		class_alias( \EDD\Admin\Exports\Exporters\Earnings::class, 'EDD_Batch_Earnings_Report_Export' );
		class_alias( \EDD\Admin\Exports\Exporters\ProductSales::class, 'EDD_Batch_Sales_Export' );
		class_alias( \EDD\Admin\Exports\Exporters\SalesEarnings::class, 'EDD_Batch_Sales_Earnings_Export' );
		class_alias( \EDD\Admin\Exports\Exporters\Orders::class, 'EDD_Batch_Payments_Export' );
		class_alias( \EDD\Admin\Exports\Exporters\TaxedOrders::class, 'EDD_Batch_Taxed_Orders_Export' );
		class_alias( \EDD\Admin\Exports\Exporters\Customers::class, 'EDD_Batch_Customers_Export' );
		class_alias( \EDD\Admin\Exports\Exporters\TaxedCustomers::class, 'EDD_Batch_Taxed_Customers_Export' );
		class_alias( \EDD\Admin\Exports\Exporters\Downloads::class, 'EDD_Batch_Downloads_Export' );
		class_alias( \EDD\Admin\Exports\Exporters\APIRequests::class, 'EDD_Batch_API_Requests_Export' );
		class_alias( \EDD\Admin\Exports\Exporters\DownloadHistory::class, 'EDD_Batch_File_Downloads_Export' );
	}
);
