<?php
/**
 * Invoice handling for taxes.
 *
 * @package     EDD\Pro\Taxes\VAT
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Taxes\VAT;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * Invoices.
 *
 * @since 3.5.0
 */
class Invoices implements SubscriberInterface {

	/**
	 * Get the subscribed events.
	 *
	 * @since 3.5.0
	 * @return array
	 */
	public static function get_subscribed_events(): array {
		return array(
			'after_setup_theme'                   => 'disable_default_templates',
			'edd_invoices_invoice_contacts'       => 'invoice_contacts',
			'edd_invoices_after_company_details'  => 'add_company_details',
			'edd_invoices_after_customer_details' => 'add_customer_vat',
			'edd_invoices_invoice_items_table'    => array( 'add_additional_info', 15 ),
		);
	}

	/**
	 * Disable the default invoice contacts template.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	public function disable_default_templates() {
		remove_action( 'edd_invoices_invoice_contacts', 'edd_invoices_do_invoice_contacts' );
	}

	/**
	 * Outputs the invoice contacts template.
	 *
	 * @since 3.5.0
	 * @param \EDD\Orders\Order $order The order object.
	 * @return void
	 */
	public function invoice_contacts( $order ) {
		$template  = ! class_exists( '\\EDDInvoices' ) ? 'invoice/invoice-contacts' : 'invoice-contacts';
		$order_vat = Order::get( $order->id );
		if ( $order_vat->is_eu_payment ) {
			$template = 'invoice/eu-contacts';
		}

		edd_get_template_part( $template );
	}

	/**
	 * Add website url and email address to the "invoices from" column.
	 *
	 * @since 3.5.0
	 * @param \EDD\Orders\Order $order The order object.
	 * @return void
	 */
	public function add_company_details( $order ) {
		$show_website = edd_get_option( 'edd_vat_show_website_address', false );
		$show_email   = edd_get_option( 'edd_vat_company_email', '' );

		if ( ! $show_website && ! $show_email ) {
			return;
		}

		if ( ! empty( $show_email ) ) {
			?>
			<div class="storefront__email">
				<a href="mailto:<?php echo esc_html( antispambot( $show_email ) ); ?>"><?php echo esc_html( $show_email ); ?></a>
			</div>
			<?php
		}

		if ( $show_website ) {
			?>
			<div class="storefront__url">
				<a href="<?php echo esc_url( home_url() ); ?>"><?php echo esc_url( home_url() ); ?></a>
			</div>
			<?php
		}
	}

	/**
	 * Add customer vat details inside the invoice details.
	 *
	 * @since 3.5.0
	 * @param \EDD\Orders\Order $order The order object.
	 * @return void
	 */
	public function add_customer_vat( $order ) {
		$order_vat = Order::get( $order->id );
		if ( ! $order_vat->is_eu_payment ) {
			return;
		}

		$billed_to_vat_lines = array();

		if ( ! empty( $order_vat->vat_number ) ) {
			/* translators: %s is the VAT number */
			$billed_to_vat_lines[] = sprintf( __( 'VAT number: %s', 'easy-digital-downloads' ), esc_html( $order_vat->vat_number ) );

			if ( ! empty( $order_vat->name ) ) {
				/* translators: %s is the registered company name */
				$billed_to_vat_lines[] = sprintf( __( 'Company name: %s', 'easy-digital-downloads' ), esc_html( $order_vat->name ) );
			}
			if ( ! empty( $order_vat->address ) ) {
				/* translators: %s is the registered company address */
				$billed_to_vat_lines[] = sprintf( __( 'Address: %s', 'easy-digital-downloads' ), esc_html( $order_vat->address ) );
			}
		}

		if ( empty( $billed_to_vat_lines ) ) {
			return;
		}

		?>
		<p class="edd-eu-vat-receipt-customer-vat">
			<strong><?php esc_html_e( 'VAT Details', 'easy-digital-downloads' ); ?></strong><br/>
			<?php echo wp_kses( implode( '<br/>', $billed_to_vat_lines ), array( 'br' => array() ) ); ?>
		</p>
		<?php
	}

	/**
	 * Add additional information to invoice table.
	 *
	 * @since 3.5.0
	 * @param \EDD\Orders\Order $order The order object.
	 * @return void
	 */
	public function add_additional_info( $order ) {
		$order_vat = Order::get( $order->id );
		if ( ! $order_vat->is_eu_payment ) {
			return;
		}

		if ( ! $order_vat->is_reverse_charged && empty( (float) $order->tax ) ) {
			return;
		}

		?>
		<div class="invoice-element">
			<?php
			if ( $order_vat->is_reverse_charged ) {
				esc_html_e( 'VAT reverse charged', 'easy-digital-downloads' );
			} elseif ( ! empty( (float) $order->tax ) ) {
				printf(
					/* translators: %s is the VAT tax rate */
					esc_html__( 'VAT charged at %s%%', 'easy-digital-downloads' ),
					esc_html( $order->tax_rate )
				);
			}
			?>
		</div>
		<?php
	}
}
