<?php
/**
 * VAT information relating to an EDD order record.
 *
 * @package     EDD\Pro\Taxes\VAT
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Taxes\VAT;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Company VAT information.
 *
 * @since 3.5.0
 */
class Company {

	/**
	 * VAT number.
	 *
	 * @var string
	 */
	public $vat_number;

	/**
	 * UK VAT number.
	 *
	 * @var string
	 */
	public $uk_vat_number;

	/**
	 * Company name.
	 *
	 * @var string
	 */
	public $name;

	/**
	 * Address of the company.
	 *
	 * @var array
	 */
	public $address;

	/**
	 * Formatted address of the company.
	 *
	 * @var string
	 */
	public $formatted_address;

	/**
	 * Fetches all settings related to Company VAT.
	 *
	 * @since 3.5.0
	 * @return Company
	 */
	public static function get() {
		$company_vat = new self();

		$company_vat->vat_number    = edd_get_option( 'edd_vat_number' );
		$company_vat->uk_vat_number = edd_get_option( 'edd_uk_vat_number' );
		$company_vat->name          = edd_get_option( 'edd_vat_company_name' );

		$company_vat->address = array(
			'line1'   => edd_get_option( 'edd_vat_address_line_1' ),
			'line2'   => edd_get_option( 'edd_vat_address_line_2' ),
			'city'    => edd_get_option( 'edd_vat_address_city' ),
			'zip'     => edd_get_option( 'edd_vat_address_code' ),
			'country' => apply_filters( 'edd_vat_invoice_address_country_code', edd_get_option( 'edd_vat_address_invoice', '' ) ),
		);

		$company_vat->formatted_address = \EDD\Utils\Address::format_for_display( $company_vat->address );

		return $company_vat;
	}

	/**
	 * Get the company VAT number(s) for output.
	 *
	 * @since 3.5.0
	 * @return string
	 */
	public static function get_numbers() {
		$vat_numbers = array_filter(
			array(
				edd_get_option( 'edd_vat_number' ),
				edd_get_option( 'edd_uk_vat_number' ),
			)
		);

		return implode( ' / ', $vat_numbers );
	}
}
