<?php
/**
 * Downloads block.
 *
 * @package     EDD\Blocks\Pro
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.1
 */

namespace EDD\Pro\Blocks;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * Downloads block.
 *
 * @since 3.5.1
 */
class Downloads implements SubscriberInterface {

	/**
	 * Get subscribed events.
	 *
	 * @since 3.5.1
	 * @return array
	 */
	public static function get_subscribed_events(): array {
		return array(
			'edd_blocks_downloads_attributes' => 'filter_downloads_attributes',
			'edd_blocks_download_classes'     => array( 'filter_download_classes', 10, 3 ),
			'enqueue_block_editor_assets'     => 'enqueue',
		);
	}

	/**
	 * Filter downloads block attributes to add pro functionality.
	 *
	 * @since 3.5.1
	 * @param array $attributes Block attributes.
	 * @return array Modified attributes.
	 */
	public function filter_downloads_attributes( $attributes ): array {
		if ( empty( $attributes['featured_styling_enabled'] ) ) {
			return $attributes;
		}

		// Get button colors with fallbacks.
		$button_colors = edd_get_option( 'button_colors', array() );
		$button_colors = wp_parse_args(
			$button_colors,
			array(
				'background' => '#428bca',
				'text'       => '#ffffff',
			)
		);

		// Add pro-specific defaults for featured styling.
		$pro_defaults = array(
			'featured_styling_enabled'           => false,
			'featured_badge_enabled'             => false,
			'featured_border_enabled'            => false,
			'featured_badge_text'                => __( 'Featured', 'easy-digital-downloads' ),
			'featured_badge_font_size'           => '12px',
			'featuredBadgeColor'                 => '',
			'customFeaturedBadgeColor'           => $button_colors['text'],
			'featuredBadgeBackgroundColor'       => '',
			'customFeaturedBadgeBackgroundColor' => $button_colors['background'],
			'featured_border_color'              => $button_colors['background'],
			'featured_border_width'              => '2px',
			'featured_border_style'              => 'solid',
			'featured_border_radius'             => '3px',
		);

		$attributes = wp_parse_args( $attributes, $pro_defaults );

		$inline_styles = $this->add_inline_styles( $attributes );
		if ( ! empty( $inline_styles ) ) {
			$this->enqueue();
			wp_add_inline_style( 'edd-blocks-pro-downloads', $inline_styles );
		}

		return $attributes;
	}

	/**
	 * Filter download classes.
	 *
	 * @since 3.5.1
	 * @param array $classes Download classes.
	 * @param int   $post_id Download ID.
	 * @param array $attributes Block attributes.
	 * @return array Modified classes.
	 */
	public function filter_download_classes( $classes, $post_id, $attributes ) {
		if ( empty( $attributes['featured_styling_enabled'] ) ) {
			return $classes;
		}

		$classes[] = 'edd-blocks__download--styled';
		if ( ! empty( $attributes['featured_badge_enabled'] ) ) {
			$classes[] = 'edd-blocks__download--featured-badge';
		}

		if ( ! empty( $attributes['featured_border_enabled'] ) ) {
			$classes[] = 'edd-blocks__download--featured-border';
		}

		return $classes;
	}

	/**
	 * Enqueue the pro downloads styles.
	 *
	 * @since 3.5.1
	 * @return void
	 */
	public function enqueue() {
		if ( wp_style_is( 'edd-blocks-pro-downloads', 'enqueued' ) ) {
			return;
		}

		wp_enqueue_style( 'edd-blocks-pro-downloads', EDD_PLUGIN_URL . 'assets/pro/css/downloads.min.css', array( 'edd-blocks' ), EDD_VERSION );
	}

	/**
	 * Add inline styles for featured downloads.
	 *
	 * @since 3.5.1
	 * @param array $attributes Block attributes.
	 * @return string CSS styles for featured downloads.
	 */
	private function add_inline_styles( $attributes ) {
		if ( empty( $attributes['featured_styling_enabled'] ) ) {
			return '';
		}

		$style_parts = array();

		// Badge styling (only if badge is enabled).
		if ( ! empty( $attributes['featured_badge_enabled'] ) ) {
			$style_parts = array_merge( $style_parts, $this->get_badge( $attributes ) );
		}

		// Border styling (only if border is enabled).
		if ( ! empty( $attributes['featured_border_enabled'] ) ) {
			$style_parts = array_merge( $style_parts, $this->get_border( $attributes ) );
		}

		return ! empty( $style_parts ) ? '.wp-block-edd-downloads {' . implode( '; ', array_filter( $style_parts ) ) . '}' : '';
	}

	/**
	 * Get badge styles.
	 *
	 * @since 3.5.1
	 * @param array $attributes Block attributes.
	 * @return array CSS styles for badge.
	 */
	private function get_badge( $attributes ): array {
		$style_parts = array();
		// Badge text color - use palette color if available, otherwise custom color.
		$badge_color = '';
		if ( ! empty( $attributes['featuredBadgeColor'] ) ) {
			$badge_color = 'var( --wp--preset--color--' . esc_attr( $attributes['featuredBadgeColor'] ) . ' )';
		} elseif ( ! empty( $attributes['customFeaturedBadgeColor'] ) ) {
			$badge_color = esc_attr( $attributes['customFeaturedBadgeColor'] );
		}
		if ( $badge_color ) {
			$style_parts[] = '--edd-featured-badge-color: ' . $badge_color;
		}

		// Badge background color - use palette color if available, otherwise custom color.
		$badge_bg_color = '';
		if ( ! empty( $attributes['featuredBadgeBackgroundColor'] ) ) {
			$badge_bg_color = 'var( --wp--preset--color--' . esc_attr( $attributes['featuredBadgeBackgroundColor'] ) . ' )';
		} elseif ( ! empty( $attributes['customFeaturedBadgeBackgroundColor'] ) ) {
			$badge_bg_color = esc_attr( $attributes['customFeaturedBadgeBackgroundColor'] );
		}
		if ( $badge_bg_color ) {
			$style_parts[] = '--edd-featured-badge-bg: ' . $badge_bg_color;
		}

		// Badge text.
		if ( ! empty( $attributes['featured_badge_text'] ) ) {
			$style_parts[] = '--edd-featured-badge-text: "' . esc_js( $attributes['featured_badge_text'] ) . '"';
		}

		// Badge font size.
		if ( ! empty( $attributes['featured_badge_font_size'] ) ) {
			$style_parts[] = '--edd-featured-badge-font-size: ' . esc_attr( $attributes['featured_badge_font_size'] );
		}

		return $style_parts;
	}

	/**
	 * Get border styles.
	 *
	 * @since 3.5.1
	 * @param array $attributes Block attributes.
	 * @return array CSS styles for border.
	 */
	private function get_border( $attributes ) {
		$style_parts = array();

		// Border color.
		if ( ! empty( $attributes['featured_border_color'] ) ) {
			$style_parts[] = '--edd-featured-border-color: ' . esc_attr( $attributes['featured_border_color'] );
		}

		// Border width.
		if ( ! empty( $attributes['featured_border_width'] ) ) {
			$style_parts[] = '--edd-featured-border-width: ' . esc_attr( $attributes['featured_border_width'] );
		}

		// Border style.
		if ( ! empty( $attributes['featured_border_style'] ) ) {
			$style_parts[] = '--edd-featured-border-style: ' . esc_attr( $attributes['featured_border_style'] );
		}

		// Border radius.
		if ( ! empty( $attributes['featured_border_radius'] ) ) {
			$style_parts[] = '--edd-featured-border-radius: ' . esc_attr( $attributes['featured_border_radius'] );
		}

		return $style_parts;
	}
}
