<?php
/**
 * Set up VAT exporters.
 *
 * @package     EDD\Pro\Admin\Exports
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Admin\Exports;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codecoverage-ignore

use EDD\EventManagement\SubscriberInterface;
use EDD\Admin\Exports\Registry;

/**
 * VAT Class.
 *
 * @since 3.5.0
 */
class VAT implements SubscriberInterface {

	/**
	 * Gets the subscribed events.
	 *
	 * @since 3.5.0
	 * @return array
	 */
	public static function get_subscribed_events(): array {
		return array(
			'edd_export_init' => 'register_exporters',
		);
	}

	/**
	 * Registers the exporters.
	 *
	 * @since 3.5.0
	 * @param Registry $registry The registry instance.
	 * @return void
	 */
	public function register_exporters( Registry $registry ) {
		static $is_registered;
		if ( $is_registered ) {
			return;
		}
		try {
			foreach ( $this->get_exporters() as $exporter_id => $args ) {
				$registry->register_exporter( $exporter_id, $args );
			}
			$is_registered = true;
		} catch ( \Exception $e ) {
			error_log( $e->getMessage() );
		}
	}

	/**
	 * Gets the exporters.
	 *
	 * @since 3.5.0
	 * @return array
	 */
	private function get_exporters(): array {
		return array(
			'eu_vat'       => array(
				'label'       => __( 'EU VAT Report', 'easy-digital-downloads' ),
				'description' => __( 'Download a CSV of all European VAT collected, which you can use for your EU VAT/MOSS tax return.', 'easy-digital-downloads' ),
				'class'       => Exporters\EUVAT::class,
				'view'        => EDD_PLUGIN_DIR . 'pro/admin/views/export-eu-vat.php',
			),
			'eu_vat_sales' => array(
				'label'       => __( 'EC Sales List Report', 'easy-digital-downloads' ),
				'description' => __( 'Download a CSV of all European Sales to VAT-registered customers, which you can use for your EC Sales List tax return.', 'easy-digital-downloads' ),
				'class'       => Exporters\EUVATSales::class,
				'view'        => EDD_PLUGIN_DIR . 'pro/admin/views/export-eu-vat-sales.php',
			),
		);
	}
}
