<?php
/**
 * VAT Result class.
 *
 * @package     EDD\Pro\Taxes\VAT
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Taxes\VAT;

/**
 * Stores the result of a VAT number check.
 */
class Result {

	/**
	 * Error codes.
	 */
	const NO_VAT_NUMBER                  = 1;
	const NO_COUNTRY_CODE                = 2;
	const INVALID_VAT_NUMBER             = 3;
	const INVALID_COUNTRY_CODE           = 4;
	const VAT_NUMBER_INVALID_FOR_COUNTRY = 5;
	const INVALID_INPUT                  = 6;
	const API_ERROR                      = 7;
	const MS_MAX_CONCURRENT_REQ          = 8;

	/**
	 * The VAT number.
	 *
	 * @var string
	 */
	public $vat_number = '';

	/**
	 * The two letter country code (e.g. US).
	 *
	 * @var string
	 */
	public $country_code = '';

	/**
	 * If the VAT number is valid.
	 *
	 * @var bool true
	 */
	public $valid = false;

	/**
	 * The company name, if VAT number is valid.
	 *
	 * @var string
	 */
	public $name = '';

	/**
	 * The company address, if VAT number is valid.
	 *
	 * @var string
	 */
	public $address = '';

	/**
	 * Request Identifier parameter returned by the api
	 * if store VAT details are correct.
	 *
	 * @var string
	 */
	public $consultation_number = '';

	/**
	 * The error code if there was an error.
	 *
	 * @var string
	 */
	public $error = '';

	/**
	 * Constructs a new result object for the supplied VAT number and country code.
	 *
	 * @since 3.5.0
	 * @param string $vat_number   The VAT number the result applies to.
	 * @param string $country_code The two letter country code.
	 */
	public function __construct( $vat_number, $country_code ) {
		$this->vat_number   = $vat_number;
		$this->country_code = $country_code;
	}

	/**
	 * Is the VAT number valid?
	 *
	 * @since 3.5.0
	 * @return bool
	 */
	public function is_valid() {
		return (bool) $this->valid;
	}

	/**
	 * Convert result to string.
	 *
	 * @since 3.5.0
	 * @return string
	 */
	public function __toString() {
		$result = array( $this->name, $this->address, $this->error );

		return apply_filters( 'edd_vat_check_result_to_string', implode( "\r\n", array_filter( $result ) ), $this );
	}

	/**
	 * Reset the status of the result.
	 *
	 * @since 3.5.0
	 * @return self
	 */
	public function reset() {
		$this->valid = false;
		$this->error = '';

		return $this;
	}

	/**
	 * Gets an error message given an Result code.
	 *
	 * @since 3.5.0
	 * @param int $code The error code.
	 * @return string
	 */
	public static function get_message( $code ) {
		switch ( $code ) {
			case Result::NO_VAT_NUMBER:
				$error = __( 'Please enter a VAT number.', 'easy-digital-downloads' );
				break;

			case Result::NO_COUNTRY_CODE:
				$error = __( 'Please select a country.', 'easy-digital-downloads' );
				break;

			case Result::INVALID_VAT_NUMBER:
				$error = __( 'The VAT number is invalid.', 'easy-digital-downloads' );
				break;

			case Result::INVALID_COUNTRY_CODE:
				$error = __( 'The VAT number applies to EU countries only.', 'easy-digital-downloads' );
				break;

			case Result::VAT_NUMBER_INVALID_FOR_COUNTRY:
				$error = __( 'Your billing country must match the country for the VAT number.', 'easy-digital-downloads' );
				break;

			case Result::INVALID_INPUT:
				$error = __( 'The country or VAT number is invalid.', 'easy-digital-downloads' );
				break;

			case Result::API_ERROR:
				$error = __( 'We\'re having trouble checking your VAT number. Please try again or contact our support team.', 'easy-digital-downloads' );
				break;

			case Result::MS_MAX_CONCURRENT_REQ:
				$error = __( 'We\'re having trouble checking your VAT number. The VIES service is unreachable.', 'easy-digital-downloads' );
				break;

			default:
				$error = $code;
		}

		return apply_filters( 'edd_vat_error_code_to_string', $error, $code );
	}
}
