<?php
/**
 * EDD Service.
 *
 * @package     EDD\Pro\Taxes\Services
 * @copyright Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Taxes\Services;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Pro\Taxes\VAT\Result;

/**
 * EDD Service.
 */
class EDD extends Service {

	/**
	 * The pro license.
	 *
	 * @var \EDD\Licensing\License
	 */
	private $license;

	/**
	 * Constructor.
	 *
	 * @since 3.5.0
	 * @param Result $result The result.
	 */
	public function __construct( Result $result ) {
		parent::__construct( $result );

		$this->license = new \EDD\Licensing\License( 'pro' );
	}

	/**
	 * Checks if the service is available.
	 *
	 * @since 3.5.0
	 * @return bool
	 */
	public function requirements_met(): bool {
		if ( empty( $this->license->key ) ) {
			return false;
		}

		if ( $this->license->is_expired() ) {
			return false;
		}

		return 'valid' === $this->license->license;
	}

	/**
	 * Handles the request and returns the result.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	protected function validate(): void {
		$request = $this->make_request();
		if ( is_wp_error( $request->response ) ) {
			$this->result->error = Result::API_ERROR;

			return;
		}

		$response_body = json_decode( $request->body );

		edd_debug_log( 'VAT Sense request response: ' . print_r( $response_body, true ) );

		if ( 200 === $request->code && ! empty( $response_body->data->valid ) ) {
			$this->result->valid = true;

			if ( ! empty( $response_body->data->company->company_name ) ) {
				$this->result->name = $response_body->data->company->company_name;
			}

			if ( ! empty( $response_body->data->company->company_address ) ) {
				$this->result->address = implode( apply_filters( 'edd_vat_check_result_address_separator', ', ' ), array_filter( (array) $response_body->data->company->company_address ) );
			}

			return;
		}

		if ( ! empty( $response_body->consulation_number ) ) {
			$this->result->consultation_number = $response_body->consulation_number;
		}

		if ( ! empty( $response_body->error->detail ) ) {
			$this->result->error = $response_body->error->detail;
		} elseif ( ! empty( $response_body->error ) && is_string( $response_body->error ) ) {
			$this->result->error = $response_body->error;
		}
	}

	/**
	 * Makes the request.
	 *
	 * @since 3.5.0
	 * @return \EDD\Utils\RemoteRequest
	 */
	protected function make_request() {
		$args = $this->get_headers();
		if ( is_wp_error( $args ) ) {
			return $args;
		}

		return new \EDD\Utils\RemoteRequest(
			add_query_arg( $this->get_query_args(), $this->get_endpoint() ),
			$args
		);
	}

	/**
	 * Gets the endpoint.
	 *
	 * @since 3.5.0
	 * @return string
	 */
	protected function get_endpoint(): string {
		return 'https://services.easydigitaldownloads.com/validate-vat-number';
	}

	/**
	 * Gets the query args.
	 *
	 * @since 3.5.0
	 * @return array
	 */
	protected function get_query_args(): array {
		$query_args = array(
			'vat_number' => $this->vat_number,
		);

		// Add requester VAT number if provided and it's a UK VAT number.
		$requester_vat_number = $this->get_requester_number();
		if ( ! empty( $requester_vat_number ) ) {
			$query_args['requester_vat_number'] = $requester_vat_number;
		}

		return $query_args;
	}

	/**
	 * Gets the headers.
	 *
	 * @since 3.5.0
	 * @return array|\WP_Error
	 */
	protected function get_headers(): array {
		return array(
			'headers' => array(
				'X-License-Key' => $this->license->key,
			),
			'timeout' => 30,
		);
	}

	/**
	 * Gets the requester VAT number.
	 * This differs from the parent method in that it will return
	 * the EU VAT number if the customer's country is not GB.
	 *
	 * @since 3.5.0
	 * @return string
	 */
	protected function get_requester_number(): string {
		if ( 'GB' === $this->country_code ) {
			$requester_vat_number = edd_get_option( 'edd_uk_vat_number', '' );
			if ( ! empty( $requester_vat_number ) && false !== strpos( $requester_vat_number, 'GB' ) ) {
				return $requester_vat_number;
			}

			return '';
		}

		return edd_get_option( 'edd_vat_number', '' );
	}
}
