<?php
/**
 * Invoice Generator.
 *
 * @package     EDD\Pro\Invoices
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Invoices;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

/**
 * Class Generator
 *
 * @since 3.5.0
 */
class Generator {

	/**
	 * Holds any error messages.
	 *
	 * @var \WP_Error
	 * @since 3.5.0
	 */
	protected $errors;

	/**
	 * Order object for the current invoice.
	 *
	 * @var \EDD\Orders\Order|false
	 * @since 3.5.0
	 */
	public $order;

	/**
	 * EDD\Pro\Invoices\Generator constructor.
	 *
	 * @since 3.5.0
	 */
	public function __construct() {
		$this->errors = new \WP_Error();
	}

	/**
	 * Adds a new error.
	 *
	 * @param string $message Error message to display to the user.
	 * @param string $code    Internal error ID.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	protected function add_error( $message, $code = 'invalid_request' ) {
		$this->errors->add( $code, $message );
	}

	/**
	 * Returns an array of error messages.
	 *
	 * @since 3.5.0
	 * @return array
	 */
	public function get_error_messages() {
		return $this->errors->get_error_messages();
	}

	/**
	 * Validates the current request.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	public function validate_request() {
		try {
			$this->get_payment_from_request();
		} catch ( \Exception $e ) {
			$this->add_error( __( 'Invalid payment ID specified.', 'easy-digital-downloads' ), 'invalid_payment_id' );

			return;
		}

		if ( ! $this->current_user_can_view_invoice() ) {
			if ( empty( $this->get_error_messages() ) ) {
				$this->add_error( __( 'You do not have permission to view this invoice.', 'easy-digital-downloads' ), 'permission_denied' );
			}
		}
	}

	/**
	 * Whether or not the request is valid.
	 *
	 * If true, the user can proceed to view the invoice.
	 * If false, the user either doesn't have permission
	 *
	 * @since 3.5.0
	 * @return bool
	 */
	public function is_valid_request() {
		return ! $this->errors->has_errors();
	}

	/**
	 * Attempts to parse the payment ID from the URL.
	 *
	 * @since 3.5.0
	 * @return void
	 * @throws \Exception If the payment ID is invalid.
	 */
	private function get_payment_from_request() {
		$payment_id = false;

		if ( isset( $_GET['payment_key'] ) ) {
			$payment_id = edd_get_purchase_id_by_key( urldecode( $_GET['payment_key'] ) );
		}

		if ( isset( $_GET['payment_id'] ) ) {
			$payment_id = urldecode( $_GET['payment_id'] );
		}

		if ( empty( $payment_id ) ) {
			throw new \Exception();
		}

		$this->order = edd_get_order( intval( $payment_id ) );

		if ( empty( $this->order->id ) || (int) $this->order->id !== (int) $payment_id ) {
			throw new \Exception();
		}
	}

	/**
	 * Determines whether or not the current user can view the current invoice.
	 *
	 * @since 3.5.0
	 * @return bool
	 */
	private function current_user_can_view_invoice() {

		if ( ! $this->validate_uri() ) {
			return false;
		}

		// If the user can view sensitive shop data, they can view the invoice.
		if ( current_user_can( 'view_shop_sensitive_data' ) ) {
			return true;
		}

		$user_id = (int) $this->order->user_id;

		// Current user is the owner of the payment.
		if ( is_user_logged_in() && (int) get_current_user_id() === $user_id ) {
			return true;
		}

		return $this->guest_can_view_invoice();
	}

	/**
	 * Validates the URI to ensure that the invoice and order match.
	 *
	 * @since 3.5.0
	 * @return bool
	 */
	private function validate_uri() {
		// Check that the invoice hash is present and the order ids match.
		$id         = (int) filter_input( INPUT_GET, 'payment_id', FILTER_SANITIZE_NUMBER_INT );
		$order_hash = filter_input( INPUT_GET, 'invoice', FILTER_SANITIZE_SPECIAL_CHARS );
		if ( ! $order_hash || $id !== (int) $this->order->id ) {
			return false;
		}

		// Check that the hash is valid.
		return hash_equals( md5( $this->order->id . $this->order->payment_key . $this->order->email ), $order_hash );
	}

	/**
	 * Determines whether or not a guest can view the current invoice.
	 *
	 * @since 3.5.0
	 * @return bool
	 */
	private function guest_can_view_invoice() {
		if ( is_user_logged_in() ) {
			return false;
		}

		// If the purchase session is active, the guest can view the invoice.
		if ( edd_get_purchase_session() ) {
			return true;
		}

		// Allow guests to view the invoice if the order is not associated with a user.
		if ( empty( $this->order->user_id ) ) {
			return true;
		}

		// Otherwise, force the user to log in to view their invoice.
		$login_uri = edd_get_login_page_uri();
		if ( ! $login_uri ) {
			$login_uri = wp_login_url();
		}
		$this->add_error(
			sprintf(
			/* translators: %1$s Opening anchor tag, do not translate. %2$s Closing anchor tag, do not translate. */
				__( 'You must %1$slog in%2$s to view this invoice.', 'easy-digital-downloads' ),
				'<a href="' . esc_url( $login_uri ) . '">',
				'</a>'
			),
			'login_required'
		);

		return false;
	}

	/**
	 * Saves billing information from the request.
	 *
	 * @since 3.5.0
	 * @return void
	 * @throws \Exception If the billing information could not be saved.
	 */
	public function save_billing_information() {
		if ( empty( $this->order->id ) ) {
			throw new \Exception( __( 'Missing payment ID.', 'easy-digital-downloads' ) );
		}

		$address                = array_map( 'sanitize_text_field', $_POST['edd-payment-address'] );
		$address['name']        = ! empty( $_POST['edd-payment-user-name'] ) ? sanitize_text_field( $_POST['edd-payment-user-name'] ) : '';
		$address['address']     = $address['line1'];
		$address['address2']    = $address['line2'];
		$address['region']      = $address['state'];
		$address['postal_code'] = $address['zip'];
		if ( ! empty( $_POST['address-id'] ) ) {
			edd_update_order_address( absint( $_POST['address-id'] ), $address );
		} else {
			$address['order_id'] = $this->order->id;
			edd_add_order_address( $address );
		}

		$meta = array(
			'edd-payment-vat'     => 'invoices_vat',
			'edd-payment-notes'   => 'invoices_notes',
			'edd-payment-company' => 'invoices_company',
		);

		foreach ( $meta as $form_field => $meta_key ) {
			if ( ! empty( $_POST[ $form_field ] ) ) {
				edd_update_order_meta( $this->order->id, $meta_key, sanitize_text_field( $_POST[ $form_field ] ) );
			} else {
				edd_delete_order_meta( $this->order->id, $meta_key );
			}
		}
	}
}
