<?php
/**
 * Pro Empty Cart Behavior Loader
 *
 * @package     EDD\Pro\Checkout\EmptyCart
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Checkout;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

use EDD\EventManagement\SubscriberInterface;

/**
 * Pro Empty Cart Behavior Loader class.
 *
 * @since 3.5.0
 */
class EmptyCart implements SubscriberInterface {

	/**
	 * Returns an array of events that this subscriber wants to listen to.
	 *
	 * @since 3.5.0
	 * @return array
	 */
	public static function get_subscribed_events() {
		return array(
			'template_redirect'            => 'handle_empty_cart',
			'edd_empty_cart_redirect_page' => 'handle_empty_cart_redirect_page',
			'edd_empty_cart_redirect_url'  => 'handle_empty_cart_redirect_url',
		);
	}

	/**
	 * Handle empty cart behavior on checkout page.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	public function handle_empty_cart() {

		// Only run on checkout page.
		if ( ! edd_is_checkout() ) {
			return;
		}

		// Only run if cart is empty.
		if ( ! empty( edd_get_cart_contents() ) ) {
			return;
		}

		// Get the behavior setting.
		$behavior = edd_get_option( 'empty_cart_behavior', 'message' );

		// If the behavior is set to message, do nothing.
		if ( 'message' === $behavior ) {
			return;
		}

		/**
		 * Handle our redirect behavior.
		 *
		 * This allows us to use a single action to handle all redirect behaviors.
		 *
		 * @since 3.5.0
		 * @param string $behavior The behavior to handle.
		 * @return void
		 */
		do_action( 'edd_empty_cart_' . $behavior );
	}

	/**
	 * Handle the redirect to a page.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	public function handle_empty_cart_redirect_page() {
		$page_id = absint( edd_get_option( 'empty_cart_redirect_page', 0 ) );
		if ( $page_id ) {
			$page_url = get_permalink( $page_id );
			if ( $page_url ) {
				// If a permalink was found, redirect to it.
				wp_safe_redirect( $page_url );
				exit;
			}

			// If we did not find a permalink, use the standard message.
			return;
		}
	}

	/**
	 * Handle the redirect to a URL.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	public function handle_empty_cart_redirect_url() {
		$redirect_url = edd_get_option( 'empty_cart_redirect_url', '' );

		if ( empty( $redirect_url ) ) {
			return;
		}

		$redirect_host = wp_parse_url( $redirect_url, PHP_URL_HOST );

		// If we failed to parse a host, use the standard message.
		if ( empty( $redirect_host ) ) {
			return;
		}

		/**
		 * wp_safe_redirect() will not work for external URLs unless
		 * we add the host to the allowed hosts list.
		 *
		 * @see https://developer.wordpress.org/reference/functions/wp_safe_redirect/
		 */
		add_filter(
			'allowed_redirect_hosts',
			function ( $hosts ) use ( $redirect_host ) {
				$hosts[] = $redirect_host;
				return $hosts;
			}
		);

		wp_safe_redirect( $redirect_url );
		exit;
	}
}
