<?php
/**
 * Receipt class.
 *
 * @package     EDD\Pro\Taxes\VAT
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Taxes\VAT;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * Receipt class.
 */
class Receipt implements SubscriberInterface {

	/**
	 * Get the subscribed events.
	 *
	 * @since 3.5.0
	 * @return array
	 */
	public static function get_subscribed_events(): array {
		return array(
			'edd_order_receipt_after_table' => 'render',
		);
	}

	/**
	 * Show VAT on the purchase receipt.
	 *
	 * @since 3.5.0
	 * @param \EDD\Orders\Order $order The order object.
	 */
	public function render( $order ) {
		if ( ! edd_get_option( 'edd_vat_purchase_receipt', false ) ) {
			return;
		}

		$vat_details = $this->get_vat_details( $order );
		if ( empty( $vat_details ) ) {
			return;
		}

		$customer_details = $this->get_customer_details( $order );
		$company_details  = $this->get_company_details( $order );

		if ( empty( $customer_details ) && empty( $company_details ) ) {
			return;
		}

		?>
		<div class="edd-receipt-vat">
			<?php
			if ( ! empty( $customer_details ) ) {
				?>
				<div class="edd-receipt-vat__customer">
					<h3><?php echo esc_html( apply_filters( 'edd_vat_purchase_receipt_bill_to_heading', __( 'Billed To', 'easy-digital-downloads' ) ) ); ?></h3>
					<p class="edd-receipt-vat__customer-address">
					<?php echo wp_kses( implode( '<br/>', $customer_details ), array( 'br' => array() ) ); ?></p>

					<?php if ( $vat_details ) : ?>
						<p class="edd-receipt-vat__customer-details">
							<strong><?php esc_html_e( 'VAT Details', 'easy-digital-downloads' ); ?></strong><br/>
							<?php echo wp_kses( implode( '<br/>', $vat_details ), array( 'br' => array() ) ); ?>
						</p>
					<?php endif; ?>
				</div>
				<?php
			}

			if ( ! empty( $company_details ) ) {
				?>
				<div class="edd-receipt-vat__company">
					<h3><?php echo esc_html( apply_filters( 'edd_vat_purchase_receipt_sold_by_heading', __( 'Sold By', 'easy-digital-downloads' ) ) ); ?></h3>
					<p><?php echo wp_kses( implode( '<br/>', $company_details ), array( 'br' => array() ) ); ?></p>
				</div>
				<?php
			}

			?>
		</div>
		<?php
	}

	/**
	 * Gets the customer details.
	 *
	 * @param \EDD\Orders\Order $order The order object.
	 * @return array
	 */
	private function get_customer_details( $order ): array {
		$customer_details = array();

		$customer = edd_get_customer( $order->customer_id );
		if ( $customer ) {
			$customer_details[] = $customer->name;
		}

		$customer_address = \EDD\Utils\Address::format_for_display( $order->get_address() );
		if ( $customer_address ) {
			$customer_details[] = nl2br( $customer_address );
		}

		return $customer_details;
	}

	/**
	 * The VAT details for the order.
	 *
	 * @since 3.5.0
	 * @param \EDD\Orders\Order $order The order object.
	 * @return array
	 */
	private function get_vat_details( $order ) {
		$vat_details = array();
		$order_vat   = Order::get( $order->id );
		if ( ! $order_vat->is_eu_payment || empty( $order_vat->vat_number ) ) {
			return $vat_details;
		}

		/* translators: %s is the VAT number */
		$vat_details[] = sprintf( __( 'VAT number: %s', 'easy-digital-downloads' ), esc_html( $order_vat->vat_number ) );

		if ( ! empty( $order_vat->name ) ) {
			/* translators: %s is the registered company name */
			$vat_details[] = sprintf( __( 'Company name: %s', 'easy-digital-downloads' ), esc_html( $order_vat->name ) );
		}
		if ( ! empty( $order_vat->address ) ) {
			/* translators: %s is the registered company address */
			$vat_details[] = sprintf( __( 'Address: %s', 'easy-digital-downloads' ), esc_html( $order_vat->address ) );
		}

		return $vat_details;
	}

	/**
	 * Get the company details for the order.
	 *
	 * @since 3.5.0
	 * @param \EDD\Orders\Order $order The order object.
	 * @return array
	 */
	private function get_company_details( $order ) {
		$company_details = array();
		$company         = Company::get();

		if ( ! empty( $company->name ) ) {
			$company_details[] = $company->name;
		}

		if ( ! empty( $company->formatted_address ) ) {
			$company_details[] = nl2br( $company->formatted_address );
		}

		if ( ! empty( $company->vat_number ) ) {
			/* translators: %s is the VAT number */
			$company_details[] = sprintf( __( 'EU VAT number: %s', 'easy-digital-downloads' ), $company->vat_number );
		}

		if ( ! empty( $company->uk_vat_number ) ) {
			/* translators: %s is the VAT number */
			$company_details[] = sprintf( __( 'UK VAT number: %s', 'easy-digital-downloads' ), $company->uk_vat_number );
		}

		return $company_details;
	}
}
