<?php
/**
 * Abstract Service class for VAT services.
 *
 * @package     EDD\Pro\Taxes\Services
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Taxes\Services;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Pro\Taxes\VAT\Result;

/**
 * Service.
 */
abstract class Service {

	/**
	 * Result.
	 *
	 * @var Result
	 */
	protected $result;

	/**
	 * VAT number.
	 *
	 * @var string
	 */
	protected $vat_number;

	/**
	 * The country code.
	 *
	 * @var string
	 */
	protected $country_code;

	/**
	 * Constructor.
	 *
	 * @since 3.5.0
	 * @param Result $result The result.
	 */
	public function __construct( Result $result ) {
		$this->result       = $result;
		$this->vat_number   = $this->sanitize_vat_number( $this->result->vat_number );
		$this->country_code = $this->result->country_code;
	}

	/**
	 * Gets the result.
	 *
	 * @since 3.5.0
	 * @return Result
	 */
	public function get_result(): Result {
		$this->validate();

		return $this->result;
	}

	/**
	 * Sanitizes the VAT number.
	 *
	 * @since 3.5.0
	 * @param string $vat_number The VAT number.
	 * @return string
	 */
	public function sanitize_vat_number( string $vat_number ): string {
		return str_replace( array( ' ', '.', '-' ), '', strtoupper( $vat_number ) );
	}

	/**
	 * Checks if the service is available.
	 *
	 * @since 3.5.0
	 * @return bool
	 */
	abstract public function requirements_met(): bool;

	/**
	 * Handles the request and returns the result.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	abstract protected function validate(): void;

	/**
	 * Gets the endpoint.
	 *
	 * @since 3.5.0
	 * @return string
	 */
	abstract protected function get_endpoint(): string;
}
