<?php
/**
 * Loader for Invoices functionality.
 *
 * @package     EDD\Pro\Invoices
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Invoices;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;
use EDD\EventManagement\EventManager;

/**
 * Class Loader
 *
 * @since 3.5.0
 */
class Loader implements SubscriberInterface {

	/**
	 * Get the events to subscribe to.
	 *
	 * @since 3.5.0
	 * @return array
	 */
	public static function get_subscribed_events() {
		if ( class_exists( '\\EDDInvoices' ) ) {
			return array();
		}

		return array(
			'plugins_loaded'    => 'add_events',
			'after_setup_theme' => 'include_files',
			'admin_init'        => 'include_admin_files',
		);
	}

	/**
	 * Add the events.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	public function add_events() {
		$invoices_classes = array();
		if ( is_admin() ) {
			$invoices_classes[] = new Admin\Bulk();
		}
		$events = new EventManager();
		foreach ( $invoices_classes as $invoices_class ) {
			$events->add_subscriber( $invoices_class );
		}
	}

	/**
	 * Include the files.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	public function include_files() {
		$files = array(
			'actions',
			'emails',
			'functions',
			'shortcodes',
			'templates',
		);

		foreach ( $files as $file ) {
			require_once EDD_PLUGIN_DIR . "pro/invoices/{$file}.php";
		}
	}

	/**
	 * Include the admin files.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	public function include_admin_files() {
		require_once EDD_PLUGIN_DIR . 'pro/admin/invoices.php';
	}
}
