<?php
/**
 * VAT Class.
 *
 * @package     EDD\Pro\Forms\Checkout
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Forms\Checkout;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\Forms\Fields\FieldInterface;

/**
 * The VAT field.
 *
 * @since 3.5.0
 */
class VAT implements FieldInterface {

	/**
	 * The VAT details.
	 *
	 * @var \EDD\Taxes\VAT\Result
	 */
	private $vat_details;

	/**
	 * Constructor.
	 *
	 * @since 3.5.0
	 * @param \EDD\Taxes\VAT\Result $vat_details The VAT details.
	 */
	public function __construct( $vat_details ) {
		$this->vat_details = $vat_details;
	}

	/**
	 * Render the field.
	 *
	 * @since 3.5.0
	 * @return void
	 */
	public function render(): void {
		$classes = array( 'edd-card-vat-wrap' );
		if ( \EDD\Checkout\Validator::has_block() ) {
			$classes[] = 'edd-blocks-form__group';
			$classes[] = 'edd-blocks-form__group-' . $this->get_key();
		}
		?>
		<div id="edd-card-vat-wrap" class="<?php echo esc_attr( $this->get_css_class_string( $classes ) ); ?>" style="display: none;">
			<?php
			$this->do_label();
			$this->do_description();
			$this->do_input();
			?>
		</div>
		<?php
	}

	/**
	 * Render the input.
	 *
	 * @since 3.5.0
	 */
	public function do_input(): void {
		?>
		<div class="edd-vat-number-wrap">
			<?php
			$input = new \EDD\HTML\Text(
				wp_parse_args(
					array(
						'placeholder' => apply_filters( 'edd_vat_checkout_vat_number_placeholder', __( 'e.g. DE123456789', 'easy-digital-downloads' ) ),
						'value'       => ! empty( $this->data->vat_number ) ? $this->data->vat_number : '',
					),
					array(
						'name'         => $this->get_key(),
						'id'           => $this->get_id(),
						'class'        => $this->get_css_class_string( array( 'edd-input', 'edd-vat-number-input' ) ),
						'required'     => edd_field_is_required( $this->get_id() ),
						'include_span' => false,
					)
				)
			);
			$input->output();
			?>
			<div class="edd-vat-check__control">
				<button type="button" name="edd-vat-check" id="edd-vat-check-button" class="button edd-vat-check-button"><?php esc_attr_e( 'Validate', 'easy-digital-downloads' ); ?></button>
			</div>
		</div>
		<?php
	}

	/**
	 * Get the field ID.
	 *
	 * @since 3.5.0
	 * @return string
	 */
	public function get_id(): string {
		return 'edd-vat-number';
	}

	/** Get the field label.
	 *
	 * @since 3.5.0
	 * @return string
	 */
	public function get_label(): string {
		return esc_html__( 'VAT Number', 'easy-digital-downloads' );
	}

	/**
	 * Get the description.
	 *
	 * @since 3.5.0
	 * @return string
	 */
	public function get_description(): string {
		return apply_filters( 'edd_vat_checkout_vat_field_description', __( 'Enter the VAT number of your company.', 'easy-digital-downloads' ) );
	}

	/**
	 * Renders the label.
	 *
	 * @since 3.5.0
	 */
	protected function do_label(): void {
		?>
		<label for="<?php echo esc_attr( $this->get_id() ); ?>" class="edd-label">
			<?php
			echo esc_html( $this->get_label() );
			if ( edd_field_is_required( $this->get_id() ) ) {
				echo EDD()->html->show_required();
			}
			?>
		</label>
		<?php
	}

	/**
	 * Display a description.
	 *
	 * @since 3.5.0
	 */
	protected function do_description(): void {
		if ( empty( $this->get_description() ) || \EDD\Checkout\Validator::has_block() ) {
			return;
		}
		?>
		<p class="edd-description"><?php echo esc_html( $this->get_description() ); ?></p>
		<?php
	}

	/**
	 * Get the field key.
	 *
	 * @since 3.5.0
	 * @return string
	 */
	protected function get_key(): string {
		return 'vat_number';
	}

	/**
	 * Get the CSS class string.
	 *
	 * @since 3.5.0
	 * @param array $classes The classes.
	 * @return string
	 */
	protected function get_css_class_string( $classes = array() ): string {
		return implode( ' ', array_unique( array_filter( $classes ) ) );
	}
}
