<?php
/**
 * Invoices settings.
 *
 * @package     EDD\Pro\Admin\Settings
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Admin\Settings;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;

/**
 * Class Invoices
 *
 * @since 3.5.0
 */
class Invoices implements SubscriberInterface {
	/**
	 * Returns an array of events that this subscriber wants to listen to.
	 *
	 * @return array
	 */
	public static function get_subscribed_events() {
		return array(
			'edd_settings_sections_gateways' => 'register_section',
			'edd_settings_gateways'          => 'register_settings',
		);
	}

	/**
	 * Registers the Invoices settings section.
	 *
	 * @since 3.5.0
	 * @param array $sections Existing settings sections.
	 * @return array
	 */
	public function register_section( $sections ) {
		$sections['edd-invoices'] = __( 'Invoices', 'easy-digital-downloads' );

		return $sections;
	}

	/**
	 * Registers the Invoices settings.
	 *
	 * @since 3.5.0
	 * @param array $settings Existing settings.
	 * @return array
	 */
	public function register_settings( $settings ) {
		$invoice_settings = array(
			'edd-invoices-disable-editing'      => array(
				'id'    => 'edd-invoices-disable-editing',
				'name'  => __( 'Disable Address Editing?', 'easy-digital-downloads' ),
				'check' => __( 'Do not allow customers to edit the invoice.', 'easy-digital-downloads' ),
				'type'  => 'checkbox_toggle',
				'desc'  => sprintf(
					/* translators: %s: manage_shop_settings capability */
					__( 'Prevent customers from changing their billing information (name, address) that\'s displayed on the invoice. Users with the %s capability can always edit this information.', 'easy-digital-downloads' ),
					'<code>manage_shop_settings</code>'
				),
			),
			'edd-invoices-disable-free'         => array(
				'id'    => 'edd-invoices-disable-free',
				'name'  => __( 'Disable Invoices for Free Purchases', 'easy-digital-downloads' ),
				'check' => __( 'Optionally disable invoice generation for free purchases.', 'easy-digital-downloads' ),
				'type'  => 'checkbox_toggle',
			),
			'edd-invoices-ubl'                  => array(
				'id'    => 'edd-invoices-ubl',
				'name'  => __( 'Enable Electronic Invoices', 'easy-digital-downloads' ),
				'check' => __( 'Optionally enable electronic (UBL) invoices for customers. This is currently a beta feature.', 'easy-digital-downloads' ),
				'type'  => 'checkbox_toggle',
				'desc'  => __( 'Please ensure that your business information under Downloads > Settings is complete.<br>Store owners can download electronic invoices for individual orders or from Downloads > Reports > Export.', 'easy-digital-downloads' ),
			),
			'edd-invoices-page'                 => array(
				'id'      => 'edd-invoices-page',
				'name'    => __( 'Invoice Page', 'easy-digital-downloads' ),
				'desc'    => __( 'Which Page contains the [edd_invoices] shortcode? This page is used to display the form customers can use to update how their billing information appears on the invoice.', 'easy-digital-downloads' ),
				'type'    => 'select',
				'options' => edd_get_pages(),
			),
			'edd-invoices-logo'                 => array(
				'id'   => 'edd-invoices-logo',
				'name' => __( 'Logo URL', 'easy-digital-downloads' ),
				'type' => 'upload',
				'size' => 'regular',
			),
			'edd-invoices-styles'               => array(
				'id'   => 'edd-invoices-styles',
				'name' => __( 'Invoice Style', 'easy-digital-downloads' ),
				'type' => 'invoices_styles',
			),
			'edd-invoices-company-name'         => array(
				'id'   => 'edd-invoices-company-name',
				'name' => __( 'Company Name', 'easy-digital-downloads' ),
				'desc' => __( 'Company Name shown on Invoices', 'easy-digital-downloads' ),
				'type' => 'text',
				'size' => 'regular',
			),
			'edd-invoices-use-business-address' => array(
				'id'    => 'edd-invoices-use-business-address',
				'name'  => __( 'Use Business Address', 'easy-digital-downloads' ),
				'desc'  => __( 'Use the business address for the company address on invoices.', 'easy-digital-downloads' ),
				'type'  => 'checkbox_toggle',
				'class' => 'edd-invoices-use-business-address',
			),
			'edd-invoices-company-address'      => array(
				'id'       => 'edd-invoices-company-address',
				'name'     => __( 'Company Address', 'easy-digital-downloads' ),
				'desc'     => __( 'This address block will display on your invoices.', 'easy-digital-downloads' ),
				'type'     => 'textarea',
				'std'      => function_exists( 'edd_invoices_get_company_address' ) ? edd_invoices_get_company_address() : '',
				'class'    => 'edd-invoices-company-address',
				'readonly' => edd_get_option( 'edd-invoices-use-business-address' ),
			),
			'edd-invoices-number'               => array(
				'id'   => 'edd-invoices-number',
				'name' => __( 'Registration Number', 'easy-digital-downloads' ),
				'desc' => __( 'Company Registration Number', 'easy-digital-downloads' ),
				'type' => 'text',
				'size' => 'regular',
			),
			'edd-invoices-vat'                  => array(
				'id'   => 'edd-invoices-tax',
				'name' => __( 'Tax/VAT Number', 'easy-digital-downloads' ),
				'desc' => __( 'Company Tax/VAT Number', 'easy-digital-downloads' ),
				'type' => 'text',
				'size' => 'regular',
			),
			'edd-invoices-additional-text'      => array(
				'id'   => 'edd-invoices-additional-text',
				'name' => __( 'Additional Text', 'easy-digital-downloads' ),
				'desc' => __( 'Any text entered here will appear at the bottom of each invoice.', 'easy-digital-downloads' ),
				'type' => 'rich_editor',
				'size' => '10',
			),
		);

		return array_merge( $settings, array( 'edd-invoices' => $invoice_settings ) );
	}
}
