<?php
/**
 * Admin order handling for taxes.
 *
 * @package     EDD\Pro\Admin\Orders
 * @copyright   Copyright (c) 2025, Sandhills Development, LLC
 * @license     https://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       3.5.0
 */

namespace EDD\Pro\Admin\Orders;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit; // @codeCoverageIgnore

use EDD\EventManagement\SubscriberInterface;
use EDD\Pro\Taxes\VAT\Order as VAT;

/**
 * Admin order handling for taxes.
 *
 * @since 3.5.0
 */
class Taxes implements SubscriberInterface {

	/**
	 * Get the subscribed events.
	 *
	 * @since 3.5.0
	 * @return array
	 */
	public static function get_subscribed_events() {
		return array(
			'edd_get_order_details_sections' => 'order_details',
			'edd_updated_edited_purchase'    => 'updated_edited_purchase',
		);
	}

	/**
	 * Add the VAT details to the order details.
	 *
	 * @since 3.5.0
	 * @param array $sections The sections.
	 * @return array
	 */
	public function order_details( $sections ) {
		if ( edd_is_add_order_page() ) {
			return $sections;
		}

		$sections[] = array(
			'id'       => 'vat',
			'label'    => __( 'VAT', 'easy-digital-downloads' ),
			'icon'     => 'id-alt',
			'callback' => array( $this, 'show_vat_details' ),
		);

		return $sections;
	}

	/**
	 * Show the VAT details.
	 *
	 * @since 3.5.0
	 * @param \EDD\Orders\Order $order The order.
	 */
	public function show_vat_details( $order ) {
		if ( edd_is_add_order_page() ) {
			return;
		}

		$order_vat = VAT::get( $order->id );
		$address   = $order->get_address();

		?>
		<div id="edd-vat-details" class="edd-order-details__vat">
			<div class="edd-form-group">
				<label for="edd-payment-vat-number" class="edd-form-group__label">
					<?php esc_html_e( 'VAT number:', 'easy-digital-downloads' ); ?>
				</label>
				<div class="edd-form-group__control">
					<input type="text" id="edd-payment-vat-number" name="edd-payment-vat-number" value="<?php echo esc_attr( $order_vat->vat_number ); ?>" class="medium-text" />
				</div>
			</div>
			<div class="edd-form-group">
				<label for="edd-payment-vat-consultation-number" class="edd-form-group__label">
					<?php esc_html_e( 'Consultation number:', 'easy-digital-downloads' ); ?>
				</label>
				<div class="edd-form-group__control">
					<input type="text" id="edd-payment-vat-consultation-number" name="edd-payment-vat-consultation-number" value="<?php echo esc_attr( $order_vat->consultation_number ); ?>" class="medium-text" />
				</div>
			</div>
			<div class="edd-form-group">
				<div class="edd-form-group__control">
					<input type="checkbox" id="edd-payment-vat-number-valid" name="edd-payment-vat-number-valid" value="1" <?php checked( $order_vat->is_vat_number_valid ); ?> />
					<label for="edd-payment-vat-number-valid">
						<?php esc_html_e( 'VAT number valid', 'easy-digital-downloads' ); ?>
					</label>
				</div>
			</div>
			<div class="edd-form-group">
				<div class="edd-form-group__control">
					<input type="checkbox" id="edd-payment-vat-reverse-charged" name="edd-payment-vat-reverse-charged" value="1" <?php checked( $order_vat->is_reverse_charged ); ?> />
					<label for="edd-payment-vat-reverse-charged">
						<?php esc_html_e( 'VAT reverse charged', 'easy-digital-downloads' ); ?>
					</label>
				</div>
			</div>
			<div class="edd-form-group">
				<label for="edd-payment-vat-company-name" class="edd-form-group__label">
					<?php esc_html_e( 'Company name:', 'easy-digital-downloads' ); ?>
				</label>
				<div class="edd-form-group__control">
					<input type="text" id="edd-payment-vat-company-name" name="edd-payment-vat-company-name" value="<?php echo esc_attr( $order_vat->name ); ?>" class="large-text" />
				</div>
			</div>
			<div class="edd-form-group">
				<label for="edd-payment-vat-company-address" class="edd-form-group__label">
					<?php esc_html_e( 'Company address:', 'easy-digital-downloads' ); ?>
				</label>
				<div class="edd-form-group__control">
					<textarea id="edd-payment-vat-company-address" name="edd-payment-vat-company-address" class="large-text"><?php echo esc_textarea( $order_vat->address ); ?></textarea>
				</div>
			</div>
			<br />
			<h4><?php esc_html_e( 'Location Evidence', 'easy-digital-downloads' ); ?></h4>
			<p>
				<strong><?php esc_html_e( 'Billing country:', 'easy-digital-downloads' ); ?></strong>
				<?php echo esc_html( edd_get_country_name( $address->country ) ); ?>
			</p>
			<p>
				<strong><?php esc_html_e( 'IP address:', 'easy-digital-downloads' ); ?></strong>
				<?php
				echo edd_payment_get_ip_address_url( $order->id ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				?>
			</p>
		</div>
		<?php
	}

	/**
	 * Update the VAT details when the order is updated.
	 *
	 * @since 3.5.0
	 * @param int $order_id The order ID.
	 */
	public function updated_edited_purchase( $order_id ) {
		if ( ! current_user_can( 'edit_shop_payments' ) ) {
			return;
		}

		$vat_number = isset( $_POST['edd-payment-vat-number'] ) ? sanitize_text_field( $_POST['edd-payment-vat-number'] ) : '';
		if ( ! empty( $vat_number ) ) {
			edd_update_order_meta( $order_id, '_edd_payment_vat_number', $vat_number );
		} else {
			edd_delete_order_meta( $order_id, '_edd_payment_vat_number' );
		}

		$vat_number_valid = filter_input( INPUT_POST, 'edd-payment-vat-number-valid', FILTER_VALIDATE_BOOLEAN );
		if ( ! empty( $vat_number_valid ) ) {
			edd_update_order_meta( $order_id, '_edd_payment_vat_number_valid', $vat_number_valid );
		} else {
			edd_delete_order_meta( $order_id, '_edd_payment_vat_number_valid' );
		}

		$vat_reverse_charged = filter_input( INPUT_POST, 'edd-payment-vat-reverse-charged', FILTER_VALIDATE_BOOLEAN );
		if ( ! empty( $vat_reverse_charged ) ) {
			edd_update_order_meta( $order_id, '_edd_payment_vat_reverse_charged', $vat_reverse_charged );
		} else {
			edd_delete_order_meta( $order_id, '_edd_payment_vat_reverse_charged' );
		}

		$vat_company_name = isset( $_POST['edd-payment-vat-company-name'] ) ? sanitize_text_field( $_POST['edd-payment-vat-company-name'] ) : '';
		if ( ! empty( $vat_company_name ) ) {
			edd_update_order_meta( $order_id, '_edd_payment_vat_company_name', $vat_company_name );
		} else {
			edd_delete_order_meta( $order_id, '_edd_payment_vat_company_name' );
		}

		$vat_company_address = isset( $_POST['edd-payment-vat-company-address'] ) ? sanitize_textarea_field( $_POST['edd-payment-vat-company-address'] ) : '';
		if ( ! empty( $vat_company_address ) ) {
			$vat_company_address = implode( ', ', array_filter( explode( "\n", $vat_company_address ) ) );
			edd_update_order_meta( $order_id, '_edd_payment_vat_company_address', $vat_company_address );
		} else {
			edd_delete_order_meta( $order_id, '_edd_payment_vat_company_address' );
		}

		$vat_consultation_number = isset( $_POST['edd-payment-vat-consultation-number'] ) ? sanitize_text_field( $_POST['edd-payment-vat-consultation-number'] ) : '';
		if ( ! empty( $vat_consultation_number ) ) {
			edd_update_order_meta( $order_id, '_edd_payment_vat_consultation_number', $vat_consultation_number );
		} else {
			edd_delete_order_meta( $order_id, '_edd_payment_vat_consultation_number' );
		}
	}
}
