<?php
function bb_errors( $code ) {
	if ( empty( $code ) ) {
		return;
	}

	$codes = [
		'409' => __( 'There is an existing Service with the selected name. Please try with a different "Service Title".', 'brindle-booking' ),
		'409' => __( 'There is an existing Service with the selected name. Please try with a different "Service Title".', 'brindle-booking' ),
		'309' => __( 'There is an existing Category with the selected name. Please try with a different "Category".', 'brindle-booking' ),
	];

	return $codes[$code];
}

function bb_get_category_services( $cat_id = null ) {
	$args = [
		'post_type'      => 'bb_service',
		'post_status'    => 'publish',
		'posts_per_page' => -1,
	];

	if ( ! empty( $cat_id ) ) {
		$args['tax_query']['relation'] = 'AND';

		array_push( $args['tax_query'], [
			'taxonomy' => 'bb_service_category',
			'field'    => 'id',
			'terms'    => $cat_id,
		] );
	}

	$query = new WP_Query( $args );

	return $query;
}

function bb_schedule_types() {
	return [
		'15' => __( '15 Minutes', 'brindle-booking' ),
		'30' => __( '30 Minutes', 'brindle-booking' ),
		'60' => __( '60 Minutes', 'brindle-booking' ),
		'120' => __( '120 Minutes', 'brindle-booking' ),
		'240' => __( '240 Minutes', 'brindle-booking' ),
	];
}

function bb_schedule_selected_value( $value ) {
	$type_site_option = ! empty( get_site_option( 'bb_schedule_type' ) ) ? absint( get_site_option( 'bb_schedule_type' ) ) : 30;

	return ( isset( $_GET['type'] ) && absint( $_GET['type'] ) === $value ) || ( ! isset( $_GET['type'] ) && $type_site_option === $value ) ? 'selected="selected"' : '';
}

function bb_get_appointments_atts( $cat_id, $service_id, $date, $staff_member = false ) {
	$args = [
		'post_type'      => 'bb_appointment',
		'post_status'    => 'publish',
		'posts_per_page' => -1,
		'fields'         => 'ids',
		'meta_query'     => [
			'relation' => 'AND',
			[
				'key'     => '_bb_appointment_date',
				'value'   => $date,
			],
		]
	];

	if ( ! empty( $service_id ) ) {
		$args['meta_query'] = [
			'relation' => 'AND',
			[
				'key'     => '_bb_appointment_category',
				'value'   => $cat_id,
			],
			[
				'key'     => '_bb_appointment_service',
				'value'   => $service_id,
			],
			[
				'key'     => '_bb_appointment_date',
				'value'   => $date,
			],
		];
	}

	if ( ! empty( $staff_member ) ) {
		array_push( $args['meta_query'], [
			'key'     => '_bb_staff_member',
			'value'   => $staff_member,
		] );
	}

	$atts = [
		'dates' => [],
		'slots' => []
	];

	$appointments = get_posts( $args );

	if ( empty( $appointments ) ) {
		return [ 'dates' => [], 'slots' => [] ];
	}

	foreach ( $appointments as $post ) {
		if ( ! empty( $slot = get_post_meta( $post, '_bb_appointment_slot', true ) ) ) {
			array_push( $atts['slots'], $slot );
		}

		if ( ! empty( $date = get_post_meta( $post, '_bb_appointment_date', true ) ) ) {
			array_push( $atts['dates'], $date );
		}
	}

	return $atts;
}

function bb_get_services_for_select() {
	$services = get_posts( [
		'post_type'      => 'bb_service',
		'post_status'    => 'publish',
		'posts_per_page' => -1,
	] );

	$options = [ '' => __( 'Select a Service', 'brindle-booking' ) ];

	if ( ! empty( $services ) ) {
		foreach ( $services as $service ) {
			$options[$service->ID] = $service->post_title;
		}
	}

	return $options;
}

function bb_get_category_for_select() {
	$categories = get_terms( [
		'taxonomy'   => 'bb_service_category',
		'hide_empty' => false,
	] );

	$options = [ '' => __( 'Select a Service Category', 'brindle-booking' ) ];

	if ( ! empty( $categories ) ) {
		foreach ( $categories as $category ) {
			$options[$category->term_id] = $category->name;
		}
	}

	return $options;
}

function bb_check_if_appointment_exists( $cat_id, $service_id, $date, $slot = NULL, $staff_member = false ) {
	$args = [
		'post_type'      => 'bb_appointment',
		'post_status'    => 'publish',
		'posts_per_page' => -1,
		'fields'         => 'ids',
		'meta_query'     => [
			'relation' => 'AND',
			[
				'key'   => '_bb_appointment_date',
				'value' => $date,
			],
			[
				'key'   => '_bb_appointment_slot',
				'value' => $slot,
			]
		]
	];

	if ( ! empty( $service_id ) ) {
		array_push( $args['meta_query'], [
			'key'   => '_bb_appointment_category',
			'value' => $cat_id,
		] );

		array_push( $args['meta_query'], [
			'key'   => '_bb_appointment_service',
			'value' => $service_id,
		] );
	}

	if ( ! empty( $staff_member ) ) {
		array_push( $args['meta_query'], [
			'key'     => '_bb_staff_member',
			'value'   => $staff_member,
		] );
	}

	return get_posts( $args );
}

function bb_get_appointment_methods() {
	$admin_options = [
		'bb_method_in_store'     => __( 'In Store', 'brindle-booking' ),
		'bb_method_at_facility'  => __( 'At Facility', 'brindle-booking' ),
		'bb_method_online_video' => __( 'Online Video', 'brindle-booking' ),
		'bb_method_phone_call'   => __( 'Phone Call', 'brindle-booking' ),
		'bb_method_delivery'     => __( 'Delivery', 'brindle-booking' ),
	];

	$options = [ '0' => __( 'Select a Method', 'brindle-booking' ) ];

	foreach ( $admin_options as $key => $label ) {
		if ( ! empty( get_site_option( $key, true ) ) ) {
			$options[sanitize_title( $label )] = $label;
		}
	}

	return $options;
}

function bb_methods_post_meta( $method ) {
	$options = [
		'in-store'     => __( 'In Store', 'brindle-booking' ),
		'at-facility'  => __( 'At Facility', 'brindle-booking' ),
		'online-video' => __( 'Online Video', 'brindle-booking' ),
		'phone-call'   => __( 'Phone Call', 'brindle-booking' ),
		'delivery'     => __( 'Delivery', 'brindle-booking' ),
	];

	return $options[$method];
}

function bb_payment_methods_post_meta( $method ) {
	$options = [
		'stripe'       => __( 'Credit Card (100% secure via Stripe)', 'brindle-booking' ),
		'on_location'  => __( 'Pay on Location', 'brindle-booking' ),
		'with_invoice' => __( 'Pay via Invoice Later', 'brindle-booking' ),
	];

	return $options[$method];
}

function bb_get_user_data_fields() {
	$fields = [
		'bb-form-payment-name'   => [
			'label'       => __( 'Full Name', 'brindle-booking' ),
			'placeholder' => __( 'Angie Smith', 'brindle-booking' ),
			'active'      => get_site_option( 'bb_collected_details_full_name', true ),
			'required'    => get_site_option( 'bb_collected_details_full_name_required', true ),
		],
		'bb-form-payment-email' => [
			'label'       => __( 'Email', 'brindle-booking' ),
			'placeholder' => __( 'Please enter a valid email', 'brindle-booking' ),
			'active'      => get_site_option( 'bb_collected_details_email', true ),
			'required'    => get_site_option( 'bb_collected_details_email_required', true ),
		],
		'bb-form-payment-phone' => [
			'label'       => __( 'Phone', 'brindle-booking' ),
			'placeholder' => __( 'Please enter a valid phone number', 'brindle-booking' ),
			'active'      => get_site_option( 'bb_collected_details_phone', true ),
			'required'    => get_site_option( 'bb_collected_details_phone_required', true ),
		],
		'bb-form-payment-address' => [
			'label'       => __( 'Address', 'brindle-booking' ),
			'placeholder' => __( 'Please enter a valid address', 'brindle-booking' ),
			'active'      => get_site_option( 'bb_collected_details_address', true ),
			'required'    => get_site_option( 'bb_collected_details_address_required', true ),
		],
	];

	return $fields;
}

function bb_validate_field_format( $type, $value ) {
	if ( $type === 'text' ) {
		return true;
	}

	if ( $type === 'checkbox' ) {
		return true;
	}

	if ( $type === 'select' ) {
		return boolval( $value );
	}

	if ( $type === 'email' ) {
		return ! empty( filter_var( $value, FILTER_VALIDATE_EMAIL ) ) ? true : false;
	}

	if ( $type === 'phone' ) {
		$value = preg_replace( '/[–—―‒()\-\s+]/', '', $value );

		if ( ! is_numeric( $value ) ) {
			return false;
		}

		if ( strlen( $value ) < 8 || strlen( $value ) > 15 ) {
			return false;
		}

		return true;
	}
}

function bb_check_requred_fields( $data = [] ) {
	if ( empty( $data ) ) {
		return false;
	}

	$field_validation = [
		'validated' => true,
		'fields'    => [
			'name'      => [
				'type'          => 'text',
				'required'      => [
					'validation'    => true,
					'error_message' => __( 'This field is required.', 'brindle-booking' )
				],
				'format'        => [
					'validation'    => true,
					'error_message' => __( 'This field is required.', 'brindle-booking' )
				]
			],
			'email'     => [
				'type'          => 'email',
				'required'      => [
					'validation'    => true,
					'error_message' => __( 'This field is required.', 'brindle-booking' )
				],
				'format'        => [
					'validation'    => true,
					'error_message' => __( 'Please enter a valid email.', 'brindle-booking' )
				]
			],
			'phone'     => [
				'type'          => 'phone',
				'required'      => [
					'validation'    => true,
					'error_message' => __( 'This field is required.', 'brindle-booking' )
				],
				'format'        => [
					'validation'    => true,
					'error_message' => __( 'Please enter a valid phone number.', 'brindle-booking' )
				]
			],
			'address'     => [
				'type'          => 'text',
				'required'      => [
					'validation'    => true,
					'error_message' => __( 'This field is required.', 'brindle-booking' )
				],
				'format'        => [
					'validation'    => true,
					'error_message' => __( 'This field is required.', 'brindle-booking' )
				]
			],
			'payment-method'     => [
				'type'          => 'select',
				'required'      => [
					'validation'    => true,
					'error_message' => __( 'Please select a payment method.', 'brindle-booking' )
				],
				'format'        => [
					'validation'    => true,
					'error_message' => __( 'This field is required.', 'brindle-booking' )
				]
			],
			'agreement-checkbox' => [
				'type'          => 'checkbox',
				'required'      => [
					'validation'    => true,
					'error_message' => __( 'This field is required.', 'brindle-booking' )
				],
				'format'        => [
					'validation'    => true,
					'error_message' => __( 'This field is required.', 'brindle-booking' )
				]
			],
			'staff-member' => [
				'type'          => 'select',
				'required'      => [
					'validation'    => true,
					'error_message' => __( 'Please select a staff member.', 'brindle-booking' )
				],
				'format'        => [
					'validation'    => true,
					'error_message' => __( 'Please select a staff member.', 'brindle-booking' )
				]
			],
			'appointment-method' => [
				'type'          => 'select',
				'required'      => [
					'validation'    => true,
					'error_message' => __( 'Please select an appointment method.', 'brindle-booking' )
				],
				'format'        => [
					'validation'    => true,
					'error_message' => __( 'Please select an appointment method.', 'brindle-booking' )
				]
			]
		]
	];

	$is_payment_required = false;

	$online_payments   = get_site_option( 'bb_enable_online_payments', true );
	$location_payments = get_site_option( 'bb_enable_payment_on_location', true );
	$invoice_payments  = get_site_option( 'bb_enable_payment_via_invoice', true );

	$payment_methods = [
		0 => $online_payments,
		1 => $location_payments,
		2 => $invoice_payments
	];

	foreach ( $payment_methods as $method ) {
		if ( empty( $method ) ) {
			continue;
		}

		$is_payment_required = true;

		break;
	}

	$global_schedule = get_site_option( 'bb_single_schedule' );

	$fields_required = [
		'name'               => get_site_option( 'bb_collected_details_full_name_required', true ),
		'email'              => get_site_option( 'bb_collected_details_email_required', true ),
		'phone'              => get_site_option( 'bb_collected_details_phone_required', true ),
		'address'            => get_site_option( 'bb_collected_details_address_required', true ),
		'payment-method'     => $is_payment_required,
		'agreement-checkbox' => true,
		'staff-member'       => empty( $global_schedule ) ? true : false,
		'appointment-method' => get_site_option( 'bb_include_appointment_method', true ),
	];

	foreach ( $data as $key => $value ) {
		if ( empty( $value ) ) {
			if ( ! empty( $fields_required[ $key ] ) ) {
				$field_validation['fields'][ $key ]['required']['validation'] = ! empty( $value ) ? true : false;

				if ( $field_validation['fields'][ $key ]['required']['validation'] === false ) {
					$field_validation['validated'] = false;
				}
			} else {
				$field_validation['fields'][ $key ]['required']['validation'] = true;
			}
			
			$field_validation['fields'][ $key ]['format']['validation'] = true;
		} else {
			$field_validation['fields'][ $key ]['required']['validation'] = true;
			$field_validation['fields'][ $key ]['format']['validation'] = bb_validate_field_format( $key, $value );

			if ( $field_validation['fields'][ $key ]['format']['validation'] === false ) {
				$field_validation['validated'] = false;
			}
		}
	}

	return $field_validation;
}

function mbbb_order_service_categories( $categories ) {
	$category_order = get_site_option( 'bb_service_categories_in_order', '' );

	if ( empty( $category_order ) ) {
		return $categories;
	}
	
	$category_order_array = explode( '|', $category_order );

	$categories_sorted = [];

	foreach ( $category_order_array as $key => $item ) {
		foreach ( $categories as $category ) {
			if ( $category->term_id !== intval( $item ) ) {
				continue;
			}

			$categories_sorted[ $key ] = $category;
		}
	}

	$categories = $categories_sorted;

	return $categories;
}

function mbbb_order_services( $services, $service_order ) {
	if ( empty( $service_order ) ) {
		return $services;
	}

	$service_order_array = explode( '|', $service_order );

	$services_sorted = [];

	foreach ( $service_order_array as $key => $item ) {
		foreach ( $services as $service ) {
			if ( $service->ID !== intval( $item ) ) {
				continue;
			}

			$services_sorted[ $key ] = $service;
		}
	}

	$services = $services_sorted;

	return $services;
}

function mbbb_get_services_page_order() {
	$order = get_site_option( 'bb_services_in_order', '' );

	return $order;
}

function mbbb_get_services_full_order() {
	$order = get_site_option( 'bb_services_in_order', '' );

	return $order;
}

function mbbb_format_minutes( $minutes ) {
    if ( $minutes < 1 ) {
        return;
    }

    $minutes = intval( $minutes );

    $hours = floor( $minutes / 60);

    if ( ! empty( $hours ) ) {
    	$hours = sprintf( '%s %s', strval( $hours ), __( 'hour(s)', 'brindle-booking' ) );
    } else {
    	$hours = '';
    }

    $minutes = ( $minutes % 60);

    if ( ! empty( $minutes ) ) {
    	$separator = '';

    	if ( ! empty( $hours ) ) {
	    	$separator = __( 'and', 'brindle-booking' );
    	}

    	$minutes = sprintf( '%s %s %s', $separator, strval( $minutes ), __( 'minutes', 'brindle-booking' ) );
    } else {
    	$minutes = '';
    }

    $final_separator = '';

    if ( ! empty( $hours ) ) {
 	   $final_separator = ' ';
    }

    return sprintf( '%s%s%s', $hours, $final_separator, $minutes );
}