<?php
/**
 * Coupon Integrations Class
 *
 * @package     AffiliateWP Vanity Coupon Codes
 * @subpackage  Core
 * @copyright   Copyright (c) 2021, Awesome Motive Inc
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since	1.0
 */

namespace AffiliateWP_Vanity_Coupon_Codes\Integrations;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Implements the coupon integrations.
 * Handles EDD, WC, and MemberPress.
 * 
 * @since 1.0
 *
 */
class Coupon_Integrations {
	/**
	 * Update coupon by it's given integration.
	 *
	 * @since 1.0
	 * 
	 * @param object $coupon {
	 *     @type int    $affiliate_id Affiliate ID.
	 *     @type int    $coupon_id    Coupon ID.
	 *     @type string $integration  Integration context.
	 *     @type string $vanity_code  New vanity coupon code.
	 *     @type string $current_code Current coupon code.
	 *     @type string $type         Type (manual or dynamic).
	 * }
	 * @return bool True if updated successfully. Otherwise false.
	 */
	public function update_coupon( $coupon ) {
		if ( empty( $coupon ) && ! is_object( $coupon ) ) {
			return false;
		}

		// Required fields. Send error?
		if ( empty( $coupon->coupon_id ) || empty( $coupon->integration ) || empty( $coupon->vanity_code ) || empty( $coupon->current_code ) || empty( $coupon->type ) ) {
			return false;
		}

		// Set up coupon variables.
		$affiliate_id = intval( $coupon->affiliate_id );
		$coupon_id    = intval( $coupon->coupon_id );
		$integration  = $coupon->integration;
		$new_code     = $coupon->vanity_code;
		$old_code     = $coupon->current_code;
		$type         = $coupon->type;

		// Update EDD coupon code.
		if ( 'edd' === $integration ) {
			if ( function_exists( 'edd_update_adjustment' ) ) {
				return edd_update_adjustment( $coupon_id, array( 'code' => $new_code ) );
			}

			return update_post_meta( $coupon_id, '_edd_discount_code', $new_code, $old_code );
		}

		// Update WooCommerce coupon code.
		if ( 'woocommerce' === $integration ) {
			// Handle dynamic WC code.
			if ( 'dynamic' === $type ) {
				return affiliate_wp()->affiliates->coupons->update_coupon( $coupon_id, array(
					'coupon_code' => $new_code,
					'locked' => true,
				) );
			}
			// Handle manual WC code.
			return wp_update_post( array(
				'ID'         => $coupon_id,
				'post_title' => $new_code,
				'post_name'  => $new_code,
			), true );
		}

		// Update MemberPress coupon code.
		if ( 'memberpress' === $integration ) {
			// Handle the MemberPress code.
			$updated = wp_update_post( array(
				'ID'         => $coupon_id,
				'post_title' => $new_code,
			), true );
			// Reconnect to affiliate.
			if ( $updated ) {
				return update_post_meta( $coupon_id, 'affwp_discount_affiliate', $affiliate_id );
			}
		}
		return false;
	}

	/**
	 * Check if coupon exists by it's given integration.
	 *
	 * @since 1.0
	 *
	 * @param string $new_code    New vanity code.
	 * @param string $integration Integration context.
	 * @return object|bool Returns object if exists. Otherwise false.
	 */
	public function check_integration_for_existing_code( $new_code, $integration ) {
		// Required fields.
		if ( empty( $new_code ) || empty( $integration ) ) {
			return false;
		}

		// Ensure the new code is a string.
		if ( ! is_string( $new_code ) ) {
			$new_code = strval( $new_code );
		}

		// Check EDD.
		if ( 'edd' === $integration ) {
			if ( function_exists( 'edd_get_adjustments' ) ) {
				return edd_get_adjustments( array(
					'code'   => $new_code,
					'status' => 'active',
					'type'   => 'discount',
				) );
			}

			return get_posts( array(
				'post_type'   => 'edd_discount',
				'post_status' => 'active',
				'meta_query'  => array(
					array(
						'key'     => '_edd_discount_code',
						'value'   => $new_code,
						'compare' => 'EXISTS',
					),
				),
			) );
		}

		// Check WooCommerce - manual and dynamic.
		if ( 'woocommerce' === $integration ) {
			// Check manual.
			$existing_code = get_posts( array(
				'post_type' => 'shop_coupon',
				'title'     => $new_code,
			) );

			// If empty, also check WC dynamic.
			if ( empty( $existing_code ) ) {
				$existing_code = affwp_get_coupon( $new_code );
			}
			return $existing_code;
		}

		// Check MemberPress.
		if ( 'memberpress' === $integration ) {
			$existing_code = get_posts( array(
				'post_type'   => 'memberpresscoupon',
				'title'       => $new_code,
			) );
			return $existing_code;
		}

		return false;
	}



}